package org.lsst.ccs.utilities.structs;

import javax.swing.tree.TreeNode;
import java.beans.ConstructorProperties;
import java.io.Serializable;
import java.util.*;

/**
 * Immutable simple tree that can be used by swing Trees.
 *
 * @author bamade
 * @param <T>
 */
public class TreeBranch<T extends Serializable> implements Serializable, TreeNode, Iterable<T> {

    private static final long serialVersionUID = -8299651867210784530L;
    /**
     * the parent node. null if node is root.
     */
    private TreeBranch<T> parent;
    /**
     * list of children
     */
    private ArrayList<TreeBranch<T>> children;
    /**
     * the value attached to this node
     */
    private T content;
    /**
     * @ImplNote : there is a problem here: what is the currentIndex of the rootNode?
     * if 0 this could lead to bugs, if -1 other problems arise
     */
    private int currentIndex = -1;

    /**
     * to be used initially to build the root (avoid otherwise)
     *
     * @param content
     */
    public TreeBranch(T content) {
        this.content = content;
    }

    /**
     * Avoid using this constructor: for beans purposes use
     * the constructor annotated with <TT>ConstructorProperties</TT>
     */
    protected TreeBranch() {
    }

    /**
     * prefer the use of this constructor when building a tree
     * (instead of using sequence one-arg constructor + addChild)
     *
     * @param parent may be null if node is intended as a root Node.
     * @param content
     */
    @ConstructorProperties({"realParent", "content"})
    public TreeBranch(TreeBranch<T> parent, T content) {
        //this.parent = parent;
        this.content = content;
        if (parent != null) {
            parent.addChild(this);
        }
    }

    /**
     * adds a child to the current node.
     * Prefer the use of the 2-args constructor if possible;
     *
     * @param child
     */
    public synchronized void addChild(TreeBranch<T> child) {
        if (child == null) {
            throw new IllegalArgumentException("null child");
        }
        if (child.parent != null) {
            throw new IllegalArgumentException(this + " node " + child + " already part of another tree " + child.parent);
        }
        if (children == null) {
            children = new ArrayList<>();
        }
        int index = children.size();
        children.add(child);
        child.currentIndex = index;
        child.parent = this;
    }

    /**
     * @return the data stored in this node
     */
    public T getContent() {
        return content;
    }

    /**
     *
     * @param content the content to be set
     */
    public void setContent(T content) {
        this.content = content;
    }

    /**
     * @return the current index of this node in the parent's
     * child list. For the rootNode the index is -1 ;
     */
    public int getCurrentIndex() {
        return currentIndex;
    }

    /**
     * @return an iterator of contents where tree is explored in preOrder.
     */
    @Override
    public Iterator<T> iterator() {
        return new PreOrderIterator<>(this);
    }

    /**
     * @return an iterator of nodes where tree is explored in preOrder.
     */
    public Iterator<TreeBranch<T>> nodeIterator() {
        return new PreOrderNodeIterator<>(this);
    }

    /**
     * @return the list of immediate children of this node or an empty list if none
     */
    public List<TreeBranch<T>> getChildren() {
        if (children != null) {
            return children;
        }
        return Collections.emptyList();
    }

    /**
     * @return an iterator over immediate children or an empty Iterator if none
     */
    public Iterator<TreeBranch<T>> getChildIterator() {
        if (children != null) {
            return children.iterator();
        }
        return Collections.emptyIterator();
    }

    /**
     * @return the list of parent nodes starting from the rootNode
     */
    public List<TreeBranch<T>> getPath() {
        ArrayList<TreeBranch<T>> res = new ArrayList<>();
        TreeBranch<T> curNode = this;
        while (null != (curNode = curNode.getRealParent())) {
            res.add(curNode);
        }
        Collections.reverse(res);
        return res;
    }

    /**
     * @param childIndex
     * @return the child at the requested index or <TT>null</TT>
     * if out of range.
     * @ImplNote TODO: specify what happens if childIndex out of range
     */
    @Override
    public TreeNode getChildAt(int childIndex) {
        if (children != null) {
            if (childIndex >= children.size() || childIndex < 0) {
                return null;
            }
            TreeBranch<T> res = children.get(childIndex);
            assert childIndex == res.currentIndex;
            return res;
        }
        return null;
    }

    /**
     * @return the number of children of this node
     */
    @Override
    public int getChildCount() {
        if (children == null) {
            return 0;
        }
        return children.size();
    }

    /**
     * @return the parent of this node with type
     * <TT>TreeNode</TT> (TreeNode interface contract)
     */
    @Override
    public TreeNode getParent() {
        return parent;
    }

    /**
     * same as <TT>getParent</TT> but with proper return type
     *
     * @return a <TT>TreeBranch</TT> node or null when at the top
     * of the tree.
     */
    public TreeBranch<T> getRealParent() {
        return parent;
    }

    /**
     * @param node should be a child node of the current node
     * @return the index in the child list or -1 if the argument
     * is not found in this list
     */
    @Override
    public int getIndex(TreeNode node) {
        if (node.getParent() != this) {
            return -1;
        }
        if (children != null) {
            if (node instanceof TreeBranch) {
                TreeBranch branch = (TreeBranch) node;
                return branch.currentIndex;
            }
        }
        return -1;
    }

    /**
     * since the tree is immutable this method is not intended to be called during building operations.
     * It is here for the <TT>TreeNode</TT> contract.
     *
     * @return
     */
    @Override
    public boolean getAllowsChildren() {
        return children != null;
    }

    /**
     * @return true if this node has no children
     */
    @Override
    public boolean isLeaf() {
        return children == null;
    }

    /**
     * @return true of this node is the top of the tree.
     */
    public boolean isRootNode() {
        return parent == null;
    }

    /**
     * @return an enumeration over children of this node
     * (TreeNode interface contract)
     */
    @Override
    public Enumeration children() {
        //was return children.elements(); when Vector was used
        return new Enumeration() {
            final Iterator iter = children.iterator();

            @Override
            public boolean hasMoreElements() {
                return iter.hasNext();
            }

            @Override
            public Object nextElement() {
                return iter.next();
            }
        };
    }
    //utilities

    /**
     * @return a String representing the value attached to this node.
     */
    @Override
    public String toString() {
        return String.valueOf(content);
    }

    /**
     * returns a String that represents the tree
     *
     * @return
     */
    public String deepString() {
        return deepString(0);
    }

    protected String deepString(int level) {
        StringBuilder stb = new StringBuilder();
        for (int ix = 0; ix < level; ix++) {
            stb.append('\t');
        }
        stb.append(toString()).append('\n');
        int levelChilds = level + 1;
        if (children != null) {
            for (TreeBranch child : children) {
                stb.append(child.deepString(levelChilds));
            }
        }
        return stb.toString();
    }

    /**
     * inner class that delegates to <TT>PreOrderNodeIterator</TT>
     * to fetch content in pre-order.
     *
     * @param <X>
     */
    protected static class PreOrderIterator<X extends Serializable> implements Iterator<X> {

        private final PreOrderNodeIterator<X> preOrderNodeIterator;

        public PreOrderIterator(TreeBranch<X> rootNode) {
            preOrderNodeIterator = new PreOrderNodeIterator<>(rootNode);
        }

        @Override
        public X next() {
            TreeBranch<X> node = preOrderNodeIterator.next();
            return node.getContent();
        }

        @Override
        public boolean hasNext() {
            return preOrderNodeIterator.hasNext();
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("remove");
        }
    }

    /**
     * Inner class to fetch nodes in pre-order.
     *
     * @param <X>
     */
    protected static class PreOrderNodeIterator<X extends Serializable> implements Iterator<TreeBranch<X>> {

        private final Stack<Iterator<TreeBranch<X>>> stack = new Stack<>();

        public PreOrderNodeIterator(TreeBranch<X> rootNode) {
            //hack to generalize the algorithm: the rootNode is seen as a virtual member of a child list
            ArrayList<TreeBranch<X>> edenChildList = new ArrayList<>(1);
            edenChildList.add(rootNode);
            stack.push(edenChildList.iterator());
        }

        @Override
        public TreeBranch<X> next() {
            // get two aspirins before reading code! this is an idiom
            // gets Iterator on the stack
            Iterator<TreeBranch<X>> iter = stack.peek();
            // gets the next element
            TreeBranch<X> node = iter.next();
            // gets an Iterator over children
            Iterator<TreeBranch<X>> itChildren = node.getChildIterator();
            // if nothing else to read at next iteration -> pop
            if (!iter.hasNext()) {
                stack.pop();
            }
            // if child Iterator not empty push child Iterator
            if (itChildren.hasNext()) {
                stack.push(itChildren);
            }
            return node;
        }

        /**
         * is there an iterator on the stack and
         * is there something to read ?
         *
         * @return
         */
        @Override
        public boolean hasNext() {
            return (!stack.empty() && stack.peek().hasNext());
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("remove");
        }
    }
}
