package org.lsst.ccs.utilities.misc;

import java.net.URL;
import java.net.URLStreamHandler;
import java.net.URLStreamHandlerFactory;
import java.util.*;

/**
 * Allows registering multiple {@link URLStreamHandlerFactory} instances within the same JVM.
 * Note: factories are registered on the best effort basis, those added too late may be ignored.
 *
 * @author onoprien
 */
public class ExtendableURLStreamHandlerFactory implements URLStreamHandlerFactory {
    
    static private ExtendableURLStreamHandlerFactory factory;
    static private final HashMap<String, URLStreamHandlerFactory> byProtocol = new HashMap<>(0);
    static private final ArrayList<URLStreamHandlerFactory> others = new ArrayList<>(0);

    /**
     * Registers a factory.
     * 
     * @param handlerFactory Factory.
     * @param protocols Protocols this factory should be used for. If none is specified, the factory
     *                  will be tried for any protocol for which a factory has not been explicitly specified.
     */
    synchronized static public void add(URLStreamHandlerFactory handlerFactory, String... protocols) {
        if (protocols.length == 0) {
            others.add(handlerFactory);
        } else {
            for (String protocol : protocols) {
                if (byProtocol.containsKey(protocol)) {
                    throw new IllegalStateException("Factory already set for protocol "+ protocol +".");
                } else {
                    byProtocol.put(protocol, handlerFactory);
                }
            }
        }
        if (factory == null) {
            factory = new ExtendableURLStreamHandlerFactory();
            try {
                URL.setURLStreamHandlerFactory(factory);
            } catch (Error x) {
                factory = null;
                throw new IllegalStateException("URLStreamHandlerFactory has already been set for this JVM.", x);
            }
        }
    }

    /**
     * Called by the framework.
     * 
     * @param protocol Protocol.
     * @return Newly created stream handler.
     */
    @Override
    public URLStreamHandler createURLStreamHandler(String protocol) {
        URLStreamHandlerFactory f;
        ArrayList<URLStreamHandlerFactory> ff = null;
        synchronized(ExtendableURLStreamHandlerFactory.class) {
            f = byProtocol.get(protocol);
            if (f == null) {
                ff = new ArrayList<>(others);
            }
        }
        if (f == null) {
            for (URLStreamHandlerFactory fac : ff) {
                URLStreamHandler out = fac.createURLStreamHandler(protocol);
                if (out != null) {
                    return out;
                }
            }
            return null;
        } else {
            return f.createURLStreamHandler(protocol);
        }
    }

}
