package org.lsst.ccs.gconsole.util.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.*;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.swing.AbstractAction;
import javax.swing.AbstractListModel;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.base.Const;

/**
 * Dialog for searching through a list of strings.
 *
 * @author onoprien
 */
public final class SearchDialog extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    static private String SD_KEY = "_sdkey_";
    
    private final ArrayList<String> all;
    
    private int maxResults = 30;
    private String description;
    private ArrayList<Button> actions;
    
    private String prevText = "";
    private ArrayList<String> results; // null if too many
    private List<String> selected; // null if results is null

    private JTextField field;
    private JList resultsList;
    private JButton cancelButton;

// -- Life cycle : -------------------------------------------------------------
    
    public SearchDialog(List<String> items, String title, Component parent) {
        super(parent == null ? null : SwingUtilities.getWindowAncestor(parent), title, ModalityType.MODELESS);
        all = new ArrayList<>(items);
    }
    
    private void init() {
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent we) {
                results = null;
                selected = null;
                dispose();
            }
        });
        
        JPanel root = new JPanel(new BorderLayout());
        add(root);
        root.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));

        if (actions != null) actions.trimToSize();
        
        field = new JTextField();
        field.addCaretListener(e -> updateResults());
        if (description == null) {
            root.add(field, BorderLayout.NORTH);
        } else {
            JPanel panel = new JPanel();
            panel.setLayout(new BorderLayout());
            JTextArea descrArea = new JTextArea(description);
            descrArea.setOpaque(false);
            descrArea.setEditable(false);
            panel.add(descrArea, BorderLayout.NORTH);
            panel.add(Box.createRigidArea(Const.VDIM), BorderLayout.CENTER);
            panel.add(field, BorderLayout.SOUTH);
            panel.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, 0, Const.VSPACE, 0));
            root.add(panel, BorderLayout.NORTH);
        }
        
        resultsList = new JList();
        root.add(new JScrollPane(resultsList), BorderLayout.CENTER);
        resultsList.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        resultsList.setPrototypeCellValue("D1234567890123456789012345678901234567890");
        resultsList.setVisibleRowCount(Math.min(maxResults, 20));
        resultsList.addListSelectionListener(e -> updateSelection());
        
        Box box = Box.createHorizontalBox();
        root.add(box, BorderLayout.SOUTH);
        box.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
        if (actions != null) {
            for (Action act : actions) {
                box.add(new JButton(act));
                box.add(Box.createRigidArea(Const.HDIM));
            }
        }
        box.add(Box.createHorizontalGlue());
        cancelButton = new JButton("Cancel");
        cancelButton.addActionListener(e -> cancel());
        box.add(cancelButton);
        
        setSize(getPreferredSize());
        pack();
        setLocationRelativeTo(getParent());
        field.requestFocusInWindow();
        updateResults();
        setVisible(true);
    }

    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the maximum number of matching items to display in the list of results.
     * The default is 20.
     * 
     * @param maxResults Maximum number of results to display.
     */
    public void setMaxResults(int maxResults) {
        this.maxResults = maxResults;
    }

    /**
     * Sets the descriptive text to be displayed by this dialog.
     * The default is no description.
     * 
     * @param description Description.
     */
    public void setDescription(String description) {
        this.description = description;
    }

    public void addAction(Button action) {
        if (actions == null) {
            actions = new ArrayList<>(1);
        }
        action.putValue(SD_KEY, this);
        actions.add(action);
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    public List<String> getMatches() {
        return results == null ? null : Collections.unmodifiableList(results);
    }
    
    public List<String> getSelected() {
        return selected == null ? null : Collections.unmodifiableList(selected);
    }
    
    
// -- Searching : --------------------------------------------------------------
    
    public List<String> search() {
        init();
        return getSelected();
    }
    
    public void cancel() {
        results = null;
        selected = null;
        dispose();
    }
    
    
// -- Class for adding action buttons : ----------------------------------------
    
    static public abstract class Button extends AbstractAction {
        
        public Button(String name) {
            super(name);
        }
        
        public Button(String name, Icon icon) {
            super(name, icon);
        }

        public void setEnabled(SearchDialog dialog) {
            List<String> matches = dialog.getMatches();
            setEnabled(matches != null && !matches.isEmpty());
        }

        abstract public void actionPerformed(SearchDialog dialog);
        
        @Override
        public final void actionPerformed(ActionEvent event) {
            actionPerformed((SearchDialog) getValue(SD_KEY));
        }
    
    }
    
    
// -- Local classes and methods : ----------------------------------------------
    
    private class Model extends AbstractListModel<String> {

        @Override
        public int getSize() {
            if (results == null) {
                return prevText.isEmpty() ? 0 : 1;
            } else {
                return results.size();
            }
        }

        @Override
        public String getElementAt(int index) {
            return results == null ? "More than "+ maxResults +" matches" : results.get(index);
        }
    }
    
    private void updateResults() {
        String text = field.getText();
        if (text.equals(prevText)) return;
        selected = null;
        if (text.isEmpty()) {
            results = null;
        } else {
            ArrayList<String> items = new ArrayList<>(maxResults);
            int n = 0;
            try {
                Pattern p = Pattern.compile(text.replaceAll("\\.", "\\\\.").replaceAll("\\*", ".*").replaceAll("\\?", "."));
                for (String candidate : all) {
                    if (p.matcher(candidate).matches()) {
                        if (++n > maxResults) {
                            setForeground(Color.BLACK);
                            items = null;
                            break;
                        } else {
                            items.add(candidate);
                        }
                    }
                }
            } catch (PatternSyntaxException x) {
            }
            results = items;
            if (results == null) {
                field.setForeground(Color.BLACK);
            } else {
                field.setForeground(results.isEmpty() ? Color.RED : Color.BLACK);
                selected = Collections.emptyList();
            }
        }
        prevText = text;
        resultsList.setModel(new Model());
        field.requestFocusInWindow();
        actions.forEach(action -> action.setEnabled(this));
    }
    
    private void updateSelection() {
        selected = resultsList == null ? null : resultsList.getSelectedValuesList();
        actions.forEach(action -> action.setEnabled(this));
    }
    
    
// -- Test : -------------------------------------------------------------------
    
    static public void main(String... args) {
        SwingUtilities.invokeLater(() -> {
            List<String> items = Arrays.asList("uno", "dos", "tres", "trei.nta", "sorok/chetyre/s/polovinoy");
            SearchDialog d = new SearchDialog(items,"Title",null);
            d.setDescription("Enter search string");
            
            Button b = new Button("Print") {
                @Override
                public void setEnabled(SearchDialog dialog) {
                    List<String> ss = dialog.getSelected();
                    setEnabled(ss != null);
                }

                @Override
                public void actionPerformed(SearchDialog dialog) {
                    List<String> ss = dialog.getSelected();
                    System.out.println(ss);
//                    dialog.dispose();
                }
                
            };
            d.addAction(b);
            
            
            List<String> selected = d.search();
            System.out.println(selected);
        });
    }    
    
}
