package org.lsst.ccs.gconsole.util.swing;

import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import javax.swing.ImageIcon;

/** This is a convenience class to handle creating a single composite icon from several icons, and caching the
 *  created icons to eliminate duplicate work. This class is basically used as a key into a map, allowing us to
 *  define both a hashCode and equals in a single place.
 */
public class CompositeIcon {

    static private final byte PADDING = 2;
    static private final HashMap<CompositeIcon, ImageIcon> cache = new HashMap<>(0);

    private final ImageIcon[] icons;

    private CompositeIcon(ImageIcon... icons) {
        this.icons = icons;
    }

    static public ImageIcon get(ImageIcon... icons) {
        return switch(icons.length) {
            case 0 -> null;
            case 1 -> icons[0];
            default -> {
                int nNull = 0;
                for (ImageIcon ic : icons) {
                    if (ic == null) nNull++;
                }
                if (nNull > 0) {
                    yield get(Arrays.asList(icons).stream().filter(i -> i != null).toList());
                }
                CompositeIcon combo = new CompositeIcon(icons);
                ImageIcon icon = cache.get(combo);
                if (icon == null) {
                    icon = combine(icons);
                    cache.put(combo, icon);
                }
                yield icon;
            }
        };
    }
    
    static public ImageIcon get(List<ImageIcon> icons) {
        return switch(icons.size()) {
            case 0 -> null;
            case 1 -> icons.get(0);
            default -> get(icons.toArray(new ImageIcon[0]));
        };
    }

    static private ImageIcon combine(ImageIcon... icons) {
        int totalWidth = (icons.length - 1) * PADDING;
        int minHeight = 0;
        for (int i = 0; i < icons.length; ++i) {
            totalWidth += icons[i].getIconWidth();
            if (icons[i].getIconHeight() > minHeight) {
                minHeight = icons[i].getIconHeight();
            }
        }
        BufferedImage compositeImage = new BufferedImage(totalWidth, minHeight, BufferedImage.TYPE_INT_ARGB);
        Graphics graphics = compositeImage.createGraphics();
        int x = 0;
        for (int i = 0; i < icons.length; ++i) {
            final ImageIcon icon = icons[i];
            graphics.drawImage(icon.getImage(), x, 0, null);
            x += icon.getIconWidth() + PADDING;
        }
        return new ImageIcon(compositeImage);
    }

    @Override
    public int hashCode() {
        int hash = icons.length;
        for (int i = 0; i < icons.length; ++i) {
            hash += icons[i].hashCode() * (i + 1);
        }
        return hash;
    }

    @Override
    public boolean equals(final Object obj) {
        if (!(obj instanceof CompositeIcon)) return false;
        CompositeIcon other = (CompositeIcon) obj;
        if (icons.length != other.icons.length) return false;
        for (int i = 0; i < icons.length; ++i) {
            if (icons[i].hashCode() != other.icons[i].hashCode()) return false;
        }
        return true;
    }
}
