package org.lsst.ccs.gconsole.services.aggregator;

import java.util.*;
import org.lsst.ccs.bus.data.AgentInfo;

/**
 * {@link AgentChannel} whose value is fixed .
 *
 * @author onoprien
 */
public class StaticChannel implements AgentChannel {

// -- Fields : -----------------------------------------------------------------
    
    private final String path;
    private final AgentInfo agent;
    
    private final Object value;
    protected HashMap<String,Object> data;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an instance.
     * @param agent Agent this channel belongs to.
     * @param path Unique path identifying this channel.
     * @param value Fixed value of this channel.
     */
    public StaticChannel(AgentInfo agent, String path, Object value) {
        this.agent = agent;
        this.path = path;
        this.value = value;
    }

    
// -- Getters : ----------------------------------------------------------------

    /**
     * Return the unique path of this channel.
     * @return Channel path in (agent name)/(local path) form.
     */
    @Override
    public String getPath() {
        return path;
    }
    
    /**
     * Returns the name of the {@code Agent} this channel belongs to.
     * @return Agent name.
     */
    @Override
    public String getAgentName() {
        return agent.getName();
    }
    
    /**
     * Returns the descriptor of the {@code Agent} this channel belongs to. 
     * @return Agent descriptor.
     */
    @Override
    public AgentInfo getAgent() {
        return agent;
    }
    
    /**
     * Returns the path of this channel with the agent name and the following slash stripped.
     * Local paths are unique inside an agent.
     * @return Local path.
     */
    @Override
    public String getLocalPath() {
        return path.substring(path.indexOf("/")+1);
    }
    
    /**
     * Returns the value of the channel attribute identified by the {@code key}.
     * 
     * @param key Attribute key. If {@code VALUE} or {@code null}, the central value is returned.
     * @return Attribute value, or {@code null} if this channel does not have the specified attribute.
     */
    @Override
    @SuppressWarnings("unchecked")
    public <T> T get(Object key) {
        if (key == null) return get();
        String sKey = key.toString();
        if (Key.VALUE.equals(sKey)) return get();
        T out;
        try {
            out = (T) agent.getAgentProperty(sKey);
        } catch (ClassCastException x) {
            out = null;
        }
        return out;
    }
    
    /**
     * Returns the current central value of this channel.
     * @return Current value.
     */
    @Override
    @SuppressWarnings("unchecked")
    public <T> T get() {
        return (T) value;
    }

    @Override
    public List<String> getAttributes() {
        return Collections.emptyList();
    }
    

// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the value of the specified attribute.
     * The attribute value does not change if the new value is equal to the old one.
     * 
     * @param key The attribute is identified by the string returned by {@code toString()} method of the key object.
     *            If {@code key} is {@code null} or evaluates to {@link AgentChannel.Key#VALUE}, the central value is set.
     * @param value The new value.
     * @return {@code true} if the attribute value has changed as a result of this call.
     */
    @Override
    public boolean set(Object key, Object value) {
        return false;
    }
    
    /**
     * Sets the value of this channel.
     * The value will not change if the new value is equal to the old one.
     * 
     * @param value New value.
     * @return {@code true} if the value has changed as a result of this call.
     */
    @Override
    public boolean set(Object value) {
        return false;
    }
    
    
// -- Overriding Object : ------------------------------------------------------

    @Override
    public String toString() {
        return path;
    }

    @Override
    public boolean equals(Object obj) {
        return path.equals(obj) || (obj instanceof AgentChannel && path.equals(((AgentChannel)obj).getPath()));
    }

    @Override
    public int hashCode() {
        return path.hashCode();
    }

}
