package org.lsst.ccs.gconsole.services.aggregator;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.CommandState;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.bus.states.PhaseState;

/**
 * {@link AgentChannel} to be used for states whose values can be assigned one of the {@link DataProviderState} categories.
 * This can be useful for color coding, etc.
 * <p>
 * <i>Implementation note. Currently, {@code DataProviderState} categories are assigned manually in the constructor, only
 * to core states. If we ever want to assign categories to subsystem-specific states, it would make sense to shift those
 * assignments to state {@code enum} definitions.</i>
 *
 * @author onoprien
 */
public class StateChannel extends BasicChannel {

// -- Fields : -----------------------------------------------------------------
    
    private final Map<String,DataProviderState> value2state;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an instance.
     * @param path Unique path identifying this channel.
     * @param agent Agent this channel belongs to.
     * @param attributes Static attributes and metadata associated with this channel.
     * @param value Current value.
     */
    public StateChannel(String path, AgentInfo agent, Map<String,Object> attributes, Object value) {
        super(path, agent, attributes, value);
        String localPath = path.substring(path.indexOf("/"));
        switch (localPath) {
            case MARK_STATE + "AlertState":
                value2state = new HashMap<>(8);
                value2state.put(AlertState.NOMINAL.toString(), DataProviderState.NOMINAL);
                value2state.put(AlertState.WARNING.toString(), DataProviderState.WARNING);
                value2state.put(AlertState.ALARM.toString(), DataProviderState.ALARM);
                break;
            case MARK_STATE + "ConfigurationState":
                value2state = new HashMap<>(8);
                value2state.put(ConfigurationState.CONFIGURED.toString(), DataProviderState.NOMINAL);
                value2state.put(ConfigurationState.DIRTY.toString(), DataProviderState.WARNING);
                value2state.put(ConfigurationState.UNCONFIGURED.toString(), DataProviderState.ALARM);
                break;
            case MARK_STATE + "OperationalState":
                value2state = new HashMap<>(8);
                value2state.put(OperationalState.NORMAL.toString(), DataProviderState.NOMINAL);
                value2state.put(OperationalState.ENGINEERING_OK.toString(), DataProviderState.WARNING);
                value2state.put(OperationalState.ENGINEERING_FAULT.toString(), DataProviderState.ALARM);
                break;
            case MARK_STATE + "PhaseState":
                value2state = new HashMap<>(8);
                value2state.put(PhaseState.OPERATIONAL.toString(), DataProviderState.NOMINAL);
                value2state.put(PhaseState.INITIALIZING.toString(), DataProviderState.WARNING);
                value2state.put(PhaseState.CLOSING.toString(), DataProviderState.WARNING);
                value2state.put(PhaseState.OFF_LINE.toString(), DataProviderState.OFF_LINE);
                break;
            case MARK_STATE + "CommandState":
                value2state = new HashMap<>(4);
                value2state.put(CommandState.READY.toString(), DataProviderState.NOMINAL);
                value2state.put(CommandState.ACTIVE.toString(), DataProviderState.WARNING);
                break;
            default:
                value2state = null;
        }
    }
    
// -- Setting state:  ----------------------------------------------------------

    @Override
    public synchronized boolean set(Object value) {
        boolean out =  super.set(value);
        if (out && value2state != null) {
            set(Key.STATE, value == null ? null : value2state.get((String)value));
        }
        return out;
    }

}
