package org.lsst.ccs.gconsole.services.aggregator;

import java.util.*;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * An event fired by {@link AgentStatusAggregator} to notify {@link AgentStatusListener}s of changes in the agent's published data.
 *
 * @author onoprien
 */
public class AgentStatusEvent extends EventObject {

// -- Fields : -----------------------------------------------------------------
    
    protected final List<AgentInfo> agents;
    protected final StatusMessage message;
    protected final Map<AgentChannel,List<String>> statusChanges;
    protected final List<AgentChannel> addedChannels;
    protected final List<AgentChannel> removedChannels;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Generic constructor.
     * 
     * @param aggregator Status aggregator instance.
     * @param agent Source agent descriptor.
     * @param statusChanges Channels with changes.
     * @param addedChannels Added channels.
     * @param removedChannels Removed channels.
     */
    AgentStatusEvent(AgentStatusAggregator aggregator, List<AgentInfo> agents, StatusMessage message, Map<AgentChannel,List<String>> statusChanges, List<AgentChannel> addedChannels, List<AgentChannel> removedChannels) {
        super(aggregator);
        if (agents == null && message != null) {
            this.agents = Collections.singletonList(message.getOriginAgentInfo());
        } else {
            this.agents = agents;
        }
        this.message = message;
        this.statusChanges = statusChanges == null ? Collections.emptyMap() : statusChanges;
        this.addedChannels = addedChannels == null ? Collections.emptyList() : addedChannels;
        this.removedChannels = removedChannels == null ? Collections.emptyList() : removedChannels;
    }
    
    
// -- Getters : ----------------------------------------------------------------

    /**
     * Returns a reference to the status aggregator that fired this event.
     * @return Status aggregator instance.
     */
    @Override
    public AgentStatusAggregator getSource() {
        return (AgentStatusAggregator) source;
    }

    /**
     * Returns the list of remote agents whose data is included in this event.
     * @return List of contributing agents.
     */
    public List<AgentInfo> getAgents() {
        return agents;
    }

    /**
     * Returns the list of names of remote agents whose data is included in this event.
     * @return List of names of contributing agents.
     */
    public List<String> getAgentNames() {
        return agents.stream().map(a -> a.getName()).collect(Collectors.toList());
    }
    
    /**
     * Returns the status message that triggered this event.
     * @return Status message that triggered this event, or {@code null} if this event was not triggered by a bus message.
     */
    public StatusMessage getMessage() {
        return message;
    }

    /**
     * Returns a map of modified channels to lists of keys of changed attributes.
     * Removed or added channels do not appear in this map.
     * This method may return non-empty map only for events passed to {@code statusChanged(...)} method of the listener.
     * @return Map of channels to modified attributes lists.
     */
    public Map<AgentChannel, List<String>> getStatusChanges() {
        return statusChanges;
    }
    
    /**
     * Returns a list of new channels.
     * This method may return non-empty list only for events passed to
     * {@code configure(...)} and {@code statusChanged(...)} methods of the listener.
     * @return List of new channels.
     */
    public List<AgentChannel> getAddedChannels() {
        return addedChannels;
    }
    
    /**
     * Returns a list of channels that are no longer published by the agent.
     * The strings in the returned list are original channel paths.
     * This method returns (possibly) non-empty map only for events passed to {@code statusChanged(...)} method of the listener.
     * <p>
     * Note: with the current implementation, this method never gets called.
     * It is declared in anticipation of implementing filter and dictionary updates.
     * @return List of removed channels.
     */
    public List<AgentChannel> getRemovedChannels() {
        return removedChannels;
    }
    
    /**
     * Returns {@code true} if this event has no added, modified, or removed channels.
     * @return {@code true} if this event has no added, modified, or removed channels.
     */
    public boolean isEmpty() {
        return statusChanges.isEmpty() && addedChannels.isEmpty() && removedChannels.isEmpty();
    }
    

// -- Overriding Object : ------------------------------------------------------

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("SA event: ");
        sb.append("[").append(String.join(",", agents.stream().map(a -> a.getName()).collect(Collectors.toList()))).append("] ");
        if (!addedChannels.isEmpty()) sb.append("Added ").append(addedChannels.size()).append(" ");
        if (!removedChannels.isEmpty()) sb.append("Removed ").append(removedChannels.size()).append(" ");
        if (!statusChanges.isEmpty()) sb.append("Changed ").append(statusChanges.size()).append(" ");
        return sb.toString();
    }

}
