package org.lsst.ccs.gconsole.plugins.trending.dataselection;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import javax.swing.table.AbstractTableModel;
import org.freehep.application.PropertyUtilities;
import org.freehep.application.studio.Studio;

/**
 * Maintains a list of data type presets known to the application.
 *
 * @author onoprien
 */
public class PresetList extends AbstractTableModel {

// -- Private parts : ----------------------------------------------------------
    
    private final String KEY = "org.lsst.trending.datatypes";
    private final Studio app;
    
    private ArrayList<DataType> data;

// -- Construction and initialization : ----------------------------------------
    
    public PresetList(Studio application) {
        app = application;
        restore();
    }
    

// -- Getters and setters : ----------------------------------------------------
    
    /** Returns a list of {@code maxSize} most recently used time windows ordered by name. */
    public List<DataType> getRecent(int maxSize) {
        List<DataType> out = new ArrayList<>(data);
        Collections.sort(out, DataType.compareByTime());
        if (maxSize < out.size()) out = out.subList(0, maxSize);
        Collections.sort(out, DataType.compareByName());
        return out;
    }
    
    /** Returns preset with the specified index. */
    public DataType get(int index) {
        return data.get(index);
    }
    
    /** Returns the index of the specified preset in this list, or -1 if this list does not contain the preset. */
    public int indexOf(DataType type) {
        return data.indexOf(type);
    }
    
    /**
     * Returns time window with the specified name.
     * @param name Name of the data type.
     * @return Data type, or {@code null} if this preset list does not contain a type with the specified name.
     */
    public DataType get(String name) {
        for (DataType tw : data) {
            if (name.equals(tw.toString())) return tw;
        }
        return null;
    }
    
    
// -- Modifying the list : -----------------------------------------------------
    
    /**
     * Adds {@code type} to this list. If this list already contains either
     * identical or otherwise identical but persistent type, the list remains
     * unchanged and the existing type is returned. If this list contains
     * identically named but different type, that type is removed.
     *
     * @param type Type to add.
     * @return Type that has been added or retained in the list.
     */
    public DataType insert(DataType type) {
        
        if (type == null) {
            return null;
        } else if (type.getRaw()) {
            return DataType.RAW;
        } else if (type.getBins() == 0) {
            return DataType.DEFAULT;
        }
        
        String name = type.getName();
        int nElements = data.size();
            
        int i = 0;
        for (; i < nElements; i++) {
            DataType e = data.get(i);
            int out = name.compareTo(e.getName());
            if (out == 0) {
                if (type.isReplaceableBy(e)) {
                    return e.touch();
                } else {
                    data.set(i, type);
                    break;
                }
            } else if (out < 0) {
                data.add(i, type);
                break;
            }
        }
        if (i == nElements) {
            data.add(type);
        }

        if (type.isPersistent()) save();
        fireTableStructureChanged();
        return type;
    }
    
    /**
     * Removes the specified window from this list.
     * Updates application properties if necessary.
     */
    public void delete(DataType type) {
        if (data.remove(type) && type.isPersistent()) {
            save();
        }
        fireTableStructureChanged();
    }
    

// -- Implementing TableModel : ------------------------------------------------

    @Override
    public int getRowCount() {
        return data.size();
    }

    @Override
    public int getColumnCount() {
        return 4;
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        switch (columnIndex) {
            case 0: return data.get(rowIndex).toString();
            case 1: return data.get(rowIndex).getRaw();
            case 2: return data.get(rowIndex).getBins();
            case 3: return data.get(rowIndex).isPersistent();
            default: throw new IllegalArgumentException();
        }
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        return String.class;
    }

    @Override
    public String getColumnName(int column) {
        switch (column) {
            case 0: return "Name";
            case 1: return "Raw data";
            case 2: return "Bins";
            case 3: return "Saved";
            default: throw new IllegalArgumentException();
        }
    }
    
    
// -- Saving to / restoring from properties : ----------------------------------
    
    /**
     * Updates information on the specified time window in application properties.
     */
    private void save() {
        Properties prop = app.getUserProperties();
        if (prop != null) {
            int n = data.size();
            ArrayList<String> savedData = null;
            if (n != 0) {
                savedData = new ArrayList<>(n);
                for (int i=0; i<n; i++) {
                    savedData.add(data.get(i).toCompressedString());
                }
            }
            PropertyUtilities.setStringCollection(prop, KEY, savedData);
        }
    }
    
    /** Restore persistent time selections from application properties. */
    private void restore() {
        Properties prop = app.getUserProperties();
        if (prop == null) {
            data = new ArrayList<>(0);
        } else {
            String[] savedData = PropertyUtilities.getStringArray(prop, KEY, new String[0]);
            data = new ArrayList<>(savedData.length);
            for (String s : savedData) {
                try {
                    data.add(DataType.parseCompressedString(s));
                } catch (IllegalArgumentException x) {
                }
            }
            Collections.sort(data, DataType.compareByName());
        }
    }
    
}
