package org.lsst.ccs.gconsole.plugins.trending;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Objects;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;
import org.lsst.ccs.gconsole.services.persist.Savable;

/**
 * Data object that can be plotted by {@link TrendPlotter}.
 * This class is thread-safe, all methods can be called on any thread.
 * If the Trend instance is associated with a plotter, the plotter will be automatically
 * notified of any relevant changes in the  trend.
 *
 * @author onoprien
 */
public class Trend implements Savable {
   

// -- Fields : -----------------------------------------------------------------
    
    private final Descriptor descriptor;
    
    private TrendPlotter plotter;
    private TrendData data;
    private long timestamp;
    private boolean loading;
    private volatile String type;
    private volatile String[] values;

// -- Life cycle : -------------------------------------------------------------
    
    public Trend(Descriptor descriptor) {
        this.descriptor = descriptor;
    }

// -- Getters and setters : ----------------------------------------------------
    
    synchronized public Descriptor getDescriptor() {
        return descriptor;
    }

    synchronized public TrendPlotter getPlotter() {
        return plotter;
    }

    synchronized void setPlotter(TrendPlotter plotter) {
        this.plotter = plotter;
    }

    synchronized public TrendData getData() {
        return data;
    }

    synchronized public void setData(TrendData data) {
        this.data = data;
        timestamp = System.currentTimeMillis();
        loading = false;
        if (plotter != null) {
            TrendPlotter p = plotter;
            SwingUtilities.invokeLater(() -> {
                p.onDataChange(this);
            });
        }
    }

    synchronized public TimeWindow getTimeWindow() {
        return plotter == null ? null : plotter.getTimeWindow();
    }

    synchronized public long getTimestamp() {
        return timestamp;
    }

    synchronized public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }
    
    synchronized public long[] getTimeRange() {
        if (data == null) {
            TimeWindow timeWindow = getTimeWindow();
            if (timeWindow == null) return null;
            long now = System.currentTimeMillis();
            return new long[] {timeWindow.getLowerEdge(now), timeWindow.getUpperEdge(now)};
        } else {
            return data.getTimeRange();
        }
    }
    
    synchronized public boolean setLoading(boolean value) {
        boolean out = value != loading;
        if (!loading && value) {
            timestamp = System.currentTimeMillis();
            loading = true;
            if (plotter != null) {
                TrendPlotter p = plotter;
                SwingUtilities.invokeLater(() -> {
                    p.onRefreshRequest(this);
                });
            }
        }
        loading = value;
        return out;
    }
    
    synchronized public boolean isLoading() {
        return loading;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public String[] getValues() {
        return values;
    }

    public void setValues(String[] values) {
        this.values = values;
    }

    
// -- Descriptor : -------------------------------------------------------------
    
    @Override
    synchronized public Descriptor save() {
        return descriptor;
    }
    
    static public class Descriptor implements Serializable, Cloneable {

        private String path;
        private String displayPath;
        
        public Descriptor() {
        }
        
        public Descriptor(String path) {
            this.path = path;
        }
        
        public Descriptor(String path, String displayPath) {
            this.path = path;
            this.displayPath = displayPath;
        }
        
        public Descriptor(Descriptor other) {
            this.path = other.path;
            this.displayPath = other.displayPath;
        }

        public String getDisplayPath() {
            return displayPath == null ? path : displayPath;
        }

        public void setDisplayPath(String displayPath) {
            this.displayPath = displayPath;
        }

        public String getPath() {
            return path;
        }

        public void setPath(String path) {
            this.path = path;
        }
        
        // Extra getters :
        
        @Transient
        public String getTitle() {
            String s = getDisplayPath();
            return s;
        }
        
        // Override Object :

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder("[TD: ");
            if (displayPath != null) {
                sb.append(displayPath).append(" --> ");
            }
            sb.append(path);
            sb.append("]");
            return sb.toString();
        }

        @Override
        public Descriptor clone() {
            try {
                return (Descriptor) super.clone();
            } catch (CloneNotSupportedException x) {
                throw new RuntimeException(x); // never
            }
        }

        @Override
        public boolean equals(Object obj) {
            if (obj instanceof Descriptor d) {
                return Objects.equals(path, d.path) && Objects.equals(displayPath, d.displayPath);
            } else {
                return false;
            }
        }

        @Override
        public int hashCode() {
            return (path + (displayPath == null ? "" : displayPath)).hashCode(); 
        }

    }
    
}
