package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.SwingWorker;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultCaret;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyleContext;
import javax.swing.text.StyledDocument;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.base.panel.DataPage;
import org.lsst.ccs.gconsole.plugins.alert.LsstAlertPlugin;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.gconsole.services.rest.LsstRestService;
import org.lsst.ccs.localdb.statusdb.server.AlertEvent;
import org.lsst.ccs.localdb.statusdb.server.AlertInfo;

/**
 * Class that retrieves and displays trending data in textual format.
 *
 * @author onoprien
 */
public class TextPage extends JPanel implements Persistable, DataPage {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CATEGORY = "TextPage";
    private final TextPage.Descriptor descriptor;
    
    static private final int LIMIT = 10000;
    static private final Color[] RAINBOW = List.of(
            new Color(0, 0, 150), 
            new Color(0, 150, 0), 
            new Color(170, 0, 0), 
            Color.MAGENTA, 
            new Color(100, 100, 100), 
            new Color(200, 140, 0), 
            new Color(0, 150, 150),
            new Color(205, 125, 125)).toArray(new Color[0]);
    
    private final LsstTrendingPlugin plugin = Console.getConsole().getSingleton(LsstTrendingPlugin.class);
    private final LsstRestService restService = (LsstRestService) plugin.getConsole().getConsoleLookup().lookup(LsstRestService.class);

    private final ArrayList<Dataset> data = new ArrayList<>(1);
    
    private final JTextPane headPanel = new JTextPane();
    private final DefaultStyledDocument headDoc = (DefaultStyledDocument) headPanel.getStyledDocument();
    private final JTextPane mainPanel = new JTextPane();
    private final DefaultStyledDocument mainDoc = (DefaultStyledDocument) mainPanel.getStyledDocument();
    private final JCheckBox filterBox, searchBox;
    private final JTextField filterField, searchField;
    private final JButton searchNext, searchPrev;
    private final Action clearAction, refreshAction;
    
// -- Life cycle : -------------------------------------------------------------
    
    public TextPage(Descriptor desc) {
        descriptor = desc.clone();
        this.setLayout(new BorderLayout());
        
        headPanel.setEditable(false);
//        headPanel.setContentType("text/plain");
        headPanel.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
        try {
            DefaultCaret caret = (DefaultCaret) headPanel.getCaret();
            caret.setUpdatePolicy(DefaultCaret.NEVER_UPDATE);
        } catch (ClassCastException x) {
        }
        headPanel.setText("");
        add(headPanel, BorderLayout.NORTH);
        
        mainPanel.setEditable(false);
//        mainPanel.setContentType("text/plain");
        mainPanel.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
        try {
            DefaultCaret caret = (DefaultCaret) mainPanel.getCaret();
            caret.setUpdatePolicy(DefaultCaret.NEVER_UPDATE);
        } catch (ClassCastException x) {
        }
        mainPanel.setText("");
        add(new JScrollPane(mainPanel), BorderLayout.CENTER);
        
        Box buttonPanel = Box.createHorizontalBox();
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
        filterBox = new JCheckBox("Filter: ");
        filterBox.setSelected(descriptor.isFilter());
        buttonPanel.add(filterBox);
        filterField = new JTextField(descriptor.getFilterString());
        filterField.setEnabled(descriptor.isFilter());
        filterField.setToolTipText("Regular expression for filtering data records.");
        filterField.addActionListener(e -> {
            descriptor.setFilterString(filterField.getText());
            reprintMain();
        });
        buttonPanel.add(filterField);
        filterBox.addActionListener(e -> {
            boolean filterOn = filterBox.isSelected();
            filterField.setEnabled(filterOn);
            descriptor.setFilter(filterOn);
            String filterString = filterField.getText();
            descriptor.setFilterString(filterString);
            if (!filterString.isBlank()) {
                reprintMain();
            }
        });
        buttonPanel.add(Box.createRigidArea(Const.HDIM2));
        searchBox = new JCheckBox("Search: ");
        searchBox.setSelected(descriptor.isSearch());
        searchBox.setEnabled(false); // FIXME: search not implemented yet
        buttonPanel.add(searchBox);
        searchField = new JTextField(descriptor.getSearchString());
        searchField.setEnabled(descriptor.isSearch());
        buttonPanel.add(searchField);
        searchNext = new JButton(new ImageIcon(Console.class.getResource("down_16.png"), "Next"));
        searchNext.setToolTipText("Next");
        searchNext.setEnabled(descriptor.isSearch());
        buttonPanel.add(searchNext);
        searchPrev = new JButton(new ImageIcon(Console.class.getResource("up_16.png"), "Previous"));
        searchPrev.setToolTipText("Previous");
        searchPrev.setEnabled(descriptor.isSearch());
        buttonPanel.add(searchPrev);
        searchBox.addActionListener(e -> {
            boolean enable = searchBox.isSelected() && !searchField.getText().isBlank();
            searchField.setEnabled(enable);
            searchNext.setEnabled(enable);
            searchPrev.setEnabled(enable);
        });
        buttonPanel.add(Box.createRigidArea(Const.HDIM2));
        JButton settingsButton = new JButton(new ImageIcon(Console.class.getResource("settings_16.png"), "Properties..."));
        settingsButton.setToolTipText("Properties...");
        settingsButton.addActionListener(e -> displayPropertiesDialog());
        buttonPanel.add(settingsButton);
        buttonPanel.add(Box.createHorizontalGlue());
        add(buttonPanel, BorderLayout.SOUTH);

        clearAction = new AbstractAction("Clear") {
            @Override
            public void actionPerformed(ActionEvent e) {
                clear();
            }
        };
        clearAction.putValue(Action.SHORT_DESCRIPTION, "Remove all channels from this page.");
        clearAction.setEnabled(false);
        refreshAction = new AbstractAction("Refresh") {
            @Override
            public void actionPerformed(ActionEvent e) {
                refresh();
            }
        };
        refreshAction.putValue(Action.SHORT_DESCRIPTION, "Refresh all channels. Do not change time window.");
        refreshAction.setEnabled(false);
    }
    
// -- Getters and setters : ----------------------------------------------------
    
    public void setDateTimeFormat(DateTimeFormatter format) {
        descriptor.setDateTimeFormat(format);
        reprintMain();
    }
    
    public TimeWindow getTimeWindow() {
        String s = descriptor.getTimeWindow();
        return s == null ? null : TimeWindow.parseCompressedString(s);
    }
    
    public void setTimeWindow(TimeWindow timeWindow) {
        TimeWindow tw = getTimeWindow();
        descriptor.setTimeWindow(timeWindow == null ? null : timeWindow.toCompressedString());
        if (timeWindow == null || !timeWindow.equalsTime(tw)) {
            refresh();
        }
    }
    
    public void add(Trend.Descriptor channel) {
        (new SwingWorker<Dataset, Object>() {
            @Override
            protected Dataset doInBackground() throws Exception {
                return fetchData(channel);
            }
            @Override
            protected void done() {
                try {
                    Dataset out = get();
                    data.add(out);
                    descriptor.add(channel);
                    reprintHead();
                    reprintMain();
                    clearAction.setEnabled(true);
                    refreshAction.setEnabled(true);
                } catch (InterruptedException | ExecutionException | RuntimeException x) {
                    Console.getConsole().error("Unable to fetch trending data for "+ channel.getDisplayPath(), x);
                }
            }
        }).execute();
    }
    
    /** Remove all trends. */
    public void clear() {
        descriptor.setTrends(null);
        data.clear();
        headPanel.setText("");
        mainPanel.setText("");
        clearAction.setEnabled(false);
        refreshAction.setEnabled(false);
    }
    
    /** Re-fetch all trends data and reprint. Do not change time window.*/
    public void refresh() {
        Trend.Descriptor[] trends = getDescriptor().getTrends();
        (new SwingWorker<Dataset[], Object>() {
            @Override
            protected Dataset[] doInBackground() throws Exception {
                Dataset[] out = new Dataset[trends.length];
                for (int i=0; i<trends.length; i++) {
                    Trend.Descriptor trend = trends[i];
                    try {
                        out[i] = fetchData(trend);
                    } catch (RuntimeException x) {
                        out[i] = new Dataset(trend, x);
                    }
                }
                return out;
            }
            @Override
            protected void done() {
                try {
                    Trend.Descriptor[] currentTrends = descriptor.getTrends();
                    if (!Objects.deepEquals(trends, currentTrends)) return; // make sure the list of trends has not changed while we were fetching data
                    Dataset[] out = get();
                    data.clear();
                    headPanel.setText("");
                    mainPanel.setText("");
                    data.addAll(Arrays.asList(out));
                    reprintHead();
                    reprintMain();
                } catch (ExecutionException | InterruptedException | CancellationException x) {
                    Console.getConsole().error("Failed to refresh trending text page", x);
                }
            }
        }).execute();
    }
    
    public boolean contains(Trend.Descriptor channel) {
        for ( Trend.Descriptor t : descriptor.getTrends()) {
            if (t.equals(channel)) {
                return true;
            }
        }
        return false;
    }
    
    
// -- Adding items to popup menu : ---------------------------------------------

    @Override
    public JPopupMenu modifyPopup(JPopupMenu menu, Component component, Point point) {
        menu.insert(new AbstractAction("Page properties...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                displayPropertiesDialog();
            }
        }, 0);
        menu.insert(clearAction, 0);
        menu.insert(refreshAction, 0);
        return menu;
    }
    
    private void displayPropertiesDialog() {
        SettingsPanel sp = new SettingsPanel();
        int out = JOptionPane.showConfirmDialog(this, sp, "Trending Text Page Properties", JOptionPane.OK_CANCEL_OPTION);
        if (out == JOptionPane.OK_OPTION) {
            sp.save();
            reprintHead();
            reprintMain();
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /** Regenerates head panel text from data. */
    private void reprintHead() {
        headPanel.setText("");
        int n = descriptor.getTrends().length;
        try {
            for (int i=0; i<n; i++) {
                Trend.Descriptor trend = descriptor.getTrends()[i];
                JCheckBox cb = new JCheckBox(trend.getDisplayPath());
                Color color;
                if (n > 1) {
                    color = RAINBOW[i];
                    cb.setForeground(color);
                } else {
                    color = null;
                }
                cb.setSelected(descriptor.isTrendEnabled(i));
                cb.addActionListener(new DataSetEnabler(i, cb));
                SimpleAttributeSet attrs = new SimpleAttributeSet();
                StyleConstants.setComponent(attrs, cb);
                headDoc.insertString(headDoc.getLength(), System.lineSeparator(), attrs);
                printHeadItem(data.get(i).header, color);
            }
        } catch (BadLocationException x) {
            headPanel.setText(x.toString());
        }
    }
    
    /** Regenerates main panel text from data. */
    private void reprintMain() {
        mainPanel.setText("");
        if (data.isEmpty()) return;
        Predicate<String> filter = null;
        if (filterBox.isSelected()) {
            try {
                filter = Pattern.compile(filterField.getText()).asPredicate();
            } catch (PatternSyntaxException x) {
            }
        }
        DataIterator it = data.size() == 1 ? new DataIterator1() : new DataIteratorN();
        while (it.hasNext()) {
            Item item = it.next();
            if (filter == null || filter.test(toString(item))) {
                printMainItem(item, it.getColor());
            }
        }
    }
    
    /**
     * Retrieves data from REST server and packages it as {@code Dataset}. Called off EDT.
     * Never returns null, but might return empty {@code Dataset} with a meaningful header if the data cannot be retrieved.
     * @throws RuntimeException If fails for any reason.
     */
    private Dataset fetchData(Trend.Descriptor channel) {
        String path = channel.getPath();
        int i1 = path.indexOf("/");
        int i2 = path.indexOf("/", i1+1);
        if (i1 < 0 || i2 < 0 || !path.substring(i1, i2+1).equals(AgentChannel.MARK_ALERT)) {
            throw new IllegalArgumentException("Cannot handle path "+ path);
        }
        String agent = path.substring(0, i1);
        String alertID = path.substring(i2+1);
        TimeWindow timeWindow = getTimeWindow();
        long now = System.currentTimeMillis();
        long begin = timeWindow.getLowerEdge(now);
        long end = timeWindow.getUpperEdge(now);
        AlertInfo.AlertInfoList alertList = restService.getAlertList(agent, begin);
        if (alertList == null) {
            throw new RuntimeException("REST server is not available.");
        }
        AlertInfo info = alertList.list.stream().filter(a -> a.getAlertId().equals(alertID)).findAny().orElseThrow();

        Item header = new Item();
        header.text = new String[5];
        header.att = new Style[5];
        header.text[0] = "Sebsystem: ";
        header.text[1] = info.getSubsystemName();
        header.att[1] = getStyle(headDoc, StyleConstants.Bold);
        header.text[2] = "Alert ID: ";
        header.text[3] = info.getAlertId() +". ";
        header.att[3] = getStyle(headDoc, StyleConstants.Bold);
        header.text[4] = info.getAlertDescription();
        header.att[4] = getStyle(headDoc);
        
        List<AlertEvent> alertEvents = restService.geAlerts(begin, end, agent +"/"+ alertID);
        if (alertEvents.size() > LIMIT) {
            alertEvents = alertEvents.subList(alertEvents.size() - LIMIT, alertEvents.size());
        }
        List<Item> items = new ArrayList<>(alertEvents.size());
        for (AlertEvent e : alertEvents) {
            Item item = new Item();
            item.time = e.getTime();
            item.text = new String[3];
            item.att = new Style[3];
            item.text[0] = info.getAlertId();
            item.text[1] = e.getSeverity().toString();
            item.att[1] = getStyle(mainDoc, LsstAlertPlugin.COLOR.get(e.getSeverity()));
            item.text[2] = e.getCause();
            items.add(item);
        }
        return new Dataset(channel, header, items);
    }
    
    private String formatTime(long millis) {
        return (descriptor.dateTimeFormat == null ? Const.DEFAULT_DT_FORMAT : descriptor.dateTimeFormat).format(Instant.ofEpochMilli(millis));
    }
    
    /**
     * Adds item to the main document.
     * @param item Item to print.
     * @param color Color that differentiates this data set from others. Typically used for the timestamp. 
     */
    private void printMainItem(Item item, Color color) {
        try {
            mainDoc.insertString(mainDoc.getLength(), formatTime(item.time) + " ", getStyle(mainDoc, color));
            for (int i = 0; i < item.text.length; i++) {
                mainDoc.insertString(mainDoc.getLength(), item.text[i] +" ", item.att[i]);
            }
            mainDoc.insertString(mainDoc.getLength(), System.lineSeparator(), getStyle(mainDoc));
        } catch (BadLocationException x) {
        }
    }
    
    /**
     * Adds item to the main document.
     * @param item Item to print.
     * @param color Color that differentiates this data set from others. Typically used for the timestamp. 
     */
    private void printHeadItem(Item item, Color color) {
        try {
            for (int i = 0; i < item.text.length; i++) {
                headDoc.insertString(headDoc.getLength(), item.text[i] +" ", item.att[i]);
            }
            headDoc.insertString(headDoc.getLength(), System.lineSeparator(), getStyle(headDoc));
        } catch (BadLocationException x) {
        }
    }
    
    /**
     * Returns style instance that can be used to add text to the main document.
     * @param att StyleConstants.Bold, Color
     * @return Style.
     */
    private Style getStyle(StyledDocument doc, Object... att) {
        String name = switch(att.length) {
            case 0 -> StyleContext.DEFAULT_STYLE;
            case 1 -> att[0].toString();
            default -> String.join("+", Arrays.stream(att).map(a -> a.toString()).toList());
        };
        Style out = doc.getStyle(name);
        if (out == null) {
            out = doc.addStyle(name, doc.getStyle(StyleContext.DEFAULT_STYLE));
            for (Object a : att) {
                if (a instanceof Color color) {
                    StyleConstants.setForeground(out, color);
                } else if (a instanceof StyleConstants key) {
                    out.addAttribute(key, true);
                } else if (a instanceof List<?> pair && pair.size() == 2) {
                    Iterator<?> it = pair.iterator();
                    out.addAttribute(it.next(), it.next());
                }
            }
        }
        return out;
    }
    
    private String toString(Item item) {
        StringBuilder sb = new StringBuilder(formatTime(item.time));
        if (item.text != null) {
            sb.append(" ").append(String.join(" ", item.text));
        }
        return sb.toString();
    }


// -- Saving/Restoring : -------------------------------------------------------
    
    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = getDescriptor().clone();
        DataPanelDescriptor pd = DataPanelDescriptor.get(this);
        desc.setPanel(pd);
        return desc;
    }
    
    @Override
    public void restore(Persistable.Descriptor d) {
        if (d instanceof Descriptor desc) {
            // FIXME: implement
        }
    }
    
    static public class Descriptor extends Persistable.Descriptor {

        private String timeWindow;
        private Trend.Descriptor[] trends;
        private boolean[] trendEnabled;
        private DateTimeFormatter dateTimeFormat;
        private boolean filter;
        private String filterString;
        private boolean search;
        private String searchString;
        private boolean latestOnTop;
        private DataPanelDescriptor panel;

        // Getters and setters:
        
        @Override
        public String getCategory() {
            return TextPage.CATEGORY;
        }
        
        @Override
        public void setCategory(String category) {
        }

        public String getTimeWindow() {
            return timeWindow;
        }

        public void setTimeWindow(String timeWindow) {
            this.timeWindow = timeWindow;
        }

        public Trend.Descriptor[] getTrends() {
            return trends == null ? new Trend.Descriptor[0] : Arrays.copyOf(trends, trends.length);
        }

        public void setTrends(Trend.Descriptor[] trends) {
            this.trends = trends == null || trends.length == 0 ? null : Arrays.copyOf(trends, trends.length);
        }

        public boolean[] getTrendEnabled() {
            if (trendEnabled == null) {
                boolean[] out = new boolean[trends == null ? 0 : trends.length];
                Arrays.fill(out, true);
                return out;
            } else {
                return Arrays.copyOf(trendEnabled, trendEnabled.length);
            }
        }

        public void setTrendEnabled(boolean[] trendEnabled) {
            if (trendEnabled == null || trendEnabled.length == 0) {
                this.trendEnabled = null;
            } else {
                for (boolean b : trendEnabled) {
                    if (!b) {
                        this.trendEnabled = Arrays.copyOf(trendEnabled, trendEnabled.length);
                        return;
                    }
                }
                this.trendEnabled = null;
            }
        }
        
        public boolean isTrendEnabled(int index) {
            return trendEnabled == null ? true : trendEnabled[index];
        }

        public void setTrendEnabled(int index, boolean b) {
            if (trendEnabled == null) {
                if (!b) {
                    trendEnabled = new boolean[trends.length];
                    for (int i=0; i<trends.length; i++) {
                        trendEnabled[i] = i != index;
                    }
                }
            } else {
                trendEnabled[index] = b;
                if (b) {
                    for (int i=0; i<trendEnabled.length; i++) {
                        if (!trendEnabled[i]) {
                            return;
                        }
                    }
                    trendEnabled = null;
                }
            }
        }

        public DateTimeFormatter getDateTimeFormat() {
            return dateTimeFormat;
        }

        public void setDateTimeFormat(DateTimeFormatter dateTimeFormat) {
            this.dateTimeFormat = dateTimeFormat;
        }

        public DataPanelDescriptor getPanel() {
            return panel;
        }

        public void setPanel(DataPanelDescriptor panel) {
            this.panel = panel;
        }

        public String getSearchString() {
            return searchString == null ? "" : searchString;
        }

        public void setSearchString(String searchString) {
            this.searchString = searchString == null || searchString.isBlank() ? null : searchString;
        }

        public boolean isSearch() {
            return search;
        }

        public void setSearch(boolean search) {
            this.search = search;
        }

        public String getFilterString() {
            return filterString == null ? "" : filterString;
        }

        public void setFilterString(String filterString) {
            this.filterString = filterString == null || filterString.isBlank() ? null : filterString;
        }

        public boolean isFilter() {
            return filter;
        }

        public void setFilter(boolean filter) {
            this.filter = filter;
        }

        public boolean isLatestOnTop() {
            return latestOnTop;
        }

        public void setLatestOnTop(boolean latestOnTop) {
            this.latestOnTop = latestOnTop;
        }
        
        // Extra operations:
        
        void add(Trend.Descriptor trend) {
            if (trends == null) {
                trends = new Trend.Descriptor[] {trend};
            } else {
                int n = trends.length;
                trends = Arrays.copyOf(trends, n+1);
                trends[n] = trend;
                if (trendEnabled != null) {
                    trendEnabled = Arrays.copyOf(trendEnabled, n+1);
                    trendEnabled[n] = true;
                }
            }
        }
        
        // Cloning:

        @Override
        public Descriptor clone() {
            Descriptor clone = (Descriptor) super.clone();
            if (trends != null) {
                int n = trends.length;
                clone.trends = Arrays.copyOf(trends, n);
                if (trendEnabled != null) {
                    clone.trendEnabled = Arrays.copyOf(trendEnabled, n);
                }
            }
            if (panel != null) {
                clone.panel = panel.clone();
            }
            return clone;
        }
        
    }
    

// -- Local classes : ----------------------------------------------------------
    
    /** Entry in a {@code Dataset}. */
    static private class Item {
        long time;
        String[] text;
        Style[] att; // FIXME: should these be strings, nulls, etc.
        Item() {}
        Item(long time, String[] text, Style[] att) {
            this.time = time;
            this.text = text;
            this.att = att;
        }
        Item(String[] text, Style[] att) {
            this(System.currentTimeMillis(), text, att);
        }
        Item(String text, Style att) {
            this(System.currentTimeMillis(), new String[] {text}, new Style[] {att});
        }
    }
    
    /** Ready-to-display data for a channel. */
    private class Dataset implements Iterable<Item> {
        final Trend.Descriptor trend;
        Item header;
        List<Item> items;
        Dataset(Trend.Descriptor trend, Item header, List<Item> items) {
            this.trend = trend;
            this.header = header;
            this.items = items;
        }
        Dataset(Trend.Descriptor trend, Exception x) {
            this.trend = trend;
            header = new Item(new String[] {trend.getDisplayPath()}, new Style[] {getStyle(headDoc)});
        }
        @Override
        public ListIterator<Item> iterator() {
            return descriptor.isLatestOnTop() ? new ListIterator<Item>() {
                ListIterator<Item> delegate = items.listIterator(items.size());
                @Override
                public boolean hasNext() {
                    return delegate.hasPrevious();
                }
                @Override
                public Item next() {
                    return delegate.previous();
                }
                @Override
                public boolean hasPrevious() {
                    return delegate.hasNext();
                }
                @Override
                public Item previous() {
                    return delegate.next();
                }
                @Override
                public int nextIndex() {
                    throw new UnsupportedOperationException();
                }
                @Override
                public int previousIndex() {
                    throw new UnsupportedOperationException();
                }
                @Override
                public void remove() {
                    throw new UnsupportedOperationException();
                }
                @Override
                public void set(Item e) {
                    throw new UnsupportedOperationException();
                }
                @Override
                public void add(Item e) {
                    throw new UnsupportedOperationException();
                }
            } : items.listIterator();
        }
    }
    
    /**
     * Iterator over items in all {@code Dataset}s, in time stamp order.
     * Should be constructed on IDT, then can be used on any thread.
     */
    private interface DataIterator extends Iterator<Item> {
        int getIndex();
        Color getColor();
    }
    
    /** Implementation for a single {@code Dataset}. */
    private class DataIterator1 implements DataIterator {
        
        private final ListIterator<Item> it = data.get(0).iterator();

        @Override
        public int getIndex() {
            return 0;
        }

        @Override
        public Color getColor() {
            return Color.BLACK;
        }

        @Override
        public boolean hasNext() {
            return it.hasNext();
        }

        @Override
        public Item next() {
            return it.next();
        }

    }
        
    /** Implementation for more than one {@code Dataset}. */    
    private class DataIteratorN implements DataIterator {
        
        private boolean empty;
        private int index = -1;
        private final TreeMap<Long,LinkedList<DSIterator>> head = descriptor.isLatestOnTop() ? new TreeMap<>(Comparator.reverseOrder()) : new TreeMap<>();
        
        DataIteratorN() {
            for (int i=0; i<data.size(); i++) {
                if (descriptor.trendEnabled == null || descriptor.trendEnabled[i]) {
                    ListIterator<Item> it = data.get(i).iterator();
                    if (it.hasNext()) {
                        DSIterator dsi = new DSIterator(it, i);
                        add(dsi);
                    }
                }
            }
            empty = head.isEmpty();
        }

        @Override
        public boolean hasNext() {
            return !empty;
        }

        @Override
        public Item next() {
            if (empty) throw new NoSuchElementException();
            Map.Entry<Long,LinkedList<DSIterator>> e = head.pollFirstEntry();
            LinkedList<DSIterator> list = e.getValue();
            DSIterator it = list.poll();
            if (!list.isEmpty()) {
                head.put(e.getKey(), list);
            }
            Item out = it.next();
            if (it.hasNext()) {
                add(it);
            } else {
                empty = head.isEmpty();
            }
            index = it.getIndex();
            return out;
        }

        @Override
        public int getIndex() {
            return index;
        }

        @Override
        public Color getColor() {
            return RAINBOW[getIndex()%RAINBOW.length];
        }
        
        private void add(DSIterator it) {
            long time = it.next().time;
            it.previous();
            head.computeIfAbsent(time, t -> new LinkedList<>()).add(it);
        }
        
        static private class DSIterator {
            private final ListIterator<Item> it;
            private final int index;
            DSIterator(ListIterator<Item> it, int index) {
                this.it = it;
                this.index = index;
            }
            Item next() {
                return it.next();
            }
            Item previous() {
                return it.previous();
            }
            boolean hasNext() {
                return it.hasNext();
            }
            int getIndex() {
                return index;
            }
        }
        
    }
    
    private class DataSetEnabler implements ActionListener {
        final int index;
        final JCheckBox checkBox;
        DataSetEnabler(int index, JCheckBox checkBox) {
            this.index = index;
            this.checkBox = checkBox;
        }
        @Override
        public void actionPerformed(ActionEvent e) {
            getDescriptor().setTrendEnabled(index, checkBox.isSelected());
            reprintMain();
        }
    }
    
    private class SettingsPanel extends JPanel {
        
        private final JCheckBox latestOnTopBox;
        
        SettingsPanel() {
            setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
            Box root = Box.createVerticalBox();
            add(root);
 
            latestOnTopBox = new JCheckBox("Display records in reverse chronological order.");
            latestOnTopBox.setSelected(descriptor.isLatestOnTop());
            root.add(latestOnTopBox);
             
            root.add(Box.createVerticalGlue());
        }
        
        void save() {
            descriptor.setLatestOnTop(latestOnTopBox.isSelected());
        }
    }
    
}
