package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.Color;
import java.awt.Component;
import static java.awt.Component.LEFT_ALIGNMENT;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;
import javax.swing.SwingUtilities;
import org.freehep.application.Application;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageManager;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.PlotFactory;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.freehep.jas.services.PlotterProvider;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.SubsystemSelectorFilter;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelType;
import org.lsst.ccs.gconsole.jas3.JasPanelManager;
import org.lsst.ccs.gconsole.plugins.trending.dataselection.DataType;
import org.lsst.ccs.gconsole.plugins.trending.dataselection.DataTypeSelector;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindowSelector;
import org.lsst.ccs.gconsole.services.persist.Creator;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.AgentPresenceManager;

/**
 * Graphical console plugin that plots historical data.
 * All public methods of this class should be called on EDT.
 *
 * @author onoprien
 */
@Plugin(name="LSST Trending Plugin",
        id="trending",
        description="Allows plotting time histories for channels from the trending database and other sources",
        loadAtStart=true)
public class LsstTrendingPlugin extends ConsolePlugin implements PlotterProvider, TrendingService {

// -- Fields : -----------------------------------------------------------------
    
    final String PANEL_GROUP = "TrendingPlots";
    
    private TrendingPreferences prefs;
    private RestSource restSource;
    private volatile PlotFactory plotFactory;
    private volatile AutoRefreshManager refreshMan;
    
    private TimeWindowSelector timeWindowSelector;
    private DataTypeSelector dataTypeSelector;
    private JToolBar toolbar;
    private JCheckBoxMenuItem autoRefreshMenu;
    
    private final ArrayList<ControlPanel> controlPanels = new ArrayList<>(0);
    private final ArrayList<TrendPage> pages = new ArrayList<>(0);
    private final ArrayList<TextPage> textPages = new ArrayList<>(0);
    private boolean restoring;
    
    private ThreadPoolExecutor dataFetcher;
    private final int DATA_FETCHER_MAX_THREADS = 3;
    private final int DATA_FETCHER_KEEP_ALIVE = 120;

    
// -- Life cycle : -------------------------------------------------------------

// <editor-fold defaultstate="collapsed">
    /**
     * Called on EDT by the framework - initialization.
     * <ul>
     * <li>Loads preferences.
     * <li>Creates menu for opening trending control pages.</ul>
     */
    @Override
    public void initialize() {
        
        prefs = new TrendingPreferences(this);
        getConsole().getConsoleLookup().add(prefs);
        
        Action act = new AbstractAction("New control panel") {
            @Override
            public void actionPerformed(ActionEvent e) {
                openControlPanel(null);
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:3", "Trending:1");
        
        act = new AbstractAction("Load control panel...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                PersistenceService service = getConsole().getSingleton(PersistenceService.class);
                try {
                    ControlPanel.Descriptor desc = (ControlPanel.Descriptor) service.load(ControlPanel.CATEGORY, "Load trending control panel", null);
                    if (desc != null) openControlPanel(desc);
                } catch (CancellationException | ClassCastException | NullPointerException x) {
                }
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:3", "Trending:2");
        
        act = new AbstractAction("New plot page") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createPage("Trending").showPage();
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:3", "Trending:3");
        
        act = new AbstractAction("Load plot page...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                PersistenceService service = getConsole().getSingleton(PersistenceService.class);
                try {
                    TrendPage.Descriptor desc = (TrendPage.Descriptor) service.load(TrendPage.CATEGORY, "Load trending plot page", null);
                    if (desc != null) {
                        TrendPage page = (TrendPage) service.make(desc);
                        pages.add(page);
                    }
                } catch (Throwable x) {
                    getConsole().getLogger().warning("Failed to restore a trending page", x);
                }
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:3", "Trending:4");
        
        AgentPresenceManager apMan = getConsole().getMessagingAccess().getAgentPresenceManager();
        apMan.addAgentPresenceListener(new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                SwingUtilities.invokeLater(() -> {
                    for (AgentInfo info : agents) {
                        AgentInfo.AgentType type = info.getType();
                        if (!(AgentInfo.AgentType.CONSOLE.equals(type) || AgentInfo.AgentType.LISTENER.equals(type))) {
                            Action act = new AbstractAction(info.getName()) {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    openAgentControlPanel(e.getActionCommand());
                                }
                            };
                            getServices().addMenu(act, "400: CCS Tools :-1:3", "Trending:100:100", "Subsystems:1");
                            act = new AbstractAction("Trending") {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    openAgentControlPanel(info.getName());
                                }
                            };
                            getServices().addMenu(act, "CCS Subsystems", info.getName() + ":-10:3");
                        }
                    }
                });
            }
            @Override
            public void disconnected(AgentInfo... agents) {
                SwingUtilities.invokeLater(() -> {
                    for (AgentInfo agent : agents) {
                        getConsole().removeMenu(" CCS Tools ", "Trending", "Subsystems", agent.getName());
                        getConsole().removeMenu("CCS Subsystems", agent.getName(), "Trending");
                    }
                });
            }
        });

    }

    /**
     * Called on EDT by the framework - startup.
     * If auto-start is on, opens an un-filtered control page. 
     * That triggers trending tool infrastructure initialization.
     */
    @Override
    public void start() {
        if (prefs.isAutoStart()) {
            openControlPanel(null);
        }
     }

    /**
     * Called on EDT by the framework - shutdown.
     * Shuts down executor that fetches data from the REST server.
     */
    @Override
    public void shutdown() {
        if (dataFetcher != null) {
            dataFetcher.shutdownNow();
        }
    }
    
    /**
     * Initializes trending infrastructure required by both service and tool.
     * All resource-intensive processes are initialized here. Called on EDT.
     */
    private void startTrending() {
        
        if (restSource != null) return;
        
        // Plot factory:
        
        plotFactory = (PlotFactory) this.getConsole().getConsoleLookup().lookup(PlotFactory.class);
        if (plotFactory == null) throw new RuntimeException("No plot factory available");
        
        // Executor that fetches data from the REST server:
        
        ThreadFactory tFactory = new ThreadFactory() {
            final ThreadFactory delegate = Executors.defaultThreadFactory();
            final AtomicInteger id = new AtomicInteger(0);
            @Override
            public Thread newThread(Runnable r) {
                Thread thread = delegate.newThread(r);
                thread.setName("Trending Data Fetcher " + id.getAndIncrement());
                thread.setDaemon(true);
                return thread;
            }
        };
        dataFetcher = new ThreadPoolExecutor(DATA_FETCHER_MAX_THREADS, DATA_FETCHER_MAX_THREADS, 
                                          DATA_FETCHER_KEEP_ALIVE, TimeUnit.SECONDS, 
                                          new LinkedBlockingQueue<>(), tFactory);
        dataFetcher.allowCoreThreadTimeOut(true);
        
        // Trending toolbar:
        
        showToolbar();

        // Create trending source
        
        restSource = new RestSource(this);
        restSource.start();
    }
    
    /**
     * Shuts down trending infrastructure required by both service and tool.
     * All resource-intensive processes should be terminated here. The trending service remains active.
     * Called on the EDT.
     */
    private void stopTrending() {
        
        if (restSource == null) return;
        
        // Stop auto-refresh if active:
        
        stopAutoRefresh();
        
        // Trending source:
        
        restSource.stop();
        restSource = null;
        
        // Executor that fetches data from the REST server:
        
        dataFetcher.shutdownNow();
        dataFetcher = null;
        
        // Trending toolbar:
        
        hideToolbar();
        
        // Plot factory:
        
        plotFactory = null;
        
        // Other cleanup:
        
        controlPanels.trimToSize();
        pages.trimToSize();
    }

    private void showToolbar() {
        
        Studio app = (Studio) Application.getApplication();
        toolbar = new JToolBar("trending");
        toolbar.setBorder(BorderFactory.createEtchedBorder());
        
        JLabel label; JMenuBar bar; JMenu menu; JMenuItem item; AbstractAction act; JButton button; ImageIcon icon; JPanel panel;
        int pos = 0;

        label = new JLabel("Trending: ");
        toolbar.add(label, pos++);
        
        // New plot page
        
        icon = new ImageIcon(getClass().getResource("grid_16.png"));
        button = new JButton(icon);
        button.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createLineBorder(Color.BLACK), 
                BorderFactory.createEmptyBorder(1, 1, 1, 1)));
        button.setToolTipText("New plot page");
        JPopupMenu pop0 = new JPopupMenu();
        pop0.add(new AbstractAction("1 x 1") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createPage(1, 1);
            }
        });
        pop0.add(new AbstractAction("2 x 1") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createPage(2, 1);
            }
        });
        pop0.add(new AbstractAction("1 x 2") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createPage(1, 2);
            }
        });
        pop0.add(new AbstractAction("2 x 2") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createPage(2, 2);
            }
        });
        pop0.add(new AbstractAction("1 x 3") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createPage(1, 3);
            }
        });
        pop0.add(new AbstractAction("2 x 3") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createPage(2, 3);
            }
        });
        pop0.add(new AbstractAction("3 x 3") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createPage(3, 3);
            }
        });
        pop0.add(new AbstractAction("4 x 4") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createPage(4, 4);
            }
        });
        pop0.add(new AbstractAction("Custom...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                int[] ii = customPageDialog(pop0);
                if (ii != null) {
                    createPage(ii[0], ii[1]);
                }
            }
        });
        button.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                pop0.show(e.getComponent(), e.getX(), e.getY());
            }
        });
        toolbar.add(button,pos++);
        toolbar.add(new JLabel(" "),pos++);
        
        // Resresh button
        
        icon = new ImageIcon(getClass().getResource("refresh_16.png"));
        button = new JButton(icon);
        button.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createLineBorder(Color.BLACK), 
                BorderFactory.createEmptyBorder(1, 1, 1, 1)));
        button.setToolTipText("Refresh plots");
        JPopupMenu pop1 = new JPopupMenu();
        item = new JMenuItem("All");
        item.addActionListener(e -> {
                PageManager pm = app.getPageManager();
                PageContext selectedPage = pm.getSelectedPage();
                List pcList = pm.pages();
                for (Object o : pcList) {
                    try {
                        PlotPage page = (PlotPage) JasPanelManager.getPanel((PageContext) o);
                        if (page != null) {
                            refresh(page, null, null);
                        }
                    } catch (NullPointerException|ClassCastException x) {
                    }
                }
                selectedPage.requestShow();
        });
        pop1.add(item);
        item = new JMenuItem("Page");
        item.addActionListener(e -> {
            Object page = getConsole().getPanelManager().getSelectedPanel(PanelType.DATA);
            if (page instanceof PlotPage p) {
                refresh(p, null, null);
            } else if (page instanceof TextPage p) {
                p.refresh();
            }
        });
        pop1.add(item);
        item = new JMenuItem("Plot");
        item.addActionListener(e -> {
            try {
                PlotPage page = getSelectedPage();
                refresh(page.currentRegion(), null, null);
            } catch (NullPointerException | ClassCastException x) {
            }
        });
        pop1.add(item);
        pop1.addSeparator();
        autoRefreshMenu = new JCheckBoxMenuItem("Auto");
        autoRefreshMenu.setSelected(false);
        autoRefreshMenu.addItemListener(e -> {
            if (e.getStateChange() == ItemEvent.SELECTED) {
                startAutoRefresh();
            } else if (e.getStateChange() == ItemEvent.DESELECTED) {
                stopAutoRefresh();
            }
        });
        pop1.add(autoRefreshMenu);
        button.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                pop1.show(e.getComponent(), e.getX(), e.getY());
            }
        });
        toolbar.add(button,pos++);
        
        // Time selector

        label = new JLabel(" Range:");
        label.setToolTipText("Select time range");
        toolbar.add(label, pos++);
        
        timeWindowSelector = new TimeWindowSelector(app);
        timeWindowSelector.setToolTipText("Select time range");
        timeWindowSelector.setEnabled(timeWindowSelector.getItemCount() > 0);
        toolbar.add(timeWindowSelector, pos++);  
        
        // Data type selector :
        
        label = new JLabel(" Data:");
        label.setToolTipText("Select data type");
        toolbar.add(label, pos++);

        
        TrendingPreferences pref = (TrendingPreferences) Console.getConsole().getConsoleLookup().lookup(TrendingPreferences.class);
        boolean raw = pref.isUseRawData();
        int bins = pref.getNBins();
        DataType prefType = new DataType(null, raw, bins, false);
        dataTypeSelector  = new DataTypeSelector(app, prefType);
        dataTypeSelector.setToolTipText("Select data type");
        toolbar.add(dataTypeSelector, pos++);
        
        // Apply button
        
        button = new JButton("Apply");
        button.addActionListener(e -> {
            ApplyDialog d = ApplyDialog.apply((Component)(e.getSource()));
            if (!d.isCancelled()) {
                TimeWindow timeWindow = d.getWhat().contains(ApplyDialog.What.TIME_RANGE) ? getSelectedTimeWindow() : null;
                DataType dataType = d.getWhat().contains(ApplyDialog.What.DATA_TYPE) ? dataTypeSelector.getSelectedDataType() : null;
                switch (d.getWhere()) {
                    case ALL:
                        PageManager pm = app.getPageManager();
                        PageContext selectedPage = pm.getSelectedPage();
                        List pcList = pm.pages();
                        for (Object o : pcList) {
                            try {
                                PlotPage page = (PlotPage) JasPanelManager.getPanel((PageContext) o);
                                if (page != null) {
                                    refresh(page, timeWindow, dataType);
                                }
                            } catch (NullPointerException | ClassCastException x) {
                            }
                        }
                        if (timeWindow != null) {
                            textPages.forEach(p -> p.setTimeWindow(timeWindow));
                        }
                        if (selectedPage != null) selectedPage.requestShow();
                        break;
                    case PAGE:
                        try {
                            Object page = getConsole().getPanelManager().getSelectedPanel(PanelType.DATA);
                            if (page instanceof PlotPage p) {
                                refresh(p, timeWindow, dataType);
                            } else if (page instanceof TextPage p && timeWindow != null) {
                                p.setTimeWindow(timeWindow);
                            }
                        } catch (NullPointerException x) {
                        }
                        break;
                    case PLOT:
                        try {
                            PlotPage page = getSelectedPage();
                            refresh(page.currentRegion(), timeWindow, dataType);
                        } catch (NullPointerException | ClassCastException x) {
                        }
                        break;
                }
            }
        });
        toolbar.add(button,pos++);
        
        // Show toolbar
        
        app.addToolBar(toolbar, toolbar.getName());
    }
    
    private void hideToolbar() {
        Studio app = (Studio) Application.getApplication();
        app.removeToolBar(toolbar);
        toolbar = null;
        timeWindowSelector = null;
    }
    
    private void startAutoRefresh() {
        if (refreshMan != null) stopAutoRefresh();
        refreshMan = new AutoRefreshManager(this);
        refreshMan.start();
    }
    
    private void stopAutoRefresh() {
        if (refreshMan == null) return;
        refreshMan.stop();
        refreshMan = null;
    }
    
    /** Called to check if keeping the infrastructure active is still necessary. */
    private void checkState() {
        if (controlPanels.isEmpty() && pages.isEmpty() && !restoring) {
            stopTrending();
        }
    }
//</editor-fold>
    
// -- Opening and closing pages : ----------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    private ControlPanel openControlPanel(ControlPanel.Descriptor descriptor) {
        try {
            startTrending();
            ControlPanel controlPanel = new ControlPanel(descriptor);
            Map<Object, Object> par = new HashMap<>();
            par.put(Panel.TITLE, controlPanel.getName());
            par.put(Panel.TYPE, PanelType.CONTROL);
            Consumer<JComponent> onClose = c -> {
                ControlPanel cp = (ControlPanel) ((JScrollPane)c).getViewport().getView();
                cp.stop();
                controlPanels.remove(cp);
                checkState();
            };
            par.put(Panel.ON_CLOSE, onClose);
            Console.getConsole().getPanelManager().open(new JScrollPane(controlPanel), par);
            controlPanels.add(controlPanel);
            controlPanel.start();
            return controlPanel;
        } catch (RuntimeException x) {
            getConsole().error("Unable to open trending control page", x);
            checkState();
            return null;
        }
    }
    
    private void closeControlPanel(ControlPanel panel) {
        Console.getConsole().getPanelManager().close(panel);
    }
    
    private TrendPage createPage(String name) {
        TrendPage page = new TrendPage(name);
        pages.add(page);
        return page;
    }
        
    void closePage(TrendPage page) {
        pages.remove(page);
    }
    
    private TextPage openTextPage(TextPage.Descriptor descriptor) {
        if (descriptor == null) descriptor = new TextPage.Descriptor();
        try {
            startTrending();
            TextPage page = new TextPage(descriptor);
            Map<Object, Object> par = new HashMap<>();
            par.put(Panel.TITLE, "Alert");
            par.put(Panel.TYPE, PanelType.DATA);
            Consumer<JComponent> onClose = c -> {
                textPages.remove(c);
                checkState();
            };
            par.put(Panel.ON_CLOSE, onClose);
            DataPanelDescriptor panDesc = descriptor.getPanel();
            if (panDesc != null && panDesc.isOpen()) {
                Map<String, Serializable> data = panDesc.getData();
                if (data != null) {
                    par.putAll(data);
                }
            }
            Console.getConsole().getPanelManager().open(page, par);
            textPages.add(page);
            return page;
        } catch (RuntimeException x) {
            getConsole().error("Unable to open trending text page", x);
            checkState();
            return null;
        }
    }
    
    private void closeTextPage(TextPage page) {
        if (page != null) {
            Console.getConsole().getPanelManager().close(page);
            textPages.remove(page);
        }
    }    
// </editor-fold>
    
// -- Getters : ----------------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    
    static public PlotPage getSelectedPage() {
        return (PlotPage) JasPanelManager.getPanel(((Studio) Application.getApplication()).getPageManager().getSelectedPage());
    }
    
    TrendingPreferences getPreferences() {
        return prefs;
    }
    
    PlotFactory getPlotFactory() {
        if (plotFactory == null) {
            startTrending();
        }
        return plotFactory;
    }
    
    ThreadPoolExecutor getExecutor() {
        return dataFetcher;
    }
    
    RestSource getTrendingSource() {
        return restSource;
    }
    
    TimeWindowSelector getTimeWindowSelector() {
        return timeWindowSelector;
    }
    
    TimeWindow getSelectedTimeWindow() {
        return timeWindowSelector.getSelectedTimeWindow();
    }
    
    DataType getSelectedDataType() {
        return dataTypeSelector.getSelectedDataType();
    }
    
    /**
     * Returns the list of trending pages.
     * Must be called on the EDT.
     * @return The list of trending pages.
     */
    List<TrendPage> getPages() {
        return Collections.unmodifiableList(pages);
    }
// </editor-fold>
    
//  -- Plotting : --------------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    /**
     * Plots the channel in the specified region.
     * This method should always be called on the EDT.
     * 
     * @param channel Channel to plot.
     * @param timeWindow Time window. If {@code null}, currently selected time window is used.
     * @param region Region to plot in.
     * @param options The current plot in the specified region is either replaced or added to,
     *                depending on whether the {@code OVERLAY} option is present.
     */
    void plot(Trend.Descriptor channel, PlotRegion region, TrendPlotter.Option... options) {
        
        // find or create plotter
        
        if (channel == null || region == null) return;
        
        TrendPlotter plotter;
        try {
        plotter = (TrendPlotter) region.currentPlot();
        } catch (ClassCastException x) {
            return;
        }
        if (plotter == null) {
            plotter = create();
            showPlotter(plotter, region);
        }
        if (!plotter.isActive()) return;
        
        // plot
        
        Trend trend = new Trend(channel);
        int mode = !plotter.isEmpty() && options.length != 0 && Arrays.asList(options).contains(TrendPlotter.Option.OVERLAY) ? Plotter.OVERLAY : Plotter.NORMAL;
        plotter.plot(trend, mode);
        
        // launch trend update
        
        final Trend finalTrend = trend;
        dataFetcher.execute(() -> refresh(finalTrend));
        
    }
    
    /**
     * Plots time history for the specified channel.
     * This method should always be called on the EDT.
     * 
     * @param channel Data channel to be plotted.
     * @param options Option that control the selection of target region and other execution parameters.
     */
    void plot(Trend.Descriptor channel, TrendPlotter.Option... options) {
        
        if (channel == null) return;
        EnumSet<TrendPlotter.Option> opts = options.length == 0 ? EnumSet.noneOf(TrendPlotter.Option.class) : EnumSet.copyOf(Arrays.asList(options));
        
        // initialize infrastructure if not already initialized:
        
        startTrending();
        
        // fetch currently selected time window
        
        TimeWindow timeWindow = getSelectedTimeWindow();
        if (timeWindow == null) return;
        
        // if using existing plot for the specified channel is requested, find it

        Trend trend = null;
        if (opts.contains(TrendPlotter.Option.EXIST)) {
            trend = findTrend(channel, timeWindow);
        }
        
        // if necessary, create new trend, plotter, region, and page
        
        if (trend == null) {
            PlotPage page = null;
            PlotRegion region = null;
            if (!opts.contains(TrendPlotter.Option.NEWPAGE)) {
                page = plotFactory.currentPage();
            }
            if (page == null) {
                page = createPage(channel.getTitle());
                if (page == null) {
                    getConsole().error("Unable to create a plot page");
                    return;
                } else {
                    region = page.currentRegion();
                }
            }
            if (region == null) {
                if (opts.contains(TrendPlotter.Option.NEWPLOT)) {
                    int n = page.numberOfRegions();
                    for (int i = 0; i < n; i++) {
                        PlotRegion r = page.region(i);
                        if (r.currentPlot() == null) {
                            region = r;
                            break;
                        }
                    }
                } else {
                    region = page.currentRegion();
                }
                if (region == null) {
                    region = page.addRegion();
                }
            }
            trend = new Trend(channel);
            
            TrendPlotter plotter;
            Plotter pl = region.currentPlot();
            if (pl != null && !(pl instanceof TrendPlotter)) {
                pl.clear();
                region.clear();
                pl = null;
            }
            if (pl == null) {
                plotter = create();
                showPlotter(plotter, region);
            } else {
                plotter = (TrendPlotter) pl;
            }
            int mode = !plotter.isEmpty() && opts.contains(TrendPlotter.Option.OVERLAY) ? Plotter.OVERLAY : Plotter.NORMAL;
            plotter.plot(trend, mode);
        }
        
        // bring to front
        
        trend.getPlotter().toFront();
        
        // launch trend update
        
        final Trend finalTrend = trend;
        dataFetcher.execute(() -> refresh(finalTrend));

    }
    
    /**
     * Refreshes all regions where the specified channel data is plotted.
     * This method should always be called on the EDT.
     * 
     * @param channel Data channel.
     */
    void refresh(Trend.Descriptor channel) {
        ArrayList<Trend> trends  = findTrends(channel);
        trends.forEach(trend -> {
            if (!trend.getTimeWindow().isFixed()) {
                dataFetcher.execute(() -> refresh(trend.getPlotter()));
            }
        });
        for (TextPage tp : textPages) {
            if (tp.contains(channel)) {
                tp.refresh();
            }
        }
    }
    
    /**
     * Refreshes all regions on the specified page.
     * This method should always be called on the EDT.
     * 
     * @param page Plot page.
     * @param timeWindow Time window, or {@code null} if the windows already associated with the plots should be used. 
     * @param dataType Data type, or {@code null} if the types already associated with the plots should be used.
     */
    void refresh(PlotPage page, TimeWindow timeWindow, DataType dataType) {
        int n = page.numberOfRegions();
        try {
            for (int i = 0; i < n; i++) {
                refresh(page.region(i), timeWindow, dataType);
            }
        } catch (IndexOutOfBoundsException x) {
            // should not happen but there is code in Jas3 that may modify the page while we're refreshing it
        }
    }
    
    /**
     * Refreshes plot in the specified region.
     * This method should always be called on the EDT.
     * 
     * @param region Plot region.
     * @param timeWindow Time window, or {@code null} if the window already associated with the plot should be used. 
     * @param dataType Data type, or {@code null} if the type already associated with the plot should be used.
     */
    void refresh(PlotRegion region, TimeWindow timeWindow, DataType dataType) {
        Plotter plotter = region.currentPlot();
        if (plotter != null && plotter instanceof TrendPlotter) {
            refresh((TrendPlotter)plotter, timeWindow, dataType);
        }
    }
    
    /**
     * Refreshes a plot.
     * This method should always be called on the EDT.
     * 
     * @param tp Plot.
     * @param timeWindow Time window, or {@code null} if the window already associated with the plot should be used. 
     * @param dataType Data type, or {@code null} if the type already associated with the plot should be used.
     */
    void refresh(TrendPlotter tp, TimeWindow timeWindow, DataType dataType) {
        if (tp.isActive()) {
            if (timeWindow != null) {
                tp.setTimeWindow(timeWindow);
            }
            if (dataType != null) {
                tp.setDataType(dataType);
            }
            dataFetcher.execute(() -> refresh(tp));
        }
    }
    
    /**
     * Assigns plotter to a region.
     * This method should always be called on the EDT.
     * 
     * @param plotter Plotter to be displayed.
     * @param region Region where the plotter should be displayed.
     */
    static void showPlotter(TrendPlotter plotter, PlotRegion region) {
        Plotter oldPlotter = region.currentPlot();
        if (oldPlotter != null) {
            oldPlotter.clear();
            region.clear();
        }
        plotter.setRegion(region);
        region.showPlot(plotter);
    }
    
    /**
     * Displays data for the specified channel in {@code TextPage}.
     * 
     * @param channel Data channel to be displayed.
     * @param option EXIST - If there is a text page with this channel open, set selected time window and refresh it;
     *               NEWPAGE - open new text page;
     *               OVERLAY - add to currently selected text page.
     */
    void text(Trend.Descriptor channel, TrendPlotter.Option option) {
        
        if (channel == null) return;
        
        // initialize infrastructure if not already initialized:
        
        startTrending();
        
        // fetch currently selected time window
        
        TimeWindow timeWindow = getSelectedTimeWindow();
        
        // open or clear the page if necessary
        
        TextPage page = switch (option) {
            case EXIST -> {
                for (TextPage p : textPages) {
                    Trend.Descriptor[] trends = p.getDescriptor().getTrends();
                    if (trends.length == 1 && trends[0].equals(channel)) {
                        yield p;
                    }
                }
                yield null;
            }
            case OVERLAY -> {
                Component c = getConsole().getPanelManager().getSelectedPanel(PanelType.DATA);
                yield  (c instanceof TextPage p) ? p : null;
            }
            default -> null;
        };
        
        if (page == null) {
            if (timeWindow != null) {
                page = openTextPage(null);
                page.setTimeWindow(timeWindow);
            }
        } else {
            TimeWindow pageTimeWindow = page.getTimeWindow();
            if ((pageTimeWindow == null || !pageTimeWindow.equalsTime(timeWindow)) && timeWindow != null) {
                page.setTimeWindow(timeWindow);
            }
        }
        if (page == null) return;
                
        // bring to front
        
        // FIXME:
        
        // add trend
        
        page.add(channel);
    }
// </editor-fold>
    
// -- Implementing TrendingService : -------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    @Override
    public void show(Trend.Descriptor trendDescriptor) {
        ThreadUtil.invokeLater(() -> plot(trendDescriptor, TrendPlotter.Option.EXIST, TrendPlotter.Option.NEWPAGE));
    }

    @Override
    public void show(TrendPlotter.Descriptor plotterDescriptor) {
        TrendPage.Descriptor pageDesc = new TrendPage.Descriptor();
        
        pageDesc.setRows(1);
        pageDesc.setColumns(1);
        pageDesc.setPlotters(new TrendPlotter.Descriptor[] {plotterDescriptor});
        show(pageDesc);
    }

    @Override
    public void show(TrendPage.Descriptor pageDescriptor) {
        ThreadUtil.invokeLater(() -> {
            PersistenceService service = getConsole().getSingleton(PersistenceService.class);
            try {
                TrendPage page = (TrendPage) service.make(pageDescriptor);
                pages.add(page);
            } catch (Throwable x) {
            }
        });
    }
// </editor-fold>

// -- Implementing PlotterProvider : -------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    @Override
    public boolean supports(Class klass) {
        return Trend.class.isAssignableFrom(klass);
    }

    @Override
    public TrendPlotter create() {
        TrendPlotter plotter = new TrendPlotter();
        DataType dataType = this.dataTypeSelector.getSelectedDataType();
        plotter.setDataType(dataType);
        TimeWindow timeWindow = this.timeWindowSelector.getSelectedTimeWindow();
        plotter.setTimeWindow(timeWindow);
        return plotter;
    }
// </editor-fold>    
    
// -- Saving/Restoring : -------------------------------------------------------

// <editor-fold defaultstate="collapsed">
    @Override
    public ComponentDescriptor save() {
        
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        if (restSource == null) return desc;
        
        // Save selected time window
        
        TimeWindow tw = getSelectedTimeWindow();
        if (tw != null) {
            desc.setTimeWindow(tw.toNamedCompressedString());
        }
        
        // Save auto refresh state
        
        desc.setAutoRefresh(refreshMan != null);
        
        // Save trending panels
        
        int n = pages.size();
        TrendPage.Descriptor[] panels = new TrendPage.Descriptor[n];
        for (int i=0; i<n; i++) {
            panels[i] = pages.get(i).save();
        }
        desc.setPages(panels);
        
        // Save control panels
        
        n = controlPanels.size();
        ControlPanel.Descriptor[] dd = new ControlPanel.Descriptor[n];
        for (int i=0; i<n; i++) {
            dd[i] = controlPanels.get(i).save();
        }
        desc.setControlPages(dd);
        
        // Save text page
        
        n = textPages.size();
        TextPage.Descriptor[] tdd = new TextPage.Descriptor[n];
        for (int i=0; i<n; i++) {
            tdd[i] = textPages.get(i).save();
        }
        desc.setTextPages(tdd);
        
        return desc;
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        if (!(storageBean instanceof Descriptor)) {
            throw new IllegalArgumentException("Illegal descriptor type: "+ storageBean.getClassName());
        }
        Descriptor desc = (Descriptor) storageBean;
        String twString = desc.getTimeWindow();
        if (twString != null) {
            restoring = true;
        }
        
        // Stop current activities
        
        ArrayList<TrendPage> panels = new ArrayList<>(pages);
        panels.forEach(page -> page.hidePage());
        
        ArrayList<ControlPanel> controls = new ArrayList<>(controlPanels);
        controls.forEach(cp -> closeControlPanel(cp));
        
        ArrayList<TextPage> tPages = new ArrayList<>(textPages);
        tPages.forEach(p -> closeTextPage(p));
        
        restoring = false;
        
        if (twString == null) {
            stopTrending();
            return true;
        }
        startTrending();
        
        // Restore selected time window
        
        TimeWindow selectedTimeWindow = TimeWindow.parseCompressedString(twString);
        if (selectedTimeWindow != null) {
            timeWindowSelector.setSelectedTimeWindow(selectedTimeWindow);
        }
        
        // Restore pages
        
        for (ControlPanel.Descriptor d : desc.getControlPages()) {
            openControlPanel(d);
        }
        
        for (TrendPage.Descriptor pageDesc : desc.getPages()) {
            PersistenceService service = getConsole().getSingleton(PersistenceService.class);
            try {
                TrendPage page = (TrendPage) service.make(pageDesc);
                pages.add(page);
            } catch (Throwable x) {
                getConsole().getLogger().warning("Failed to restore a trending page", x);
            }
        }
        
        for (TextPage.Descriptor d : desc.getTextPages()) {
            TextPage p = openTextPage(d);
            p.refresh();
        }
        
        // Restore auto-refresh state
        
        if (desc.isAutoRefresh()) {
            autoRefreshMenu.setSelected(true);
        }
        
        return true;
    }
    
    static public class Descriptor extends ComponentDescriptor {

        private String timeWindow;
        private boolean autoRefresh;
        private TrendPage.Descriptor[] pages;
        private ControlPanel.Descriptor[] controlPages;
        private TextPage.Descriptor[] textPages;        
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public String getTimeWindow() {
            return timeWindow;
        }

        public void setTimeWindow(String timeWindow) {
            this.timeWindow = timeWindow;
        }

        public TrendPage.Descriptor[] getPages() {
            return pages;
        }

        public void setPages(TrendPage.Descriptor[] pages) {
            this.pages = pages;
        }

        public TrendPage.Descriptor getPages(int index) {
            return this.pages[index];
        }

        public void setPages(int index, TrendPage.Descriptor pages) {
            this.pages[index] = pages;
        }

        public boolean isAutoRefresh() {
            return autoRefresh;
        }

        public void setAutoRefresh(boolean autoRefresh) {
            this.autoRefresh = autoRefresh;
        }

        public ControlPanel.Descriptor[] getControlPages() {
            return controlPages;
        }

        public void setControlPages(ControlPanel.Descriptor[] controlPages) {
            this.controlPages = controlPages;
        }

        public TextPage.Descriptor[] getTextPages() {
            return textPages;
        }

        public void setTextPages(TextPage.Descriptor[] textPages) {
            this.textPages = textPages;
        }
        
    }
// </editor-fold>
    
// -- Local methods : ----------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    
    private ArrayList<Trend> findTrends(Trend.Descriptor channel) {
        String path = channel.getPath();
        String displayPath = channel.getDisplayPath();
        ArrayList<Trend> out = new ArrayList<>();
        pages.forEach(page -> {
            int n = page.numberOfRegions();
            for (int i=0; i<n; i++) {
                PlotRegion region = page.region(i);
                Plotter plotter = region.currentPlot();
                if (plotter != null && plotter instanceof TrendPlotter) {
                    List<Object> data = plotter.getData();
                    for (Object d : data) {
                        Trend trend = (Trend) d;
                        if (trend.getDescriptor().getPath().equals(path) && trend.getDescriptor().getDisplayPath().equals(displayPath)) {
                            out.add(trend);
                            break;
                        }
                    }
                }
            }
        });
        return out;
    }
    
    private Trend findTrend(Trend.Descriptor channel, TimeWindow timeWindow) {
        ArrayList<Trend> all = findTrends(channel);
        Trend best = null;
        int score = 0;
        for (Trend candidate : all) {
            TrendPlotter plotter = candidate.getPlotter();
            if (plotter.getData().size() == 1) {
                int candidateScore = candidate.getTimeWindow() == timeWindow ? 1 : 0;
                candidateScore = candidateScore<<1 + (candidate.getDescriptor().getPath().equals(channel.getPath()) ? 1 : 0);
                candidateScore = candidateScore<<3 +  plotter.getRegion().getPage().numberOfRegions();
                if (candidateScore > score) {
                    best = candidate;
                    score = candidateScore;
                }
            }
        }
        return best;
    }
    
    /**
     * Fetches data and updates the trend.
     * Should be called on an executor.
     */
    private void refresh(Trend trend) {
        long now = System.currentTimeMillis();
        RestSource source = getTrendingSource();
        if (source != null) {
            trend.setLoading(true);
            TrendPlotter plot = trend.getPlotter();
            TimeWindow timeWindow;
            DataType dataType;
            if (plot == null) {
                timeWindow = getSelectedTimeWindow();
                dataType = getSelectedDataType();
            } else {
                timeWindow = plot.getTimeWindow();
                dataType = plot.getDataType();
            }
            List<TrendData> data = source.get(Collections.singletonList(trend), timeWindow.getLowerEdge(now), timeWindow.getUpperEdge(now), dataType.getRaw(), dataType.getBins());
            trend.setData(data.get(0));
        }
    }
    
    /**
     * Fetches data and updates all trends in the specified plotter.
     * Should be called on an executor.
     */
    void refresh(TrendPlotter plot) {
        List<Trend> trends = plot.getTrends();
        long now = System.currentTimeMillis();
        RestSource source = getTrendingSource();
        if (source != null) {
            trends.forEach(trend -> {
                trend.setLoading(true);
            });
            TimeWindow tw = plot.getTimeWindow();
            DataType dataType = plot.getDataType();
            List<TrendData> data = source.get(trends, tw.getLowerEdge(now), tw.getUpperEdge(now), dataType.getRaw(), dataType.getBins());
            Iterator<Trend> it = trends.iterator();
            for (int i = 0; it.hasNext(); i++) {
                Trend trend = it.next();
                TrendData td = data.get(i);
                trend.setData(td);
            }
        }
    }
    
    private void openAgentControlPanel(String agentName) {
        ControlPanel.Descriptor cpDesc = new ControlPanel.Descriptor();
        cpDesc.setName(agentName);
        PersistableAgentChannelsFilter.Descriptor filterDesc = new PersistableAgentChannelsFilter.Descriptor();
        Creator.Descriptor creDesc = new Creator.Descriptor(AgentChannelsFilter.CATEGORY, SubsystemSelectorFilter.CREATOR_PATH, agentName);
        filterDesc.setCreator(creDesc);
        cpDesc.setFilter(filterDesc);
        openControlPanel(cpDesc);
        
    }
    
    private void createPage(int rows, int columns) {
        TrendPage page = createPage("Trending");
        page.createRegions(rows, columns);
        page.showPage();
    }
    
    private int[] customPageDialog(Component parent) {
        
        JPanel panel = new JPanel();    
        panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
        panel.setAlignmentX(LEFT_ALIGNMENT);
        
        Box row = Box.createHorizontalBox();
        row.setAlignmentX(LEFT_ALIGNMENT);
        row.add(new JLabel("Rows: "));
        JFormattedTextField rowsField = new JFormattedTextField(1);
        row.add(rowsField);
        row.add(Box.createHorizontalGlue());
        panel.add(row);
        
        row = Box.createHorizontalBox();
        row.setAlignmentX(LEFT_ALIGNMENT);
        row.add(new JLabel("Columns: "));
        JFormattedTextField columnsField = new JFormattedTextField(1);
        row.add(columnsField);
        row.add(Box.createHorizontalGlue());
        panel.add(row);
        
        int response = JOptionPane.showConfirmDialog(parent, panel, "Custom trending page", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE);
        if (response == JOptionPane.OK_OPTION) {
            try {
                int[] out = new int[2];
                out[0] = (int) rowsField.getValue();
                out[1] = (int) columnsField.getValue();
                if (out[0] < 1 || out[0] > 10 || out[1] < 1 || out[1] > 10) {
                    getConsole().error("Illegal number of regions.\nTrending page cannot have more than 10 rows or columns.");
                    return null;
                }
                return out;
            } catch (Exception x) {
                return null;
            }
        } else {
            return null;
        }
    }

// </editor-fold>

// -----------------------------------------------------------------------------
    
}
