package org.lsst.ccs.gconsole.plugins.tracer.filters;

import java.util.HashSet;
import java.util.logging.Level;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusAgentInfo;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusStateBundle;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.plugins.tracer.FilteredMessage;
import org.lsst.ccs.gconsole.plugins.tracer.MessageFilter;
import org.lsst.ccs.gconsole.plugins.tracer.Tracer;

/**
 * Standard configurable filter for message viewers.
 *
 * @author onoprien
 */
public class StandardConfigurableFilter implements MessageFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final HashSet<String> agents;
    private final int logLevel;
    private final StandardFormatter formatter;

// -- Life cycle : -------------------------------------------------------------
    
    public StandardConfigurableFilter(String agent, Level level, boolean brief) {
        if (agent == null || agent.trim().isEmpty()) {
            agents = null;
        } else {
            String[] ss = agent.split(",");
            HashSet<String> aa = new HashSet<>(ss.length*2);
            for (String s : ss) {
                s = s.trim();
                if (!s.isEmpty()) {
                    aa.add(s);
                }
            }
            agents = aa.isEmpty() ? null : aa;
        }
        if (level == null) level = Level.INFO;
        logLevel = level.intValue();
        formatter = new StandardFormatter(brief, agents == null);
    }
    
// -- Filtering : --------------------------------------------------------------

    @Override
    public FilteredMessage apply(FilteredMessage filteredMessage) {
        
        BusMessage bm = filteredMessage.getMessage();
        
        if (agents != null && !agents.contains(bm.getOriginAgentInfo().getName())) {
            return null;
        }
        
        if (bm instanceof LogMessage) {
            
            String level = ((LogMessage)bm).getLevel();
            if (parseLevel(level).intValue() < logLevel) return null;
            
        } else if (bm instanceof StatusHeartBeat || bm instanceof StatusAgentInfo || bm instanceof StatusStateBundle) {
            
            return null;
            
        } else if (bm instanceof CommandAck) {
            
            return null;
            
        }

        filteredMessage = formatter.apply(filteredMessage);
        return filteredMessage;
    }
    
// -- Static utility methods : -------------------------------------------------
    
    static public Level parseLevel(String level) {
        if (level == null) return Level.ALL;
        switch (level.toUpperCase()) {
            case "OFF": case "O":
                return Level.OFF;
            case "SEVERE": case "FATAL": case "ERROR": case "S": case "E":
                return Level.SEVERE;
            case "WARNING": case "WARN": case "W":
                return Level.WARNING;
            case "INFO": case "I":
                return Level.INFO;
            case "CONFIG": case "C":
                return Level.CONFIG;
            case "FINE": case "DEBUG": case "F": case "D":
                return Level.FINE;
            case "FINER":
                return Level.FINER;
            case "FINEST": case "TRACE": case "T":
                return Level.FINEST;
            case "ALL": case "A":
                return Level.ALL;
            default:
                return Level.ALL;
        }
    }
    
// -- Creators : ---------------------------------------------------------------
    
    @Create(category = Tracer.CATEGORY,
            name = "Default",
            path = "Built-In/Core/Default",
            description = "Default message viewer that displays a minimal set of messages for each subsystem, with default formatting.")
    static public Tracer defaultTracer(
            @Par(def = Par.NULL, desc = "Viewer name. Optional.") String name,
            @Par(def = Par.NULL, desc = "Subsystem name. If not given, all subsystems are included.") String subsystem)
    {
        Tracer tracer = new Tracer();
        tracer.getDescriptor().setName(name);
        tracer.setFilter(new StandardConfigurableFilter(subsystem, Level.INFO, true));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "Messages",
            path = "Built-In/Core/Standard",
            description = "Standard configurable message viewer, with default formatting.")
    static public Tracer standardTracer(
            @Par(def = Par.NULL, desc = "Viewer name. Optional.") String name,
            @Par(def = Par.NULL, desc = "Comma-separated list of subsystem names. If not given, all subsystems are included.") String subsystem,
            @Par(def = "INFO", desc = "Log messages at or above this level are displayed.") String level,
            @Par(def = "true", desc = "Use brief formatting.") boolean brief)
    {
        Tracer tracer = new Tracer();
        tracer.getDescriptor().setName(name);
        tracer.setFilter(new StandardConfigurableFilter(subsystem, parseLevel(level), brief));
        return tracer;
    }
    
}
