package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.CommandMessageListener;
import org.lsst.ccs.messaging.LogMessageListener;
import org.lsst.ccs.messaging.StatusMessageListener;

/**
 * Graphical console plugin that creates and manages message viewers.
 * 
 * @author onoprien
 */
@Plugin(name="LSST Message Viewer Plugin",
        id="tracer21",
        description="LSST Message Viewer tool.")
public class LsstTracerPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    private final CopyOnWriteArrayList<Tracer> tracers = new CopyOnWriteArrayList<>();
    private final AgentPresenceListener agentConnectionListener;
    private final StatusMessageListener statusListener;
    private final LogMessageListener logListener;
    private final CommandMessageListener commandListener;


// -- Life cycle : -------------------------------------------------------------
    
    public LsstTracerPlugin() {
        
        statusListener = m -> onMessage(m);
        logListener = m -> onMessage(m);
        commandListener = m -> onMessage(m);
        
        agentConnectionListener = new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    AgentInfo.AgentType type = agent.getType();
                    if (!(AgentInfo.AgentType.CONSOLE.equals(type) || AgentInfo.AgentType.LISTENER.equals(type))) {
                        SwingUtilities.invokeLater(() -> {
                            Action action = new AbstractAction(agent.getName()) {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    openDefaultTracer(agent.getName());
                                }
                            };
                            getServices().addMenu(action, "400: CCS Tools :-1:2", "Message Viewer:100:100", "Subsystems:1");
                            action = new AbstractAction("Message Viewer") {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    openDefaultTracer(agent.getName());
                                }

                            };
                            getServices().addMenu(action, "CCS Subsystems", agent.getName() + ":-10:2");
                        });
                    }
                }
            }
            @Override
            public void disconnected(AgentInfo... agents) {
                SwingUtilities.invokeLater(() -> {
                    for (AgentInfo agent : agents) {
                        Console.getConsole().removeMenu(" CCS Tools ", "Message Viewer", "Subsystems", agent.getName());
                        Console.getConsole().removeMenu("CCS Subsystems", agent.getName(), "Message Viewer");
                    }
                });
            }
        };
    }
    
    @Override
    public void initialize() {
        Action act = new AbstractAction("New...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                try {
                    Tracer tracer = new Tracer();
                    tracer.getDescriptor().setName("Messages");
                    tracer = tracer.edit("New Message Viewer", null);
                    openTracer(tracer);
                } catch (RuntimeException x) {
                }
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:2", "Message Viewer:1");
        act = new AbstractAction("Load...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                PersistenceService service = PersistenceService.getService();
                try {
                    Tracer tracer = (Tracer) service.make(null, "Load message viewer", getConsole().getWindow(), Tracer.CATEGORY);
                    openTracer(tracer);
                } catch (RuntimeException x) {
                }
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:2", "Message Viewer:2");
        act = new AbstractAction("Default") {
            @Override
            public void actionPerformed(ActionEvent e) {
                openDefaultTracer(null);
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:2", "Message Viewer:3");
    }

    @Override
    public void start() {
        AgentMessagingLayer messagingAccess = getConsole().getMessagingAccess();
        messagingAccess.addStatusMessageListener(statusListener);
        messagingAccess.addCommandMessageListener(commandListener);
        messagingAccess.addLogMessageListener(logListener);
        getConsole().getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(agentConnectionListener);
    }

    @Override
    public void stop() {
        try {
            getConsole().getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(agentConnectionListener);
            AgentMessagingLayer messagingAccess = getConsole().getMessagingAccess();
            messagingAccess.removeStatusMessageListener(statusListener);
            messagingAccess.removeCommandMessageListener(commandListener);
            messagingAccess.removeLogMessageListener(logListener);
        } catch (Exception x) {
        }
    }
    
    
// -- Handling tracers : -------------------------------------------------------

    public void openTracer(Tracer.Descriptor desc) {
        Tracer tracer = (Tracer) PersistenceService.getService().make(desc);
        openTracer(tracer);
    }

    public void openTracer(Tracer tracer) {
        if (tracer == null) return;
        if (!SwingUtilities.isEventDispatchThread()) {
            SwingUtilities.invokeLater(() -> openTracer(tracer));
            return;
        }
        JComponent panel = tracer.getPanel();
        if (panel == null) return;
        
        Tracer.Descriptor desc = tracer.getDescriptor();
        HashMap<Object, Object> par = new HashMap<>();
        if (desc != null) {
            DataPanelDescriptor panDesc = desc.getPanel();
            if (panDesc != null && panDesc.isOpen()) {
                Map<String, Serializable> data = panDesc.getData();
                if (data != null) {
                    par.putAll(data);
                }
            }
        }
        par.put(Panel.TITLE, tracer.getDescriptor().getName());
        
        Consumer<JComponent> onClose = c -> {
            tracers.removeIf(t -> t.getPanel() == c);
        };
        par.put(Panel.ON_CLOSE, onClose);
        
        Consumer<JComponent> onSaveAs = c -> {
            for (Tracer t : tracers) {
                if (c == t.getPanel()) {
                    PersistenceService service = PersistenceService.getService();
                    service.saveAs(t.save(), "Save Message Viewer page", null);
                    break;
                }
            }
        };
        par.put(Panel.ON_SAVE_AS, onSaveAs);
        
        Consumer<JComponent> onEdit = c -> {
            for (Tracer t : tracers) {
                if (c == t.getPanel()) {
                    try {
                        Tracer editedTracer = t.edit("Edit page", null);
                        if (editedTracer != null && editedTracer != t) {
                            SwingUtilities.invokeLater(() -> {
                                try {
                                    getConsole().getPanelManager().close(c);
                                } catch (IndexOutOfBoundsException x) {
                                }
                                openTracer(editedTracer);
                            });
                        }
                    } catch (RuntimeException x) {
                        // edit cancelled
                    }
                    break;
                }
            }
        };
        par.put(Panel.ON_EDIT, onEdit);
        
        Console.getConsole().getPanelManager().open(panel, par);
        tracers.add(tracer);
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void openDefaultTracer(String agentName) {
        PersistenceService service = PersistenceService.getService();
        Tracer tracer = (Tracer) service.make(Tracer.CATEGORY, "Built-In/Core/Default", agentName, agentName);
        openTracer(tracer);
    }
    
    private void onMessage(BusMessage message) {
        tracers.forEach(tracer -> { // FIXME: consider offloading message processing to a separate thread
            try {
                tracer.onMessage(message);
            } catch (Exception x) {
            }
        });
    }

    
// -- Saving/restoring : -------------------------------------------------------

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        if (! (storageBean instanceof Descriptor)) return true;
        
        Descriptor desc = (Descriptor) storageBean;
        if (!tracers.isEmpty()) {
            ArrayList<Tracer> copy = new ArrayList<>(tracers);
            for (Tracer t : copy) {
                JComponent panel = t.getPanel();
                if (panel != null) {
                    getConsole().getPanelManager().close(panel);
                }
            }
            tracers.clear();
        }
                
        Tracer.Descriptor[] dd = desc.getTracers();
        if (dd != null) {
            for (Tracer.Descriptor d : dd) {
                openTracer(d);
            }
        }
        return true;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        if (!tracers.isEmpty()) {
            Tracer.Descriptor[] descriptors = new Tracer.Descriptor[tracers.size()];
            for (int i = 0; i < descriptors.length; i++) {
                descriptors[i] = tracers.get(i).save();
            }
            desc.setTracers(descriptors);
        }
        return desc;
    }

    static public class Descriptor extends ComponentDescriptor {

        private Tracer.Descriptor[] tracers;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public Tracer.Descriptor[] getTracers() {
            return tracers;
        }

        public void setTracers(Tracer.Descriptor[] tracers) {
            this.tracers = tracers;
        }
        
    }

}
