package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.Color;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.gconsole.services.persist.Creator;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;

/**
 * Step in a {@link MultistepFilter}.
 * Instances of this class are immutable.
 *
 * @author onoprien
 */
public class FilterStep {
    
    /**
     * Enumeration of filter application modes.
     * The mode determines conditions for accepting the message and passing it to subsequent steps,
     * as well as conditions for modifying the message (formatting, actions, flags, etc).
     */
    public enum Mode {
        
        /**
         * Only accept messages that satisfy this step.
         * Messages to which this filter cannot be applied are rejected.
         */
        ON("Accept and modify messages that satisfy this step", "On"),
        /**
         * Accept a message if this step is either satisfied or inapplicable, modify only if this step is satisfied.
         * For example, if this step is configured to use the embedded object as a target, but the object could not be
         * deserialized since its class definition is not available to this console, the message is accepted and
         * passed to subsequent steps, but any formatting and actions that could be applied by this step are skipped.
         */
        TRY("Accept a message if this step is either satisfied or inapplicable, modify only if this step is satisfied", "Try"),
        /**
         * Accept all messages, modify only those that satisfy this step.
         */
        PASS("Accept all messages, modify only those that satisfy this step", "Pass"),
        /**
         * Skip this step.
         */
        OFF("Skip this step", "Off");
        
        private final String toolTip;
        private final String hr;
        
        Mode(String toolTip, String humanReadable) {
            this.toolTip = toolTip;
            hr = humanReadable;
        }
        
        public String getToolTip() {
            return toolTip;
        }
        
        @Override
        public String toString() {
            return hr;
        }
    }
    
    /** Enumeration of targets to which the step operation can be applied. */
    public enum Target {
        
        /** {@code BusMessage} object. */
        MESSAGE("The operation is applied to the bus message object. If the operation <br>"
             +  " requires a string as a target, the message is converted to a string.",
             "Message"),
        
        /** Deserialized object embedded into the {@code BusMessage}. */
        OBJECT("The operation is applied to the deserialized object embedded<br>"
             + " into the bus message. If the operation requires<br>"
             + " a string as a target, the object is converted to a string.",
             "Object"),
        
        /** Name of the source subsystem. */
        SOURCE("The operation is applied to the name of the source subsystem.", "Source"),
        
        /** String produced by previous steps. */
        TEXT("The operation is applied to the string produced by formatters embedded in<br>"
           + " previously applied steps. If none of the previous steps formatted the<br>"
           + " message, the BusMessage is converted to String by calling its toString() method."
           + "",
           "Text"),
        
        /** Comma-separated list of flags attached to the message. */
        FLAG("The filter is applied to the comma-separated list of flags attached<br>"
           + " to the message.",
           "Flag"),
        
        /** Expanded template. */
        TEMPLATE("The operation is applied to the string produced<br>"
               + " by expanding the template based on contents of the message.<br>"
               + " If expanding fails, the filter is not satisfied. The template<br>"
               + " may include place holders in the <tt>${key}</tt> format,<br>"
               + " where <tt>key</tt> is a name of a subsystem property, or one<br>"
               + " of the following:<br>"
               + " <b>SOURCE:</b> Name of the source subsystem<br>"
               + " <b>SOURCE_TYPE:</b> Type of the source subsystem<br>"
               + " <b>TEXT</b> String formatted by previous steps<br>"
               + " <b>DATE</b> Message date in 'yyyy-MM-dd' format<br>"
               + " <b>TIME</b> Message time in 'HH:mm:ss' format<br>"
               + " <b>DT</b> Message date and time in 'yyyy-MM-dd HH:mm:ss' format<br>"
               + " <b>LOGGER</b> Name of the logger<br>"
               + " <b>LEVEL</b> Level of the log message<br>"
               + " <b>MESSAGE</b> Formatted message associated with the bus message<br>"
               + " <b>STATE</b> Subsystem state embedded in the status message<br>"
               + " <b>DESTINATION</b> Destination of the command message<br>"
               ,"Template");
        
        private final String toolTip;
        private final String hr;
        
        Target(String toolTip, String humanReadable) {
            this.toolTip = toolTip;
            hr = humanReadable;
        }
        
        public String getToolTip() {
            return toolTip;
        }
        
        @Override
        public String toString() {
            return hr;
        }
    }
    
    /**
     * Enumeration of operations that can be applied to targets.
     */
    public enum Method {
        /** The parameter string is a regular expression the target should match. */
        REGEX("The parameter string is a regular expression the target should match.", "Reg Ex"),
        /** The parameter string is a Unix-style wildcard the target should match. */
        WILDCARD("The parameter string is a Unix-style wildcard the target should match.", "Wildcard"),
        /** The target should contain the parameter string. */
        CONTAINS("The target should contain the parameter string.", "Contains"),
        /** The target should be equal to the parameter string. */
        EQUALS("The target should be equal to the parameter string.", "Equals"),
        /** The target should be an instance of the class specified by the parameter string, or of its subclass. */
        CLASS("The target should be an instance of the class specified by the parameter<br>"
             + " string, or of its subclass. Full and short class names can be used.", "Class"),
        /** The definition string is the path of the filter that should be applied to the target. */
        NAME("Load a built-in or previously saved filter.", "External"){
            @Override public boolean needsTarget() {return false;}
        };
        
        private final String toolTip;
        private final String hr;
        
        Method(String toolTip, String humanReadable) {
            this.toolTip = toolTip;
            hr = humanReadable;
        }
        
        public String getToolTip() {
            return toolTip;
        }
        
        public boolean needsTarget() {
            return true;
        }
        
        @Override
        public String toString() {
            return hr;
        }
    }

// -- Fields : -----------------------------------------------------------------
  
    static final private Pattern pPar = Pattern.compile("\\$\\{([^}]+)\\}");
    
    static final private SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
    static final private SimpleDateFormat timeFormat = new SimpleDateFormat("HH:mm:ss");
    static final private SimpleDateFormat dtFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    static final private HashMap<String,Function<FilteredMessage,String>> key2func;
    static {
        key2func = new HashMap<>();
        key2func.put("SOURCE", m -> m.getMessage().getOriginAgentInfo().getName());
        key2func.put("SOURCE_TYPE", m -> m.getMessage().getOriginAgentInfo().getType().name());
        key2func.put("TEXT", m -> m.getText());
        key2func.put("DATE", m -> dateFormat.format(new Date(m.getMessage().getCCSTimeStamp().getUTCInstant().toEpochMilli())));
        key2func.put("TIME", m -> timeFormat.format(new Date(m.getMessage().getCCSTimeStamp().getUTCInstant().toEpochMilli())));
        key2func.put("DT", m -> dtFormat.format(new Date(m.getMessage().getCCSTimeStamp().getUTCInstant().toEpochMilli())));
        key2func.put("LOGGER", m -> ((LogMessage)m.getMessage()).getLoggerName());
        key2func.put("LEVEL", m -> ((LogMessage)m.getMessage()).getLevel());
        key2func.put("MESSAGE", m -> ((LogMessage)m.getMessage()).getFormattedDetails());
        key2func.put("STATE", m -> ((StatusMessage)m.getMessage()).getState().toString());
        key2func.put("DESTINATION", m -> ((CommandMessage)m.getMessage()).getDestination());
    }
    
    private final Mode mode;
    private final boolean invert;
    private final boolean format;
    private final Target target;
    private final Method method;
    private final String[] code;
    private final Color color;
    private final FilteredMessage.Flag flag;
    
    private final Predicate<String> tester;
    private final Tracer delegate;
    private final String pattern;
    private final String[] template;

    
// -- Construction and initialization : ----------------------------------------
    
    public FilterStep(Mode mode, boolean invert, boolean format, Target target, Method method, String[] code, Color color, FilteredMessage.Flag flag) {

        this.mode = mode;
        this.invert = invert;
        this.format = format;
        this.target = target;
        this.method = method;
        this.flag = flag;
        this.color = color;
        this.code = code;
        
        try {
            switch (target) {
                case TEMPLATE:
                    if (code.length != 2) throw new IllegalArgumentException("Not a valid template");
                    String temp = code[0];
                    Matcher m = pPar.matcher(temp);
                    ArrayList<String> t = new ArrayList<>();
                    int s = 0;
                    while (m.find()) {
                        t.add(temp.substring(s, m.start()));
                        t.add(m.group(1));
                        s = m.end();
                    }
                    if (s < temp.length()) t.add(temp.substring(s));
                    template = t.toArray(new String[t.size()]);
                    pattern = code[1];
                    break;
                default:
                    template = null;
                    pattern = code[0];
            }
        } catch (IndexOutOfBoundsException x) {
            throw new IllegalArgumentException("Unable to create message viewer filter", x);
        }
        
        try {
            switch (method) {
                case REGEX:
                    tester = Pattern.compile(pattern).asPredicate();
                    delegate = null;
                    break;
                case WILDCARD:
                    String s = wildcardToRegex(pattern);
                    tester = Pattern.compile(s).asPredicate();
                    delegate = null;
                    break;
                case CONTAINS:
                    tester = targetString -> targetString.contains(pattern);
                    delegate = null;
                    break;
                case EQUALS:
                    tester = targetString -> targetString.equals(pattern);
                    delegate = null;
                    break;
                case NAME:
                    tester = null;
                    String path = code[0];
                    int n = code.length - 1;
                    String[] pars = new String[n];
                    for (int i=0; i<n; i++) {
                        pars[i] = code[i+1];
                    }
                    delegate = (Tracer) PersistenceService.getService().make(Tracer.CATEGORY, path, pars);
                    if (delegate == null) throw new IllegalArgumentException("Failed to load external filter");
                    break;
                case CLASS:
                    if (!pattern.matches("[a-zA-Z_$][a-zA-Z\\d_$.]+")) {
                        throw new IllegalArgumentException("Illegal class name");
                    }
                    tester = null;
                    delegate = null;
                    break;
                default:
                    tester = null;
                    delegate = null;
            }
        } catch (IllegalArgumentException | IndexOutOfBoundsException x) {
            throw new IllegalArgumentException("Unable to create message viewer filter", x);
        }
        
    }
    
    public FilterStep(Tracer delegate, Mode mode, boolean invert, boolean format, Color color, FilteredMessage.Flag flag) {
        this.mode = mode;
        this.invert = invert;
        this.format = format;
        this.target = Target.MESSAGE;
        this.method = Method.NAME;
        this.flag = flag;
        this.color = color;
        this.code = getCode(delegate);
        this.delegate = delegate;
        this.tester = null;
        this.pattern = null;
        this.template = null;
    }
    
    public FilterStep(Descriptor descriptor) {
        this(
             descriptor.getMode() == null ? Mode.ON : Mode.valueOf(descriptor.getMode()),
             descriptor.isInvert(),
             descriptor.isFormat(),
             descriptor.getTarget() == null ? null : Target.valueOf(descriptor.getTarget()),
             descriptor.getMethod() == null ? null : Method.valueOf(descriptor.getMethod()),
             descriptor.getCode(),
             descriptor.getColor() == null ? null : new Color(Integer.parseUnsignedInt(descriptor.getColor(), 16)),
             descriptor.getFlag() == null ? null : FilteredMessage.Flag.valueOf(descriptor.getFlag())
        );
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    public Mode getMode() {return mode;}
    
    public boolean isInverted() {return invert;}
    
    public boolean isFormatting() {return format;}
    
    public Target getTarget() {return target;}
    
    public Method getMethod() {return method;}
    
    public String[] getCode() {return code;}
    
    public Color getColor() {return color;}
    
    public FilteredMessage.Flag getFlag() {return flag;}

    public Tracer getDelegate() {
        return delegate;
    }

    
// -- Implementing MessageFilter : ---------------------------------------------

    public FilteredMessage apply(FilteredMessage filteredMessage) {
        
        if (mode == Mode.OFF) return filteredMessage;
        
        try {
        
            // prepare target to which the filtering will be applied
            
            Object targetObject = null;
            if (method.needsTarget()) {
                switch (target) {
                    case MESSAGE:
                        targetObject = filteredMessage.getMessage();
                        break;
                    case OBJECT:
                        targetObject = filteredMessage.getMessage().getObject();
                        break;
                    case SOURCE:
                        targetObject = filteredMessage.getMessage().getOriginAgentInfo().getName();
                        break;
                    case TEXT:
                        targetObject = filteredMessage.getText();
                        if (targetObject == null) {
                            targetObject = filteredMessage.getMessage().toString();
                        }
                        break;
                    case FLAG:
                        EnumSet<FilteredMessage.Flag> flags = filteredMessage.getFlags();
                        if (flags == null) {
                            targetObject = "";
                        } else {
                            StringBuilder sb = new StringBuilder();
                            flags.forEach(flag -> sb.append(",").append(flag));
                            targetObject = sb.substring(1);
                        }
                        break;
                    case TEMPLATE:
                        targetObject = expand(filteredMessage);
                        break;
                }
            }
        
            // apply method to target and set acceptance flag
        
            boolean accept;
            switch (method) {
                case REGEX:
                case WILDCARD:
                case CONTAINS:
                case EQUALS:
                    accept = tester.test(targetObject.toString());
                    break;
                case CLASS:
                    accept = false;
                    Class<?> c = targetObject.getClass();
                    while (c != null) {
                        if (pattern.equals(c.getName()) || pattern.equals(c.getSimpleName())) {
                            accept = true;
                            break;
                        }
                        c = c.getSuperclass();
                    }
                    break;
                case NAME:
                    FilteredMessage fm = new FilteredMessage(filteredMessage);
                    if (targetObject instanceof String) {
                        fm.setText((String) targetObject);
                    }
                    fm = delegate.getFilter().apply(fm);
                    if (fm == null) {
                        accept = false;
                    } else {
                        accept = true;
                        targetObject = fm;
                    }
                    break;
                default:
                    accept = true;
            }
                
            // invert acceptance flag if necessary
            
            if (invert) accept = !accept;

            // modify the message if accepted
        
            if (accept) {
                if (format) {
                    if (targetObject instanceof FilteredMessage) {
                        filteredMessage = (FilteredMessage) targetObject;
                    } else if (targetObject != null) {
                        filteredMessage.setText(targetObject.toString());
                    }
                }
                if (color != null) {
                    filteredMessage.setColor(color);
                }
                if (flag != null) {
                    filteredMessage.addFlag(flag);
                }
                return filteredMessage;
            } else {
                return mode == Mode.ON ? null : filteredMessage;
            }
        
        } catch (Throwable t) { // filter inapplicable
            
            return mode == Mode.TRY ? filteredMessage : null;
            
        }
        
    }
    
                
// -- Local and utility methods : ----------------------------------------------
    
    private String expand(FilteredMessage message) {
        StringBuilder sb = new StringBuilder();
        for (int i=0; i<template.length; ) {
            sb.append(template[i++]);
            if (i<template.length) {
                sb.append(expand(template[i++], message));
            }
        }
        return sb.toString();
    }
    
    private String expand(String key, FilteredMessage message) {
        
        // pre-defined keys
        
        Function<FilteredMessage,String> func = key2func.get(key);
        if (func != null) {
            return func.apply(message);
        }
        
        // agent properties
        
        String out = message.getMessage().getOriginAgentInfo().getAgentProperty(key);
        if (out != null) return out; 
        
        // added filtered message properties // FIXME: should we implement this?

//        if (out == null) {
//            Object o = message.getProperty(key);
//            if (o != null) out = o.toString();
//        }

        throw new RuntimeException(); // filter inapplicable
    }
    
    static private String wildcardToRegex(String wildcard) {
        StringBuilder s = new StringBuilder();
//        s.append('^');
        for (int i = 0, is = wildcard.length(); i < is; i++) {
            char c = wildcard.charAt(i);
            switch (c) {
                case '*':
                    s.append(".*");
                    break;
                case '?':
                    s.append(".");
                    break;
                case '(':
                case ')':
                case '[':
                case ']':
                case '$':
                case '^':
                case '.':
                case '{':
                case '}':
                case '|':
                case '\\':
                    s.append("\\");
                    s.append(c);
                    break;
                default:
                    s.append(c);
                    break;
            }
        }
//        s.append('$');
        return (s.toString());
    }
    
    static public String[] getCode(Tracer tracer) {
        String path = tracer.getDescriptor().getPath();
        if (path == null) {
            Creator.Descriptor cd = tracer.getDescriptor().getCreator();
            path = cd.getPath();
            String[] pars = cd.getParameters();
            if (pars == null) {
                return new String[] {path};
            } else {
                int n = pars.length;
                String[] out = new String[n+1];
                out[0] = path;
                System.arraycopy(pars, 0, out, 1, n);
                return out;
            }
        } else {
            return new String[] {path};
        }
    }
    
    
// -- Saving and restoring : ---------------------------------------------------
    
    public Descriptor save() {
        Descriptor desc = new Descriptor();
        if (mode != null && mode != Mode.ON) desc.setMode(mode.name());
        desc.setInvert(invert);
        desc.setFormat(format);
        if (target != null) desc.setTarget(target.name());
        if (method != null) desc.setMethod(method.name());
        desc.setCode(code);
        if (color != null) desc.setColor(Integer.toHexString(color.getRGB()));
        if (flag != null) desc.setFlag(flag.name());
        return desc;
    }
    
    static public class Descriptor implements Serializable, Cloneable {

        private String mode;
        private boolean invert;
        private boolean format;
        private String target;
        private String method;
        private String[] code;
        private String color;
        private String flag;

        public String getFlag() {
            return flag;
        }

        public void setFlag(String flag) {
            this.flag = flag;
        }

        public String getMode() {
            return mode;
        }

        public void setMode(String mode) {
            this.mode = mode;
        }

        public boolean isInvert() {
            return invert;
        }

        public void setInvert(boolean invert) {
            this.invert = invert;
        }

        public boolean isFormat() {
            return format;
        }

        public void setFormat(boolean format) {
            this.format = format;
        }

        public String getTarget() {
            return target;
        }

        public void setTarget(String target) {
            this.target = target;
        }

        public String getMethod() {
            return method;
        }

        public void setMethod(String method) {
            this.method = method;
        }

        public String getColor() {
            return color;
        }

        public void setColor(String color) {
            this.color = color;
        }

        public String[] getCode() {
            return code;
        }

        public void setCode(String[] code) {
            this.code = code;
        }

        @Override
        protected Descriptor clone() {
            try {
                return (Descriptor) super.clone();
            } catch (CloneNotSupportedException x) {
                return null; // never
            }
        }
        
    }

}
