package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import javax.swing.JComponent;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;
import org.lsst.ccs.gconsole.services.persist.Persistable;

/**
 * A monitoring data view that manages several child views.
 * This class is primarily intended as a base class for specific compound views
 * (see {@link DefaultView} for an example of use).
 * <p>
 * The current implementation supports two simple child view modes:<dl>
 * <dt>INDEPENDENT<dd>Listens to the status aggregator independently, with its own filter.
 * <dt>FORWARD<dd>Receives events forwarded by the parent view filter.
 </dl>
 * In the future, more complex modes calling {@link AbstractMonitorView} SPI on
 * child views can be added if necessary.
 * <p>
 * Calls to setFilter(...) are forwarded to children as well.
 * That means children added before and after this call are treated differently.
 * All setters should be called before {@code install()}.
 *
 * @author onoprien
 */
public class CompoundView implements PersistableMonitorView {
    
    /**
     * Enumeration of management modes for child views.
     */
    public enum Mode {
        
        /**
         * Listens to the status aggregator independently, possibly with its own filter.
         * When the compound view is installed or un-installed,  {@code install()} 
         * or {@code uninstall()} methods of its children added in this mode
         * are also called. The {@code setFilter(...)} method is forwarded as well.
         * Otherwise, child views of this type are independent, updates received by
         * the parent are not forwarded to them.
         */
        INDEPENDENT,
        
        /**
         * Receives events filtered by the compound view filter.
         * Child views of this type are not independently installed, status aggregator
         * events received by the parent view are forwarded to them.
         */
        FORWARD
    }

// -- Fields : -----------------------------------------------------------------
    
    protected Descriptor descriptor = new Descriptor();
    
    private final JComponent panel;
    private PersistableAgentChannelsFilter filter;
    private final ArrayList<PersistableMonitorView> children1 = new ArrayList<>(0); // INDEPENDENT
    private final ArrayList<PersistableMonitorView> children2 = new ArrayList<>(0); // FORWARD

// -- Life cycle : -------------------------------------------------------------
    
    public CompoundView(JComponent panel) {
        this.panel = panel;
    }
    

// -- Getters and setters : ----------------------------------------------------

    /**
     * Returns the graphical component maintained by this view.
     * @return Graphical component to be displayed by the GUI.
     */
    @Override
    public JComponent getPanel() {
        return panel;
    }

    /**
     * Sets a channels filter.
     * The call is forwarded to children added in {@code INDEPENDENT} and {@code FORWARD} modes.
     * 
     * @param filter Channels filter to be used by this view.
     */
    @Override
    public void setFilter(AgentChannelsFilter filter) {
        if (!(filter instanceof PersistableAgentChannelsFilter)) {
            throw new IllegalArgumentException(getClass().getSimpleName() +" requires PersistableAgentChannelsFilter, found "+ filter.getClass().getSimpleName());
        }
        this.filter = (PersistableAgentChannelsFilter) filter;
        children1.forEach(child -> child.setFilter(this.filter));
        children2.forEach(child -> child.setFilter(this.filter));
    }

    @Override
    public AgentChannelsFilter getFilter() {
        return filter;
    }
    
    
// -- Adding views : -----------------------------------------------------------
    
    public void addView(PersistableMonitorView view, Mode mode) {
        switch (mode) {
            case INDEPENDENT:
                children1.add(view);
                break;
            case FORWARD:
                children2.add(view);
                break;
        }
    }
    
    
// -- Listening to the aggregator : --------------------------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        children2.forEach(view -> view.connect(event));
    }

    @Override
    public void configure(AgentStatusEvent event) {
        children2.forEach(view -> view.configure(event));
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        children2.forEach(view -> view.statusChanged(event));
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        children2.forEach(view -> view.disconnect(event));
    }

    
// -- Installation : -----------------------------------------------------------

    @Override
    public void install() {
        if (!children2.isEmpty()) {
            PersistableMonitorView.super.install();
        }
        children1.forEach(view -> view.install());
    }

    @Override
    public void uninstall() {
        if (!children2.isEmpty()) {
            PersistableMonitorView.super.uninstall();
        }
        children1.forEach(view -> view.uninstall());
    }

    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor extends PersistableMonitorView.Descriptor {

        private PersistableMonitorView.Descriptor[] children;

        public PersistableMonitorView.Descriptor[] getChildren() {
            return children;
        }

        public void setChildren(PersistableMonitorView.Descriptor[] children) {
            this.children = children;
        }

        @Override
        public Descriptor clone() {
            Descriptor desc = (Descriptor) super.clone();
            if (desc.children != null) desc.children = desc.children.clone();
            return desc;
        }
        
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }
    
    /**
     * Returns a descriptor that contains information required to re-create this view in its current state.
     * @return View descriptor.
     */
    @Override
    public Descriptor save() {
        Descriptor desc = descriptor.clone();
        List<PersistableMonitorView> views = getChildren();
        PersistableMonitorView.Descriptor[] cd = new PersistableMonitorView.Descriptor[views.size()];
        for (int i=0; i<cd.length; i++) {
            cd[i] = views.get(i).save();
        }
        desc.setChildren(cd);
        return desc;
    }
    /**
     * Restores this view to the state described by the provided descriptor, to the extent possible.
     * @param descriptor View descriptor.
     */
    @Override
    public void restore(Persistable.Descriptor descriptor) {
        if (descriptor instanceof Descriptor) {
            Descriptor desc = (Descriptor) descriptor;
            PersistableMonitorView.Descriptor[] dd = desc.getChildren();
            if (dd != null) {
                List<PersistableMonitorView> vv = getChildren();
                if (dd.length == vv.size()) {
                    for (int i=0; i<dd.length; i++) {
                        vv.get(i).restore(dd[i]);
                    }
                }
            }
        }
    }
    
    private ArrayList<PersistableMonitorView> getChildren() {
        ArrayList<PersistableMonitorView> out = new ArrayList<>(children1.size() + children2.size());
        out.addAll(children1);
        out.addAll(children2);
        return out;
    }

}
