package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.io.Serializable;
import java.time.Instant;
import java.util.*;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.CommandState;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.base.filter.CoreStates;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusAggregator;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.gconsole.services.lock.Lock;
import org.lsst.ccs.gconsole.services.lock.Locker;
import org.lsst.ccs.gconsole.services.lock.LockService;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.services.persist.Savable;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Interface to be implemented by all types of command browsers.
 * A {@code Browser} handles the root page of the command browser tool.
 * All access to implementations of this class should happen on EDT.
 *
 * @author onoprien
 */
public abstract class Browser implements Savable {

// -- Fields : -----------------------------------------------------------------
    
    protected final Logger logger = Console.getConsole().getLogger();
    protected final LockService service = LockService.getService();
    
    protected JPanel browserPanel; // non-null if initialized

    final ImageIcon ICON_UNLOCKED = new ImageIcon(Lock.class.getResource("black_unlocked_27.png"), "unlocked black");
    final ImageIcon ICON_UNLOCKED_OFFLINE = new ImageIcon(Lock.class.getResource("gray_unlocked_27.png"), "unlocked gray");
    final ImageIcon ICON_LOCKED = new ImageIcon(Lock.class.getResource("black_locked_27.png"), "locked black");
    final ImageIcon ICON_LOCKED_OFFLINE = new ImageIcon(Lock.class.getResource("gray_locked_27.png"), "locked gray");
    final ImageIcon ICON_DETACHED = new ImageIcon(Lock.class.getResource("red_locked_27.png"), "locked red");
    final ImageIcon ICON_DETACHED_OFFLINE = new ImageIcon(Lock.class.getResource("light_red_locked_27.png"), "locked light red");
    final ImageIcon ICON_ATTACHED = new ImageIcon(Lock.class.getResource("green_locked_27.png"), "locked green");
    final ImageIcon ICON_ATTACHED_OFFLINE = new ImageIcon(Lock.class.getResource("light_green_locked_27.png"), "locked green");
    
    protected JLabel stateLabel;
    private AgentStatusListener statusListener; // updates agent state display
    private final AgentStatusAggregator statusAggregator = Console.getConsole().getStatusAggregator();
    
    
// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Initializes the browser if necessary, and returns its root graphics component.
     * @return Root graphics component for this browser.
     */
    public JComponent getPanel() {
        
        stateLabel = new JLabel(" ");
        statusListener = new AgentStatusListener() {
            @Override
            public void connect(AgentStatusEvent event) {
                SwingUtilities.invokeLater(() -> {
                    updateStateDisplay();
                });
            }
            @Override
            public void disconnect(AgentStatusEvent event) {
                SwingUtilities.invokeLater(() -> {
                    updateStateDisplay();
                });
            }
            @Override
            public void statusChanged(AgentStatusEvent event) {
                SwingUtilities.invokeLater(() -> {
                    updateStateDisplay();
                });
            }
        };
        String agentName = getAgentName();
        List<String> agents = agentName == null ? null : Collections.singletonList(agentName);
        statusAggregator.addListener(statusListener, agents, CoreStates.CHANNELS);
        
        browserPanel = new JPanel(new BorderLayout());
        return browserPanel;
    }
    
    /**
     * Releases resources and shuts down this browser.
     */
    public void shutdown() {
        statusAggregator.removeListener(statusListener);
        browserPanel = null;
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns the name of this command browser.
     * @return Browser name.
     */
    abstract String getName();

    /**
     * Returns the name of the currently selected agent.
     * 
     * @return Current agent name, or {@code null} if this browser does not currently associated with a specific agent.
     */
    abstract String getAgentName();
    
    
// -- Utilities : --------------------------------------------------------------
    
//    /**
//     * Returns {@code true} if the specified command is currently unavailable for use based on lock and level settings.
//     * 
//     * @param command Command to check.
//     * @param agent Lock service proxy for a remote agent.
//     * @return {@code true} If the command is locked or {@code agent} is {@code null}.
//     */
//    static public boolean isCommandLocked(DictionaryCommand command, Locker agent) {
//        if (agent == null) {
//            return true;
//        } else {
//            return ((command.getLevel() > agent.getLevel()) || 
//                    (agent.getState() != LockService.State.ATTACHED && command.getType() != CommandType.QUERY));
//        }
//    }
    
    protected void updateStateDisplay() {
        String agentName = getAgentName();
        if (agentName == null) {
            stateLabel.setText(" ");
            return;
        }
        StateBundle state = statusAggregator.getAgentState(agentName);
        if (state == null) {
            stateLabel.setText("OFFLINE");
            return;
        }
        StringBuilder sb = new StringBuilder("<html>");
        PhaseState phase = state.getState(PhaseState.class);
        if (phase != null) sb.append(phase).append(" - ");
        OperationalState ops = state.getState(OperationalState.class);
        if (ops != null) {
            switch (ops) {
                case NORMAL:
                    sb.append("<font color=\"#008800\">NORMAL</font>"); break;
                case ENGINEERING_FAULT:
                    sb.append("<font color=\"#DD0000\">ENGINEERING_FAULT</font>"); break;
                default:
                    sb.append(ops);
            }
            sb.append(" - ");
        }
        ConfigurationState con = state.getState(ConfigurationState.class);
        if (con != null) {
            switch (con) {
                case DIRTY:
                    sb.append("<font color=\"#0000DD\">MODIFIED CONFIGURATION</font>"); break;
                default:
                    sb.append(con);
            }
            sb.append(" - ");
        }
        AlertState als = state.getState(AlertState.class);
        if (als != null && !als.equals(AlertState.NOMINAL)) {
            switch (als) {
                case WARNING:
                    sb.append("<font color=\"#0000DD\">WARNING</font>"); break;
                case ALARM:
                    sb.append("<font color=\"#DD0000\">ALARM</font>"); break;
            }
            sb.append(" - ");
        }
        CommandState com = state.getState(CommandState.class);
        if (com != null) {
            switch (com) {
                case READY:
                    sb.append("<font color=\"#008800\">READY</font>"); break;
                case ACTIVE:
                    sb.append("<font color=\"#DD0000\">ACTIVE</font>"); break;
            }
        }
        stateLabel.setText(sb.toString());
    }
    
    /**
     * Computes information for rendering lock icon and enabling actions associated with the specified agent.
     * @param agent Remote agent.
     * @return Rendered lock parameters.
     */
    protected RenderedLock renderLock(Locker agent) {
        RenderedLock out = new RenderedLock();
        if (agent == null) {
            out.icon = ICON_UNLOCKED_OFFLINE;
            out.tooltip = "Subsystem offline, unlocked";
            out.enabled = true;
            out.sortKey = "9";
        } else {
            out.enabled = !agent.isAdjusting();
            switch (agent.getState()) {
                case UNLOCKED:
                    out.icon = ICON_UNLOCKED;
                    out.tooltip = "Double-click to lock";
                    out.defaultAction = LockService.Operation.LOCK.getAction(agent.getName());
                    out.availableActions = Collections.singletonList(out.defaultAction);
                    out.sortKey = "8"+ agent.getName();
                    break;
                case LOCKED:
                    AgentLock lock = agent.getLock();
                    StringBuilder sb = new StringBuilder().append("Locked by ").append(lock.getOwner());
                    if (lock instanceof AgentLockInfo lockInfo) {
                        Instant time = lockInfo.getTimeStamp().getUTCInstant();
                        sb.append(" since ").append(Const.DEFAULT_DT_FORMAT.format(time));
                    }
                    sb.append(".").toString();
                    if (agent.isOnline()) {
                        out.icon = ICON_LOCKED;
                        out.tooltip = sb.toString();
                    } else {
                        out.icon = ICON_LOCKED_OFFLINE;
                        out.tooltip = sb.append(" Offline.").toString();
                    }
                    out.defaultAction = null;
                    out.availableActions = Collections.singletonList(LockService.Operation.DESTROY.getAction(agent.getName()));
                    out.sortKey = "7"+ agent.getName();
                    break;
                case DETACHED:
                    if (agent.isOnline()) {
                        out.icon = ICON_DETACHED;
                        out.tooltip = "Locked, double-click to attach";
                    } else {
                        out.icon = ICON_DETACHED_OFFLINE;
                        out.tooltip = "Locked, Offline";
                    }
                    out.defaultAction = LockService.Operation.ATTACH.getAction(agent.getName());
                    out.availableActions = Arrays.asList(new Action[]{
                        LockService.Operation.UNLOCK.getAction(agent.getName()),
                        out.defaultAction
                    });
                    out.sortKey = "2"+ agent.getName();
                    break;
                case ATTACHED:
                    if (agent.isOnline()) {
                        out.icon = ICON_ATTACHED;
                        out.tooltip = "Locked, attached, double-click to detach";
                    } else {
                        out.icon = ICON_ATTACHED_OFFLINE;
                        out.tooltip = "Locked, Attached, Offline";
                    }
                    out.defaultAction = LockService.Operation.DETACH.getAction(agent.getName());
                    out.availableActions = Arrays.asList(new Action[]{
                        LockService.Operation.UNLOCK.getAction(agent.getName()),
                        out.defaultAction
                    });
                    out.sortKey = "1"+ agent.getName();
                    break;
            }
            if (!out.enabled) {
                out.tooltip = "Waiting for response from Lock Manager";
                out.defaultAction = null;
                out.availableActions = null;
            }
        }        
        return out;
    }
    
    
    static protected class RenderedLock implements Comparable<RenderedLock> {
        
        public ImageIcon icon;
        public String tooltip;
        public boolean enabled;
        public Action defaultAction;
        public List<Action> availableActions;
        
        private String sortKey = "";
        
        public RenderedLock(ImageIcon icon, String tooltip, boolean enabled, Action defaultAction, List<Action> availableActions) {
            this.icon = icon;
            this.tooltip = tooltip;
            this.enabled = enabled;
            this.defaultAction = defaultAction;
            this.availableActions = availableActions;
        }
        public RenderedLock() {
        }

        @Override
        public int compareTo(RenderedLock other) {
            return sortKey.compareTo(other.sortKey);
        }
    }

// -- Savin/restoring : --------------------------------------------------------

    @Override
    abstract public Descriptor save();
    
    static public class Descriptor implements Serializable {

        private String name;
        private String agent;
        private AgentPanel.Descriptor agentPanel;
        private DataPanelDescriptor page;

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public String getAgent() {
            return agent;
        }

        public void setAgent(String agent) {
            this.agent = agent;
        }

        public AgentPanel.Descriptor getAgentPanel() {
            return agentPanel;
        }

        public void setAgentPanel(AgentPanel.Descriptor agentPanel) {
            this.agentPanel = agentPanel;
        }

        public DataPanelDescriptor getPage() {
            return page;
        }

        public void setPage(DataPanelDescriptor page) {
            this.page = page;
        }

    }
    
}
