package org.lsst.ccs.gconsole.plugins.alert;

import com.l2fprod.common.swing.renderer.DefaultCellRenderer;
import java.awt.BorderLayout;
import java.awt.Component;
import static java.awt.Component.LEFT_ALIGNMENT;
import java.awt.Dialog;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableColumn;
import org.lsst.ccs.bus.data.MutedAlertRequest;
import org.lsst.ccs.gconsole.base.Const;

/**
 *
 * @author onoprien
 */
public final class MuteListDialog extends JDialog {
    
    private final TreeMap<String,TreeMap<String,MutedAlertRequest>> muteRequests;
    
    private final JComboBox<String> agentCombo = new JComboBox<>();
    private final JTable table = new JTable(new Model());
    private final JButton deleteButton = new JButton("Delete");
    
    private String selectedAgent;
    private MutedAlertRequest selectedRequest;
    private List<MutedAlertRequest> requests;
    private String[] agents;
    
    MuteListDialog(Component parent, TreeMap<String,TreeMap<String,MutedAlertRequest>> muteRequests, String agentSeed, String regExSeed) {
        
        super(parent == null ? null : SwingUtilities.getWindowAncestor(parent), "Muted alerts", Dialog.ModalityType.APPLICATION_MODAL);
        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                dispose();
            }
        });
        this.muteRequests = muteRequests;
        selectedAgent = agentSeed;
        
        Box box = Box.createHorizontalBox();
        box.setAlignmentX(LEFT_ALIGNMENT);
        box.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
        box.add(new JLabel("Subsystem:"));
        box.add(Box.createRigidArea(Const.HDIM));
        agentCombo.addActionListener(e -> {
            int prevRows = table.getModel().getRowCount();
            int i = agentCombo.getSelectedIndex();
            if (i == -1) {
                selectedAgent = null;
                requests = Collections.emptyList();
            } else {
                selectedAgent = agents[i];
                requests = new ArrayList<>(muteRequests.get(selectedAgent).values());
            }
            int currentRows = table.getModel().getRowCount();
            if (currentRows > prevRows) {
                if (prevRows > 0) ((Model) table.getModel()).fireTableRowsUpdated(0, prevRows -1);
                ((Model) table.getModel()).fireTableRowsInserted(prevRows, currentRows -1);
            } else if (currentRows < prevRows) {
                if (currentRows > 0) ((Model) table.getModel()).fireTableRowsUpdated(0, currentRows -1);
                ((Model) table.getModel()).fireTableRowsDeleted(currentRows, prevRows -1);
            } else if (currentRows > 0) {
                ((Model) table.getModel()).fireTableRowsUpdated(0, currentRows -1);
            }
        });
        box.add(agentCombo);
        add(box, BorderLayout.NORTH);
        
        for (int i = 0; i < 3; i++) {
            TableColumn column = table.getColumnModel().getColumn(i);
            int min = new JLabel(column.getHeaderValue().toString() +"      ").getPreferredSize().width;
            column.setPreferredWidth(switch (i) {case 0 -> min*3; case 1 -> min; default -> min*2;});
            column.setMinWidth(min);
            if (i == 1) {
                DefaultCellRenderer r = new DefaultCellRenderer();
                r.setHorizontalAlignment(SwingConstants.CENTER);
                column.setCellRenderer(r);
            }
        }
        table.setCellSelectionEnabled(false);
        table.setColumnSelectionAllowed(false);
        table.setRowSelectionAllowed(true);
        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        table.getSelectionModel().addListSelectionListener(e -> {
            int selectedRow = table.getSelectedRow();
            if (selectedRow == -1) {
                selectedRequest = null;
                deleteButton.setEnabled(false);
            } else {
                selectedRequest = requests.get(selectedRow);
                deleteButton.setEnabled(true);
            }
        });
        JScrollPane scrollPane = new JScrollPane(table);
        add(scrollPane, BorderLayout.CENTER);
        
        Box buttonBox = Box.createHorizontalBox();
        buttonBox.setAlignmentX(LEFT_ALIGNMENT);
        buttonBox.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
        
        buttonBox.add(Box.createRigidArea(Const.HDIM2));
        JButton button = new JButton("New");
        button.addActionListener(e -> {
            MuteDialog.show(MuteListDialog.this, agentSeed, regExSeed);
        });
        buttonBox.add(button);
        
        buttonBox.add(Box.createRigidArea(Const.HDIM2));
        deleteButton.setEnabled(false);
        deleteButton.addActionListener(e -> {
            if (selectedAgent != null && selectedRequest != null) {
                AlertViewer.sendCommandToMMM("cancelMutedAlertRequest", selectedRequest.getUniqueId());
            }
        });
        buttonBox.add(deleteButton);
        
        buttonBox.add(Box.createHorizontalGlue());
        buttonBox.add(Box.createRigidArea(Const.HDIM2));
        JButton cancelButton = new JButton("Dismiss");
        cancelButton.addActionListener(e -> {
            dispose();
        });
        buttonBox.add(cancelButton);
        buttonBox.add(Box.createRigidArea(Const.HDIM2));
        add(buttonBox, BorderLayout.SOUTH);
        
        update();
    }
    
    void update() {
        if (muteRequests.isEmpty()) {
            agents = new String[0];
            selectedAgent = null;
            requests = Collections.emptyList();
        } else {
            ArrayList<String> agentNames = new ArrayList<>(muteRequests.keySet());
            agents = agentNames.toArray(new String[0]);
            int i = agentNames.indexOf(selectedAgent);
            if (i == -1) {
                i = 0;
                selectedAgent = agents[0];
            }
            requests = new ArrayList<>(muteRequests.get(selectedAgent).values());
        }
        agentCombo.setModel(new DefaultComboBoxModel(agents));
        agentCombo.setEnabled(agents.length > 0);
        agentCombo.revalidate();
        agentCombo.repaint();
        if (selectedAgent != null) agentCombo.setSelectedItem(selectedAgent);
        revalidate();
        repaint();
    }
    
    static MuteListDialog show(Component parent, TreeMap<String,TreeMap<String,MutedAlertRequest>> muteRequests, String agentSeed, String regExSeed) {
        MuteListDialog d = new MuteListDialog(parent, muteRequests, agentSeed, regExSeed);
        d.setSize(d.getPreferredSize());
        d.pack();
        d.setLocationRelativeTo(parent);
        return d;
    }
    
    
// -- Table : ------------------------------------------------------------------
    
    private class Model extends DefaultTableModel {

        @Override
        public int getRowCount() {
            return requests == null ? 0 : requests.size();
        }

        @Override
        public int getColumnCount() {
            return 3;
        }

        @Override
        public Object getValueAt(int row, int column) {
            MutedAlertRequest r = requests.get(row);
            return switch (column) {
                case 0 -> r.getAlertIdRegEx();
                case 1 -> (r.getExpirationMillis() - System.currentTimeMillis())/60000;
                case 2 -> r.getUserName();
                default -> throw new IllegalArgumentException();
            };
        }

        @Override
        public boolean isCellEditable(int row, int column) {
            return false;
        }

        @Override
        public String getColumnName(int column) {
            return switch(column) {
                case 0 -> "Regular expression";
                case 1 -> "Minutes left";
                case 2 -> "Muted by";
                default -> throw new IllegalArgumentException();
            };
        }
        
    }
    
}
