package org.lsst.ccs.gconsole.plugins.alert;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.plugins.alert.AlertViewer.AlertNode;

/**
 * Alert acknowledgement dialog.
 *
 * @author onoprien
 */
public class AckDialog extends JDialog {
    
    private final JTextArea messageArea;
    private final Table tableOfAlerts;
    private final List<RaisedAlertHistory> alerts;
    private final JButton ackButton;
    
    String message;
    String[] alertsToAck, alertsToNack;
    
    private AckDialog(AlertNode node, Component parentComponent) {
        super(parentComponent == null ? null : SwingUtilities.getWindowAncestor(parentComponent), "Acknowledge alerts", Dialog.ModalityType.APPLICATION_MODAL);
        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
        addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                message = null;
                dispose();
            }
        });
        
        TreeMap<String,RaisedAlertHistory> aa = new TreeMap<>();
        node.getLeaves().stream().map(n -> n.getHistory()).forEach(h -> aa.put(h.getHighestAlert().getAlertId(), h));
        alerts = new ArrayList<>(aa.values());
        
        Box infoBox = Box.createVerticalBox();
        infoBox.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));

        JLabel label = new JLabel("<html>Acknowledging an alert informs operators that the issue is being taken care of.");
        label.setAlignmentX(LEFT_ALIGNMENT);
        infoBox.add(label);
        infoBox.add(Box.createRigidArea(Const.VDIM));

        label = new JLabel("<html>Subsystem: <b>"+ node.getAgentName());
        label.setAlignmentX(LEFT_ALIGNMENT);
        infoBox.add(label);
        tableOfAlerts = new Table();
        JScrollPane sp = new JScrollPane(tableOfAlerts);
        sp.setAlignmentX(LEFT_ALIGNMENT);
        infoBox.add(sp);
        infoBox.add(Box.createRigidArea(Const.VDIM));
        
        add(infoBox, BorderLayout.NORTH);
        
        Box messageBox = Box.createVerticalBox();
        messageBox.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
        label = new JLabel("<html><b>Message:");
        label.setAlignmentX(LEFT_ALIGNMENT);
        messageBox.add(label);
        messageArea = new JTextArea(4,80);
        messageArea.setAlignmentX(LEFT_ALIGNMENT);
        messageBox.add(messageArea);
        add(messageBox, BorderLayout.CENTER);
        
        Box buttonBox = Box.createHorizontalBox();
        buttonBox.setAlignmentX(LEFT_ALIGNMENT);
        buttonBox.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
        buttonBox.add(Box.createRigidArea(Const.HDIM2));
        buttonBox.add(Box.createHorizontalGlue());
        ackButton = new JButton("Commit");
        ackButton.addActionListener(e -> {
            message = messageArea.getText();
            if (message.length() > 300) {
                message = message.substring(0, 300) +" ...";
            }
            ((TableModel)tableOfAlerts.getModel()).fillIDs();
            dispose();
        });
        buttonBox.add(ackButton);
        buttonBox.add(Box.createRigidArea(Const.HDIM2));
        JButton cancelButton = new JButton("Cancel");
        cancelButton.addActionListener(e -> {
            message = null;
            dispose();
        });
        buttonBox.add(cancelButton);
        buttonBox.add(Box.createRigidArea(Const.HDIM2));
        add(buttonBox, BorderLayout.SOUTH);
    }
    
    
// -- Table of alerts : --------------------------------------------------------
    
    private class Table extends JTable {
        Table() {
            setDefaultRenderer(Object.class, new DefaultTableCellRenderer() {
                @Override
                public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
                    super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
                    if (value instanceof RaisedAlertHistory h) {
                        AlertState high = h.getHighestAlertState();
                        switch (column) {
                            case 1 -> {
                                setText(h.getHighestAlert().getAlertId());
                                setForeground(null);
                                setToolTipText(h.getHighestAlert().getDescription());
                            }
                            case 2 -> {
                                setText(high.toString());
                                setForeground(LsstAlertPlugin.COLOR.get(high));
                                setToolTipText(h.getLatestAlertCause());
                            }
                        }
                    }
                    return this;
                }
            });
            setModel(new TableModel());
            this.setRowSelectionAllowed(false);
            this.setColumnSelectionAllowed(false);
            this.setCellSelectionEnabled(false);
            for (int i = 0; i < 3; i++) {
                TableColumn column = getColumnModel().getColumn(i);
                column.setPreferredWidth(switch (i) {
                    case 0 -> (new JLabel(" Acknowledged ")).getPreferredSize().width;
                    default -> 400;
                });
            }
        }
    }
    
    private class TableModel extends AbstractTableModel {
        
        private boolean[] ack;
        
        TableModel() {
            ack = new boolean[alerts.size()];
            for (int i=0; i<ack.length; i++) {
                ack[i] = alerts.get(i).isAcknowledged();
            }
        }
        
        void fillIDs() {
            ArrayList<String> toAck = new ArrayList<>(alerts.size());
            ArrayList<String> toNack = new ArrayList<>(alerts.size());
            boolean reAck = false; // reAckBox.isSelected();
            for (int i=0; i<ack.length; i++) {
                if (ack[i]) {
                    if (reAck || !alerts.get(i).isAcknowledged()) {
                        toAck.add(alerts.get(i).getHighestAlert().getAlertId());
                    }
                } else {
                    if (alerts.get(i).isAcknowledged()) {
                        toNack.add(alerts.get(i).getHighestAlert().getAlertId());
                    }
                }
            }
            alertsToAck = toAck.toArray(new String[0]);
            alertsToNack = toNack.toArray(new String[0]);
        }

        @Override
        public int getRowCount() {
            return alerts.size();
        }

        @Override
        public int getColumnCount() {
            return 3;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            return switch (columnIndex) {
                case 0 -> ack[rowIndex];
                default -> alerts.get(rowIndex);
            };
        }

        @Override
        public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
            if (columnIndex == 0) {
                ack[rowIndex] = (boolean) aValue;
            }
        }

        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return columnIndex == 0;
        }

        @Override
        public String getColumnName(int column) {
            return switch (column) {
                case 0 -> "Acknowledged";
                case 1 -> "Alert ID";
                case 2 -> "Max severity";
                default -> throw new IllegalArgumentException();
            };
        }

        @Override
        public Class<?> getColumnClass(int columnIndex) {
            return switch (columnIndex) {
                case 0 -> Boolean.class;
                default -> String.class;
            };
        }
        
    }
    
    
// -- Display the dialog and return the message : ------------------------------
    
    /**
     * Displays alert acknowledgement confirmation dialog.
     * 
     * @param alerts Map of subsystem names to alert IDs.
     * @return Message entered by the user.
     */
    static AckDialog show(AlertViewer.AlertNode node, Component parentComponent) {
        if (parentComponent != null) {
            parentComponent = SwingUtilities.getWindowAncestor(parentComponent);
        }
        AckDialog d = new AckDialog(node, parentComponent);
        d.setSize(d.getPreferredSize());
        d.pack();
        d.setLocationRelativeTo(parentComponent);
        d.setVisible(true);
        return d.message == null ? null : d;
    }
    
}
