package org.lsst.ccs.gconsole.base.filter;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;

/**
 * Regular expression based channels filter.
 * The filter selects channels that match a regular expression and maps them to display paths based on a template.
 *
 * @author onoprien
 */
public class RegExRenamingFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CREATOR_PATH = "Built-In/Basic/Regular Expression, Renaming";
    
    private final String name;
    private final String regex;
    private final String template;
    private final List<String> agents;
    
    private final Pattern pattern;

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = AgentChannelsFilter.CATEGORY,
            name = "Regular expression filter",
            path = CREATOR_PATH,
            description = "Channels filter that uses a regular expression to select channels and leaves their paths unchanged.")
    public RegExRenamingFilter(
            @Par(def = "RegEx", desc = "Filter name.") String name,
            @Par(desc = "Regular expression to be applied to the channel path.") String regex,
            @Par(def = "true", desc = "Match the whole path. If checked, a channel is accepted only if the provided regular expression matches its entire path. Otherwise, a channel is accepted if the regular expression matches any part of the path.") boolean wholePath,
            @Par(def = Par.NULL, desc = "Display path template. Each occurrence of ${name} or $g will be replaced by the result of evaluating the corresponding group(name) or group(g) from the regular expression match. If null, the original path will be used as the display path.") String displayPath,
            @Par(def = Par.NULL, desc = "List of accepted subsystem names. If null, channels from all subsystems can be accepted.") List<String> subsystems)
    {
        this.name = null;
        if (regex == null || regex.isEmpty()) throw new IllegalArgumentException("The regular expression cannot be empty");
        if (wholePath) {
            regex = "^"+ regex +"$";
        } else {
            regex = "^.*?"+ regex +".*?$";
        }
        pattern = Pattern.compile(regex);
        this.regex = regex;
        this.template = displayPath;
        this.agents = (subsystems == null || subsystems.isEmpty()) ? null : Collections.unmodifiableList(new ArrayList<>(subsystems));
    }
    
    
// -- Implementing filter : ----------------------------------------------------

    @Override
    public String getName() {
        return name;
    }

    @Override
    public List<String> getAgents() {
        return agents;
    }

    @Override
    public List<String> getOriginChannels() {
        return Collections.singletonList(regex);
    }

    @Override
    public String getDisplayPath(String originPath) {
        Matcher matcher = pattern.matcher(originPath);
        if (matcher.matches()) {
            if (template == null) {
                return originPath;
            } else {
                StringBuffer sb = new StringBuffer();
                matcher.appendReplacement(sb, template);
                return sb.toString();
            }
        } else {
            return null;
        }
    }

    @Override
    public String getOriginPath(String displayPath) {
        return template == null ? displayPath : null;
    }
    
    
// -- Testing : ----------------------------------------------------------------
    
    static public void main(String... args) {
        RegExRenamingFilter filter = new RegExRenamingFilter(null, "(r)", true, "te$1st$1", null);
        System.out.println(filter.getDisplayPath("rr"));
        filter = new RegExRenamingFilter(null, "x(r).*(p)", false, "do/$1/once/$2", null);
        System.out.println(filter.getDisplayPath("xrrefewp"));
    }
    
}
