package org.lsst.ccs.gconsole.base.filter;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import static org.lsst.ccs.gconsole.services.aggregator.AgentChannel.MARK_STATE;

/**
 * Filter that selects channels that reflect CCS state and conditions, and are present in all subsystems.
 * This filter is used by "Whole Camera/CCS Health" monitor page.
 *
 * @author onoprien
 */
public class CCSHealthFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CREATOR_PATH = "Built-In/Whole Camera/CCS Health";
    
    static private final String GROUP_STATE = "Core States/";
    static private final String GROUP_VERSION = "Versions/";
    static private final String GROUP_RUNTIME = "Runtime Info/";
    
    static private final List<String> CHANNELS = Collections.unmodifiableList(Arrays.asList(
            MARK_STATE +"PhaseState", MARK_STATE +"OperationalState", MARK_STATE +"ConfigurationState", MARK_STATE +"AlertState", MARK_STATE +"CommandState", MARK_STATE +"locked by",
            "/static/version.agent", "/static/version.distribution", "/static/version.toolkit",
            "/runtimeInfo/[^/]+"));
    
    static private final Pattern PAT_VERSION = Pattern.compile("static/version\\.([^/]+)");
    static private final Pattern PAT_RUNTIME = Pattern.compile("runtimeInfo/+([^/]+)");
    static private final Pattern PAT_REVERSE = Pattern.compile("([^/]+/)([^/]+)/([^/]+)");
    private final int cutFront = AgentChannel.MARK_STATE.length() - 2;
    private final int cutBack = "State".length();
    
    static private boolean watch = true;  // FIXME: Temporary diagnostics for LSSTCCS-2800

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = "AgentChannelsFilter",
            name = "CCS",
            path = CREATOR_PATH,
            description = "CCS Health Monitor. Selects channels that reflect CCS state and conditions, and are present in all subsystems.")
    public CCSHealthFilter() {
    }
    
// -- Implementing filter : ----------------------------------------------------
    
    @Override
    public String getDisplayPath(AgentChannel channel) {
        String path = channel.getLocalPath();
        Matcher m = PAT_VERSION.matcher(path);
        String agent = channel.getAgentName();
        if (m.matches()) {
            return GROUP_VERSION + agent +"/"+ m.group(1);
        } else {
            m = PAT_RUNTIME.matcher(path);
            if (m.matches()) {
                return GROUP_RUNTIME + agent + "/" + m.group(1);
            } else if (path.endsWith("locked by")) {
                return GROUP_STATE + agent + path.substring(cutFront);
            } else {
                if (path.length() < 10) { // FIXME: Temporary fix and diagnostics for LSSTCCS-2800
                    if (watch) {
                        Console.getConsole().getLogger().warn("FIXME: CCSHealthFilter "+ path);
                        watch = false;
                    }
                    return null;
                }
                return GROUP_STATE + agent + path.substring(cutFront, path.length() - cutBack);
            }
        }
    }

    @Override
    public String getOriginPath(String displayPath) {
        Matcher m = PAT_REVERSE.matcher(displayPath);
        if (m.matches()) {
            switch (m.group(1)) {
                case GROUP_STATE:
                    StringBuilder sb = new StringBuilder(m.group(2)).append(AgentChannel.MARK_STATE).append(m.group(3));
                    if (!displayPath.endsWith("locked by")) {
                        sb.append("State");
                    }
                    return sb.toString();
                case GROUP_VERSION:
                    return m.group(2) +"/static/version."+ m.group(3);
                case GROUP_RUNTIME:
                    return m.group(2) +"/runtimeInfo/"+ m.group(3);
                default:
                    return null;
            }
        } else {
            return null;
        }
    }

    @Override
    public List<String> getOriginChannels() {
        return CHANNELS;
    }

    @Override
    public String getName() {
        return "CCS";
    }

    @Override
    public List<String> getFields(boolean compact) {
       return Collections.singletonList("VALUE");
    }

}

