package org.lsst.ccs.gconsole.base;

import java.util.Map;

/**
 * Base class for implementing Graphical Console plugins. Provides life cycle hooks 
 * and services that can be used by its subclasses. 
 * <p>
 * To be recognized as a plugin, a subclass should be annotated with {@link org.lsst.ccs.gconsole.annotations.Plugin @Plugin}.
 * The annotation can also be used to provide information about the plugin and some
 * configuration parameters.
 * A subclass should define a default constructor that will be called by the framework
 * to create a singleton instance. The life cycle hooks are {@link #initialize()}, 
 * {@link #start()}, {@link #stop()}, and {@link #shutdown()}. When the graphical 
 * console starts, all plugins annotated with {@code loadAtStart=true} (default)
 * are loaded, their {@link #initialize()} methods are called, then their {@link #start()}
 * methods are called. When the console shuts down, {@link #stop()} and {@link #shutdown()}
 * methods are called. In addition, the graphical console might provide a way for
 * the user to call {@link #start()} and {@link #stop()} methods interactively.
 * <p>
 * The services are implemented 
 * by forwarding methods calls to the {@link ConsolePluginServices} instance set 
 * for this plugin by the framework.
 *
 * @author onoprien
 */
abstract public class ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    private ConsolePluginServices services;


// -- Life cycle : -------------------------------------------------------------
    
    /**
     * This method is called by the framework to initialize this plugin.
     * 
     * @param provider {@code ConsolePluginServices} to be used by this plugin.
     * @throws UnsupportedOperationException if called by client code.
     */
    public final void setServicesProvider(ConsolePluginServices provider) {
        if (this.services != null) {
            throw new UnsupportedOperationException("This method should not be called by clients.");
        }
        this.services = provider;
    }

    /**
     * Called by the framework when the console starts, to initialize this plugin..
     */
    public void initialize() {}

    /**
     * Called by the framework when the console starts, after all plugins have been initialized.
     * May also be called interactively through the graphical console GUI.
     */
    public void start() {}

    /**
     * Called by the framework when the console shuts down, before the call to {@link #shutdown()}.
     * May also be called interactively through the graphical console GUI. In that
     * case, the method will not be called on the console shutdown unless the {@link #start()}
     * method is invoked again.
     */
    public void stop() {}

    /**
     * Called by the framework when the graphical console is shut down.
     */
    public void shutdown() {}
    
    
// -- Other SPI hooks : --------------------------------------------------------
    
    /**
     * Called by the framework when plugin properties are modified.
     * 
     * @param source Source of notification.
     * @param changes For the changed properties, map of keys to their new values.
     */
    public void propertiesChanged(Object source, Map<String,Object> changes) {}
    
    /**
     * Called by the framework when this plugin configuration needs to be saved.
     * 
     * @return JavaBean that holds information on the current configuration of this plugin.
     */
    public ComponentDescriptor save() {
        return null;
    }
    
    /**
     * Called by the framework to restore this plugin configuration.
     * 
     * @param storageBean JavaBean that holds configuration information.
     * @param lastRound {@code True} if this is the last call to this method, regardless of the return value.
     * @return {@code False} if this method should be called again in the next round.
     */
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        return true;
    }


// -- Access to services and utilities : ---------------------------------------
    
    /**
     * Provides access to the graphical console subsystem.
     * @return Graphical console subsystem instance.
     */
    public Console getConsole() {
        return services.getConsole();
    }
    
    /**
     * Provides access to plugin services.
     * @return An instance of {@code ConsolePluginServices} associated with this plugin.
     */
    public ConsolePluginServices getServices() {
        return services;
    }
    
}
