package org.lsst.ccs.gconsole.util.swing;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;

/**
 * Widget for displaying and selecting date and time.
 *
 * @author onoprien
 */
public class DateTimeSelector extends JPanel {

// -- Fields : -----------------------------------------------------------------
    
    private LocalDateTime dateTime;
    
    private LocalDateTime min, max;
    private DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd  HH:mm:ss");
    
    private final JTextField field;
    private final JButton button;
    
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();

// -- Life cycle : -------------------------------------------------------------
    
    public DateTimeSelector() {
        setLayout(new BorderLayout());
        field = new JTextField();
        add(field, BorderLayout.CENTER);
        field.setEditable(false);
        FontMetrics fm = field.getFontMetrics(field.getFont());
        float widthInColumns = fm.stringWidth(" 0000-00-00  00:00:00 ")/(float)fm.charWidth('m');
        field.setColumns(Math.round(widthInColumns) + 2);
        button = new JButton("...");
        add(button, BorderLayout.EAST);
        button.addActionListener(e -> {
            LocalDateTime dt = DateTimePicker.selectDateTime(dateTime, null, button);
            if (dt != null) {
                set(dt);
                fire();
            }
        });
        set(null);
    }

    @Override
    public Dimension getMinimumSize() {
        return getPreferredSize();
    }

    @Override
    public Dimension getMaximumSize() {
        return getPreferredSize();
    }
    
    
// -- Set and get values : -----------------------------------------------------
    
    /**
     * Sets current value.
     * If the specified value is {@code null} or is outside of the valid range,
     * this selector will go to "no selection" state (no date-time is displayed).
     * 
     * @param value Date and time.
     */
    public void set(LocalDateTime value) {
        if ((value == null) || (min != null && min.isAfter(value)) || (max != null && max.isBefore(value))) {
            dateTime = null;
            field.setText("");
        } else {
            dateTime = value;
            field.setText(" "+ formatter.format(dateTime) +" ");
        }
    }
   
    /**
     * Returns the currently selected date and time.
     * 
     * @return Selected date-time, or {@code null} if in "no selection" state.
     */
    public LocalDateTime get() {
        return dateTime;
    }
    
    
// -- Other setters and getters : ----------------------------------------------
    
    /**
     * Sets validity range for this selector.
     * An attempt to select a date and time outside the valid range will put this selector into "no selection" state.
     * 
     * @param start Start of the valid range.
     * @param end End of the valid range.
     */
    public void setValidRange(LocalDateTime start, LocalDateTime end) {
        min = start;
        max = end;
        if ((dateTime != null) && ((min != null && min.isAfter(dateTime)) || (max != null && max.isBefore(dateTime)))) {
            dateTime = null;
            field.setText("");
        }
    }
    
    /**
     * Sets format for displaying date and time.
     * 
     * @param pattern Format pattern (see {@link DateTimeFormatter}).
     */
    public void setFormat(String pattern) {
        formatter = DateTimeFormatter.ofPattern(pattern);
        if (dateTime !=  null) {
            field.setText(formatter.format(dateTime));
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        field.setEnabled(enabled);
        button.setEnabled(enabled);
        super.setEnabled(enabled);
    }

    @Override
    public void setToolTipText(String text) {
        super.setToolTipText(text);
        field.setToolTipText(text);
        button.setToolTipText(text);
    }
    
//    If this method is uncommented, robust parser should be provided.
//    Requering the user to use strict format is counterproductive.
//    A spinner with date-time formatter can be used instead of text field.
//    
//    /**
//     * Determines whether the text field displaying date and time is directly editable,
//     * without using the {@link DateTimePicker} dialog.
//     * 
//     * @param editable If {@code true} the text field displaying date and time is directly editable
//     */
//    public void setEditable(boolean editable) {
//        field.setEditable(editable);
//    }

    
// -- Handling listeners : -----------------------------------------------------
    
    public interface Listener {
        void changed(DateTimeSelector source);
    }
    
    public void addListener(Listener listener) {
        listeners.addIfAbsent(listener);
    }
    
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }
    
    protected void fire() {
        listeners.forEach(listener -> listener.changed(this));
    }
    
    
// -- Testing : ----------------------------------------------------------------
    
    static public void main(String... args) {
        SwingUtilities.invokeLater(() -> {
            JFrame frame = new JFrame("JDateTimeChooser Test");
            frame.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
            DateTimeSelector chooser = new DateTimeSelector();
//            chooser.set(LocalDateTime.now());
            chooser.setValidRange(LocalDateTime.now().minusMonths(1), LocalDateTime.now().plusMonths(2));
            frame.add(chooser);
            frame.pack();
            frame.setVisible(true);
        });
    }

}
