package org.lsst.ccs.gconsole.plugins.trending.timeselection;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.ListCellRenderer;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.PlotPage;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.plugins.trending.LsstTrendingPlugin;
import org.lsst.ccs.gconsole.plugins.trending.TrendPlotter;

/**
 * GUI for selecting time window for trending plots.
 * Provides a way to select time window from a list of presets and to open custom time window
 * creation and preset management dialogs, as well as access the currently selected time window.
 *
 * @author onoprien
 */
final public class TimeWindowSelector extends JComboBox {
    
// -- Private parts : ----------------------------------------------------------
       
    private final String CUSTOM_BUTTON = "Choose...";
    private final String PRESETS_BUTTON = "From Saved...";
    private final String PLOT_BUTTON = "From Plot";
    private final String DEFAULT_NAME = "custom";
    private final int MAX_PRESETS = 10; // Maximum number of presets to include in drop-down list
    
    private final PresetList timeWindowList;
    private volatile TimeWindow selection;
    
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();
    

// -- Construction and initialization : ----------------------------------------
    
    public TimeWindowSelector(Studio application) {
        
        timeWindowList = new PresetList(application);
        
        setRenderer(new Renderer(getRenderer()));
        setMaximumRowCount(MAX_PRESETS + 7);
        
        Model model = new Model(timeWindowList);
        setModel(model);
        setSelectedIndex(1);
        selection = model.getSelectedItem();
        
        addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                TimeWindow ts = myModel().getSelectedItem();
                
                switch (ts.getName()) {
                    
                    case CUSTOM_BUTTON: // custom time window has been entered
                        
                        ts = TimeWindowDialog.defineTimeWindow(TimeWindowSelector.this, selection);
                        setSelectedTimeWindow(ts);
                        break;
                        
                    case PRESETS_BUTTON: // one of the presets has been selected through presets dialog
                        
                        ts = PresetsDialog.managePresets(TimeWindowSelector.this, timeWindowList);
                        if (ts == null) ts = selection;
                        ts.touch();
                        Model model = new Model(timeWindowList);
                        int index = model.getIndexOf(ts);
                        if (index == -1) index = 1;
                        setSelection(model.getElementAt(index));
                        model.setSelectedItem(selection);
                        setModel(model);
                        break;
                        
                    case PLOT_BUTTON: // fetch type from currently selected plot
                        
                        try {
                            PlotPage page = LsstTrendingPlugin.getSelectedPage();
                            TrendPlotter plot = (TrendPlotter) page.currentRegion().currentPlot();
                            ts = plot.getTimeWindow();
                            setSelectedTimeWindow(ts);
                        } catch (Exception x) {
                            Console.getConsole().error("There is no currently selected plot.");
                            setSelectedItem(selection);
                        }
                        break;
                        
                    default: // one of the presets has been selected from drop-down list
                        
                        if (!DEFAULT_NAME.equals(ts.getName())) {
                            ts.touch();
                            if (DEFAULT_NAME.equals(selection.getName())) {
                                myModel().removeElementAt(4);
                            }
                        }
                        setSelection(ts);
                        
                }
            }
        });
        
    }
    
// -- Getters and setters : ----------------------------------------------------
    
    /** Returns currently selected time window. */
    public TimeWindow getSelectedTimeWindow() {
        return selection;
    }
   
    /**
     * Sets selected time window.
     * If the specified window has a name, the window should already be present in the list of presets.
     * 
     * @param timeWindow The time window to be selected.
     */
    public void setSelectedTimeWindow(TimeWindow timeWindow) {
        if (timeWindow == null) {
            setSelectedItem(selection == null ? TimeWindow.LAST_HOUR : selection);
        } else {
            timeWindow = timeWindowList.insert(timeWindow);
            if (timeWindow.getName().isEmpty()) {
                timeWindow.setName(DEFAULT_NAME);
                if (DEFAULT_NAME.equals(myModel().getElementAt(4).getName())) {
                    myModel().removeElementAt(4);
                }
                myModel().insertElementAt(timeWindow, 4);
                setSelectedIndex(4);
            } else {
                Model model = new Model(timeWindowList);
                model.setSelectedItem(timeWindow);
                setModel(model);
            }
            setSelection(timeWindow);
        }
    }
    
    
// -- Handling listeners : -----------------------------------------------------
    
    public interface Listener {
        void timeWindowChanged(Event event);
    }
    
    static public class Event extends EventObject {

        private final TimeWindow timeWindow;

        public Event(Object source, TimeWindow timeWindow) {
            super(source);
            this.timeWindow =  timeWindow;
        }

        public TimeWindow getTimeWindow() {
            return timeWindow;
        }
        
    }
    
    public void addListener(Listener listener) {
        listeners.add(listener);
    }
    
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }
    
    public void removeAllListeners() {
        listeners.clear();
    }
    
    private void setSelection(TimeWindow timeWindow) {
        if (!Objects.equals(selection, timeWindow)) {
            selection = timeWindow;
            Event event = new Event(this, selection);
            listeners.forEach(listener -> listener.timeWindowChanged(event));
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private Model myModel() {
        return (Model) super.getModel();
    }


// -- Custom renderer : --------------------------------------------------------
    
    private class Renderer implements ListCellRenderer {

        private final ListCellRenderer horse;
        private final JPanel separatorPanel = new JPanel(new BorderLayout());
        private final JSeparator separator = new JSeparator();

        Renderer(ListCellRenderer renderer) {
            horse = renderer;
        }

        @Override
        public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            Component comp = horse.getListCellRendererComponent(list, ((TimeWindow)value).getName(), index, isSelected, cellHasFocus);
            if (index == 0 || index == 3 || index == TimeWindowSelector.this.getItemCount()-3) {
                separatorPanel.removeAll();
                separatorPanel.add(comp, BorderLayout.CENTER);
                separatorPanel.add(separator, BorderLayout.SOUTH);
                comp = separatorPanel;
//            } else if (index > 5 && index == TimeWindowSelector.this.getItemCount()-1) {
//                separatorPanel.removeAll();
//                separatorPanel.add(comp, BorderLayout.CENTER);
//                separatorPanel.add(separator, BorderLayout.NORTH);
//                comp = separatorPanel;
            }
            return comp;
        }

    }
    
// -- Custom model : -----------------------------------------------------------
    
    private class Model extends DefaultComboBoxModel<TimeWindow> {
        
        Model(PresetList tsList) {

            // Add element for launching time range construction dialog :
            
            addElement(new TimeWindow(CUSTOM_BUTTON));

            // Add standard ranges :
            
            TimeWindow selectedItem = TimeWindow.LAST_HOUR;
            addElement(TimeWindow.LAST_HOUR);
            addElement(TimeWindow.LAST_6_HOURS);
            addElement(TimeWindow.LAST_24_HOURS);

            // Add custom ranges :
            
            List<TimeWindow> customRanges = tsList.getRecent(MAX_PRESETS);

            for (TimeWindow ts : customRanges) {
                addElement(ts);
            }
            
            // Add element for copying settings from currently selected plot: 
            
            addElement(new TimeWindow(PLOT_BUTTON, "now-3600", "now", false));

            // Add element for launching time range presets dialog :
            
            addElement(new TimeWindow(PRESETS_BUTTON));

            // Set default selection :
            
            setSelectedItem(selectedItem);
        }
    
        @Override
        public TimeWindow getSelectedItem() {
            return (TimeWindow) super.getSelectedItem();
        }
        
    }
    
}
