package org.lsst.ccs.gconsole.plugins.trending.dataselection;

import java.awt.Component;
import java.awt.Window;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;

/**
 * Dialog for managing time window presets.
 * 
 * @author onoprien
 */
public class PresetsDialog {

// -- Private parts : ----------------------------------------------------------
    
    private final JScrollPane root;
    private final JButton deleteButton, newButton, okButton;
    private final JTable table;
    
    private final PresetList presetList;
    private DataType selectedPreset;


// -- Construction and initialization : ----------------------------------------
    
    private PresetsDialog(PresetList data) {
        
        presetList = data;
        
        deleteButton = new JButton("Delete");
        deleteButton.addActionListener(e -> {
            DataType ts = getSelectedPreset();
            if (ts != null) presetList.delete(ts);
        });
        
        newButton = new JButton("New...");
        newButton.addActionListener(e -> {
            DataType ts = DataTypeDialog.defineDataType(newButton, null);
            if (ts != null) {
                ts = presetList.insert(ts);
                setSelectedPreset(ts);
            }
        });
        
        okButton = new JButton(" OK ");
        okButton.addActionListener(e -> {
            selectedPreset = getSelectedPreset();
            Window w = SwingUtilities.getWindowAncestor(okButton);
            if (w != null) w.setVisible(false);
        });
        
        table = new JTable(presetList);
        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        table.getSelectionModel().addListSelectionListener(e -> {
            DataType ts = getSelectedPreset();
            deleteButton.setEnabled(ts != null);
        });
        if (presetList.getRowCount() > 0) {
            table.getSelectionModel().setSelectionInterval(0, 0);
        }
        table.setMinimumSize(table.getPreferredSize());
        
        deleteButton.setEnabled(getSelectedPreset() != null);
        
        root = new JScrollPane(table);
    }
    
// -- Showing the dialog : -----------------------------------------------------
    
    /**
     * Displays the dialog for managing time window presets.
     * 
     * @param parent Parent component.
     * @param data List of presets known to the application.
     * @return Preset selected by the user, or {@code null} if no preset was selected or if the dialog was canceled.
     */
    static public DataType managePresets(Component parent, PresetList data) {
        PresetsDialog dialog = new PresetsDialog(data);
        JOptionPane.showOptionDialog(parent, dialog.root, "Trending Data Type Presets", 
                JOptionPane.DEFAULT_OPTION, JOptionPane.PLAIN_MESSAGE, null, 
                new JButton[] {dialog.deleteButton, dialog.newButton, dialog.okButton}, dialog.okButton);
        return dialog.selectedPreset;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private DataType getSelectedPreset() {
        int index = table.getSelectedRow();
        return index == -1 ? null : presetList.get(index);
    }
    
    private void setSelectedPreset(DataType ts) {
        int index = presetList.indexOf(ts);
        if (index == -1) {
            table.getSelectionModel().clearSelection();
        } else {
            table.getSelectionModel().setSelectionInterval(index, index);
        }
    }
}
