package org.lsst.ccs.gconsole.plugins.trending.dataselection;

import java.util.Comparator;
import java.util.Objects;

/**
 * Data type used by trending plots.
 *
 * @author onoprien
 */
final public class DataType {
    
// -- Fields : ----------------------------------------------------------
    
    static private final String DELIMITER = "$";
    
    private String name;
    private Boolean raw;
    private int bins;
    
    private boolean persist; // true if this type should be persisted between sessions
    private long lastUsed; // time when this type was last selected, millis
    
// -- Predefined types : -------------------------------------------------------
    
    static public final DataType RAW = new DataType(null, true, 0, false);
    static public final DataType DEFAULT = new DataType("Statistical", false, 0, false);
    
    
// -- Construction : -----------------------------------------------------------
    
    /**
     * Constructs an instance.
     * 
     * @param name Time window name.
     * @param raw True in raw data, false if statistical, {@code null} if unspecified.
     * @param bins Desired number of beans, 0 if unspecified.
     * @param persistent True is this time window will be persisted between console sessions.
     * @throws IllegalArgumentException If the combination of arguments is illegal.
     */
    public DataType(String name, Boolean raw, int bins, boolean persistent) {
        this.name = name;
        if (raw == null || bins < 0 || bins > 1000) {
            throw new IllegalArgumentException("Invalid trending data type parameters: "+ raw +", "+ bins);
        }
        this.raw = raw;
        this.bins = bins;
        persist = persistent;
        lastUsed = System.currentTimeMillis();
    }
    
    
// -- Setters and getters : ----------------------------------------------------
    
    public Boolean getRaw() {
        return raw;
    }

    public void setRaw(Boolean raw) {
        this.raw = raw;
    }

    public int getBins() {
        return bins;
    }

    public void setBins(int bins) {    
        this.bins = bins;
    }

    public String getName() {
        return toString();
    }
    
    public void setName(String name) {
        this.name = name;
    }
    
    /** Returns {@code true} if this data type should be persisted between sessions. */
    public boolean isPersistent() {
        return persist;
    }
    
    /** Sets the flag that indicate whether or not this data type should be persisted between sessions. */
    public void setPersistent(boolean persistent) {
        persist = persistent;
    }
    
    /** Returns the time in milliseconds when this time window was last selected. */
    public long getLastUseTime() {
        return lastUsed;
    }
    
    /** Sets the time when this data type was last selected to the current system time. */
    public DataType touch() {
        lastUsed = System.currentTimeMillis();
        return this;
    }

    /** Returns user-readable string representation of this data type. */
    @Override
    public String toString() {
        return name == null ? autoName(raw, bins) : name;
    }

    
// -- Converting to/from compressed string : -----------------------------------
    
    /** Returns string representation of this data type used for saving it in application properties. */
    public String toCompressedString() { // FIXME: review when LSSTCCS-2864 is resolved.
        StringBuilder sb = new StringBuilder();
        if (name != null) {
            sb.append(name).append(DELIMITER);
        }
        if (!raw) {
            sb.append(bins);
        }
        return sb.toString();
    }
    
    /** Parses string representation of this time window used for saving it in application properties. */
    static public DataType parseCompressedString(String s) {
        String name = null;
        Boolean raw = true;
        int bins = 0;
        int i = s.indexOf(DELIMITER);
        if (i != -1) {
            name = s.substring(0, i);
            s = s.substring(i+1);
        }
        if (!s.isEmpty()) {
            raw = false;
            bins = Integer.parseInt(s);
        }
        return new DataType(name, raw, bins, true);
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    static private String autoName(boolean raw, int bins) {
        if (raw) {
            return "Raw";
        } else if (bins > 0) {
            return "Stat "+ bins;
        } else {
            return "Stat";
        }
    }


// -- Comparisons : ------------------------------------------------------------

    @Override
    public boolean equals(Object o) {
        if (!(o instanceof DataType)) return false;
        DataType other = (DataType) o;
        return getName().equals(other.getName()) && Objects.equals(raw, other.raw) && bins == other.bins && persist == other.persist;
    }

    @Override
    public int hashCode() {
    int hash = 3;
        hash = 7 * hash + getName().hashCode();
        hash = 7 * hash + (raw == null ? 47 : raw.hashCode());
        hash = 7 * hash + bins;
        hash = 7 * hash + (persist ? 0 : 1);
        return hash;
    }
    
     public boolean isReplaceableBy(DataType other) {
         return getName().equals(other.getName()) && Objects.equals(raw, other.raw) && bins == other.bins && (!persist || other.persist);
     }
    
    /** Returns comparator that compares time windows by names. */
    static public Comparator<DataType> compareByName() {
        return new Comparator<DataType>() {
            @Override
            public int compare(DataType o1, DataType o2) {
                return o1.toString().compareTo(o2.toString());
            }
        };
    }
    
    /** Returns comparator that compares time windows by last use times, most recent first. */
    static public Comparator compareByTime() {
        return new Comparator() {
            @Override
            public int compare(Object o1, Object o2) {
                return (int) Math.signum(((DataType)o2).getLastUseTime() - ((DataType)o1).getLastUseTime());
            }
        };
    }
    
}
