package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.Color;
import java.util.EnumSet;
import org.lsst.ccs.bus.messages.BusMessage;

/**
 * A bus message that has been processed by tracer filters.
 * Provides access to the original {@link BusMessage} object and to the textual
 * representation, additional flags and properties added by filters.
 *
 * @author onoprien
 */
public class FilteredMessage {
    
    /**
     * Flags that can trigger various actions by the tracer and/or the graphical console.
     * The exact action may depend on the console configuration and on the message properties.
     */
    public enum Flag {
        /** Ignore messages until WAKE is received. */ SLEEP,
        /** Stop ignoring messages. */ WАКЕ,
        /** Stop scrolling. */ PAUSE,
        /** Resume scrolling. */ RESUME,
        /** Request user attention. */ ATTENTION,
        /** Put the console in WARNING state. */ WARNING,
        /** Put the console in ALARM state. */ ALARM,
        /** Clear the console alert state. */ CLEAR,
        /** Submit local WARNING to the graphical console alert service. */ LOCAL_WARNING,
        /** Submit local ALARM to the graphical console alert service. */ LOCAL_ALARM,
        /** Clear the local alert state. */ LOCAL_CLEAR,
        /** Mark the message. */ MARK
    }

// -- Fields : -----------------------------------------------------------------
    
    private final BusMessage message;
    private String text;
    private EnumSet<Flag> flags;
    private Color color;

    
// -- Life cycle : -------------------------------------------------------------
    
    public FilteredMessage(BusMessage busMessage) {
        this.message = busMessage;
    }
    
    public FilteredMessage(FilteredMessage filteredMessage) {
        message = filteredMessage.message;
        text = filteredMessage.text;
        EnumSet<Flag> as = filteredMessage.flags;
        flags = (as == null || as.isEmpty()) ? null : EnumSet.copyOf(as);
        color = filteredMessage.color;
    }
    
    
// -- Getters and setters : ----------------------------------------------------
    
    /**
     * Returns the wrapped bus message, usually the original message received over the buses.
     * @return The wrapped bus message object.
     */
    public BusMessage getMessage() {
        return message;
    }
    
    /**
     * Returns the textual representation associated with this filtered message.
     * @return String representation of this message.
     */
    public String getText() {
        return text == null ? message.toString() : text;
    }
    
    /**
     * Associates a textual representation with this filtered message.
     * @param text New string representation for this message.
     */
    public void setText(String text) {
        this.text = text;
    }
    
    /**
     * Returns the flags associated with this message.
     * @return Set of flags associated with this message, or {@code null} if there are no flags.
     */
    public EnumSet<Flag> getFlags() {
        return flags;
    }
    
    /**
     * Associates a flags with this message.
     * @param action The flag.
     */
    public void addFlag(Flag action) {
        if (flags == null) {
            flags = EnumSet.of(action);
        } else {
            flags.add(action);
        }
    }
    
    /**
     * Removes all flags from this message.
     */
    public void clearFlags() {
        flags = null;
    }
    
    /**
     * Returns the color to be used for printing this message.
     * @return the color associated with this message, or {@code null} if there is no associated color.
     */
    public Color getColor() {
        return color;
    }
    
    /**
     * Sets the color to be used for printing this message.
     * @param color Color for this message.
     */
    public void setColor(Color color) {
        this.color = color;
    }
    
}
