package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import java.util.AbstractMap.SimpleEntry;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.base.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;

/**
 * Abstract monitoring view that splits channels into groups, and displays one group at a time in a child view.
 * <p>
 * All methods defined by this class should be called on EDT.
 * All setter methods should be called before {@code getPage()}, {@code install()}, and {@code uninstall()}.
 * 
 * @author onoprien
 */
abstract public class GroupView extends AbstractMonitorView {

// -- Fields : -----------------------------------------------------------------
    
    protected Descriptor descriptor = new Descriptor();
    
    private final TreeMap<String,Object> views = new TreeMap<>((s1,s2) -> {
        if (s1 == null) {
            if (s2 == null) {
                return 0;
            } else {
                return -1;
            }
        } else {
            if (s2 == null) {
                return 1;
            } else {
                return s1.compareTo(s2);
            }
        }
    }); // group --> AbstractMonitorView or List<Map.Entry<String,AgentChannel>>
    private final Map<String,TreeSet<String>> subgroups = new HashMap<>(); // partial group prefix -> set of next segments; if the prefix itself is a valid group, the set it is mapped to will contain an empty string.
    
    protected int depth;
    protected String currentGroup;
    protected LinkedList<String> selectionHistory;


// -- Implementing AbstractMonitorView level 1 hooks : -------------------------

    /**
     * Updates {@code views} or calls {@code addChannels(...)} on a child view if it already exists.
     * If there is no current group, call {@code selectGroup(null)}.
     */
    @Override
    public final void addChannels(List<AgentInfo> agents, List<Map.Entry<String,AgentChannel>> channels) {
        
        // separate new channels by group
        
        Map<String,List<Map.Entry<String,AgentChannel>>> channelsByGroup = new LinkedHashMap<>();
        channels.forEach(e ->  {
            String[] ss = separateGroup(e.getKey());
            if (ss != null) {
                AgentChannel ch = e.getValue();
                List<Map.Entry<String, AgentChannel>> groupChannels = channelsByGroup.get(ss[0]);
                if (groupChannels == null) {
                    groupChannels = new ArrayList<>(128);
                    channelsByGroup.put(ss[0], groupChannels);
                }
                groupChannels.add(new SimpleEntry(ss[1], ch));
            }
        });
        
        // update {@code views} and {@code subgroups}, notify existing views
        
        List<String> newGroups = new ArrayList<>();
        channelsByGroup.forEach((group, cc) -> {
            Object o = views.get(group);
            if (o == null) {
                views.put(group, cc);
                addToSubgroups(group);
                newGroups.add(group);
            } else if (o instanceof AbstractMonitorView) {
                ((AbstractMonitorView)o).addChannels(agents, cc);
            } else {
                HashSet<String> agentNames = new HashSet<>();
                agents.forEach(a -> agentNames.add(a.getName()));
                ArrayList<Map.Entry<String,AgentChannel>> newList = new ArrayList<>();
                ((List<Map.Entry<String,AgentChannel>>)o).forEach(e -> {
                    AgentChannel ch = e.getValue();
                    if (ch == null || !agentNames.contains(ch.getAgentName())) {
                        newList.add(e);
                    }
                });
                newList.addAll(cc);
                views.put(group, newList);
            }
        });
        
        // notify subclasses
        
        if (!newGroups.isEmpty()) {
            groupsAdded(newGroups);
        }
        
        // if no group is currently selected, try to select a default one
        
        if (currentGroup == null) {
            selectGroup(null);
        }
    }

    /**
     * Updates {@code views} or calls {@code removeChannels(...)} on a child view if it exists.
     */
    @Override
    public final void removeChannels(List<AgentInfo> agents, List<Map.Entry<String,AgentChannel>> channels) {
        
        // separate new channels by group
        
        Map<String,List<Map.Entry<String,AgentChannel>>> channelsByGroup = new LinkedHashMap<>();
        channels.forEach(e ->  {
            String[] ss = separateGroup(e.getKey());
            if (ss != null) {
                AgentChannel ch = e.getValue();
                List<Map.Entry<String, AgentChannel>> groupChannels = channelsByGroup.get(ss[0]);
                if (groupChannels == null) {
                    groupChannels = new ArrayList<>(128);
                    channelsByGroup.put(ss[0], groupChannels);
                }
                groupChannels.add(new SimpleEntry(ss[1], ch));
            }
        });
        
        // update {@code views} and {@code subgroups}, notify existing views
        
        List<String> deadGroups = new ArrayList<>();
        channelsByGroup.forEach((group, cc) -> {
            Object o = views.get(group);
            if (o instanceof AbstractMonitorView) {
                AbstractMonitorView view = (AbstractMonitorView)o;
                view.removeChannels(agents, cc);
//                if (view.isEmpty()) {
//                    // FIXME: depending on {@code discardRemovedChannels} policy, might want to remove the view.
//                    // deadGroups.add(group);
//                }
            } else if (o instanceof List<?>) {
                Set<String> displayPathsToRemove = channels.stream().map(e -> e.getKey()).collect(Collectors.toSet());
                List<Map.Entry<String,AgentChannel>> groupChannels = (List<Map.Entry<String,AgentChannel>>)o;
                Iterator<Map.Entry<String,AgentChannel>> it = groupChannels.iterator();
                while (it.hasNext()) {
                    if (displayPathsToRemove.contains(it.next().getKey())) {
                        it.remove();
                    }
                }
                if (groupChannels.isEmpty()) {
                    deadGroups.add(group);
                    views.remove(group);
                    removeFromSubgroups(group);
                }
            }
        });
        
        // notify subclasses
        
        if (!deadGroups.isEmpty()) {
            groupsRemoved(deadGroups);
        }
    }

    /**
     * Forward updates to the current view, if any.
     */
    @Override
    public final void updateChannels(List< Map.Entry<String,Map.Entry<AgentChannel,List<String>>> > channels) {
        if (currentGroup != null && currentGroup.isEmpty()) return;
        List< Map.Entry<String, Map.Entry<AgentChannel, List<String>>>> currentChannels;
        if (channels == null) {
            currentChannels = null;
        } else {
            currentChannels = new ArrayList<>(128);
            channels.forEach(e -> {
                String[] ss = separateGroup(e.getKey());
                if (ss != null && Objects.equals(currentGroup, ss[0])) {
                    currentChannels.add(new SimpleEntry(ss[1], e.getValue()));
                }
            });
        }
        Object o = views.get(currentGroup);
        if (o instanceof AbstractMonitorView) {
            ((AbstractMonitorView)o).updateChannels(currentChannels);
        }
    }

    
// -- Methods overridable by subclasses : --------------------------------------
    
    /**
     * Splits display path produced by the filter into group and path inside group.
     * Group name can be {@code null}.
     * If this method returns {@code null}, the channel should not be displayed.
     * <p>
     * The implementation provided by this class splits around "//" if this view has {@code depth} 0,
     * chop off first {@code depth} segments (separated by slashes) otherwise.
     * 
     * @param displayPath Display path produced by the filter.
     * @return 2-element array: [group, path inside group].
     */
    protected String[] separateGroup(String displayPath) {
        switch (descriptor.depth) {
            case 0:
                int i = displayPath.indexOf("//");
                if (i == -1) {
                    return new String[]{null, displayPath};
                } else {
                    return new String[]{displayPath.substring(0, i), displayPath.substring(i + 2)};
                }
            case 1:
                String[] ss = displayPath.split("/+", 2);
                return ss.length == 2 ? ss : new String[]{null, displayPath};
            default:
                String[] gg = displayPath.split("/+", descriptor.depth + 1);
                StringJoiner group = new StringJoiner("/");
                int n = gg.length - 1;
                for (int ii = 0; ii < n; ii++) {
                    group.add(gg[ii]);
                }
                return new String[]{group.toString(), gg[n]};
        }
    }
    
    /**
     * Reverse {@code separateGroup(...)}.
     * 
     * @param group Group.
     * @param path Local display path inside group.
     * @return Global display path as reported by the filter associated with this view.
     */
    protected String mergeGroup(String group, String path) {
        String delimeter = descriptor.depth == 0 ? "//" : "/";
        return group == null ? path : group + delimeter + path;
    }
    
    /**
     * Splits the group name into segments.
     * 
     * @param group Group name.
     * @return Array of segments.
     */
    protected String[] getPath(String group) {
        if (group == null || group.isEmpty()) {
            return new String[0];
        } else {
            return group.split("/", descriptor.depth);
        }
    }
    
    /**
     * Creates a view for the specified group.
     * 
     * @param group Group name.
     * @return New monitor view.
     */
    protected AbstractMonitorView createView(String group) {
        AbstractMonitorView view = null;
        AbstractMonitorView.Descriptor viewDescriptor = null;
        
        // Look for a view descriptor for this group
        
        Map<String, AbstractMonitorView.Descriptor> m = getDescriptor().getViews();
        if (m != null) {
            viewDescriptor = m.get(group);
        }
        
        // Look for default descriptor
        
        if (viewDescriptor == null) {
            viewDescriptor = getDescriptor().getDefaultView();
        }
        
        // Create view
        
        if (viewDescriptor != null) {
            view = (AbstractMonitorView) PersistenceService.getService().make(viewDescriptor);
        }
        if (view == null) {
            view = (LazyTreeView) PersistenceService.getService().make(MonitorView.CATEGORY, LazyTreeView.CREATOR_PATH);
            String[] ff = getDescriptor().getFields();
            if (ff != null) {
                view.getDescriptor().setFields(ff);
            }
            ff = getDescriptor().getCompactFields();
            if (ff != null) {
                view.getDescriptor().setCompactFields(ff);
            }
        }
        
        // Set filter and formatter
        
        if (view.getFilter() == null) {
            view.setFilter(new Filter(group));
        }
        if (formatter != null) {
            view.setFormatter(formatter);
        }
        return view;
    }
    
    /**
     * Selects and displays the view for the specified group.
     * If a partial group is given, figures out the complete one.
     * If {@code ""}, the default group is selected.
     * 
     * On exit from this method {@code currentGroup} is set and {@code selectionHistory} is updated.
     * 
     * @param group Group or prefix that specifies a group of groups.
     */
    protected void selectGroup(String group) {
        
        // no views
        
        if (views.isEmpty()) {
            currentGroup = "";
            groupSelected();
            return;
        }
        
        // "" is a request to select first group from history if present, default otherwise
        
        if (group != null && group.equals("")) {
            group = selectionHistory == null || selectionHistory.isEmpty() ? null : selectionHistory.getFirst();
        }
        
        // find full group in history if partial group is given
        
        if (!views.containsKey(group) && selectionHistory != null && !selectionHistory.isEmpty()) {
            String prefix = group == null ? "" : group +"/";
            ListIterator<String> it = selectionHistory.listIterator();
            while (it.hasNext()) {
                String g = it.next();
                if (g.startsWith(prefix)) {
                    it.remove();
                    group = g;
                    break;
                }
            }
        }
        
        // find default group, starting with partial
        
        if (!views.containsKey(group)) {
            if (group == null) {
                group = views.firstKey();
            } else {
                String prefix = group +"/";
                boolean notFound = true;
                for (String g : views.navigableKeySet()) {
                    if (g != null && g.startsWith(prefix)) {
                        group = g;
                        notFound = false;
                        break;
                    }
                }
                if (notFound) group = views.firstKey();
            }
        }
        
        // set current group and update history
        
        currentGroup = group;
        if (currentGroup != null) {
            if (selectionHistory == null) {
                selectionHistory = new LinkedList<>();
            } else {
                String[] ss = getPath(currentGroup);
                Iterator<String> it = selectionHistory.iterator();
                while (it.hasNext()) {
                    String g = it.next();
                    String[] gg = getPath(g);
                    if (gg.length <= ss.length) {
                        int n = Math.min(gg.length, ss.length-1);
                        boolean remove = true;
                        for (int i=0; i<n; i++) {
                            if (!ss[i].equals(gg[i])) {
                                remove = false;
                                break;
                            }
                        }
                        if (remove) it.remove();
                    }
                }
            }
            selectionHistory.addFirst(currentGroup);
        }
        
        // update view
        
        getView(currentGroup).updateChannels(null);
        
        // notify subclass
        
        groupSelected();
    }

    @Override
    public void setFormatter(MonitorFormat formatter) {
        super.setFormatter(formatter);
        views.values().forEach(o -> {
            if (o instanceof MonitorView) {
                ((MonitorView)o).setFormatter(formatter);
            }
        });
    }
    
    
// -- Methods to be implemented by subclasses : --------------------------------
    
    abstract protected void groupsAdded(List<String> groups);
    
    abstract protected void groupsRemoved(List<String> groups);
    
    abstract protected void groupSelected();
    
    
// -- Utility methods for use by subclasses : ----------------------------------
    
    final protected AbstractMonitorView getView(String group) {
        Object o = views.get(group);
        if (o instanceof AbstractMonitorView) {
            return (AbstractMonitorView)o;
        } else {
            AbstractMonitorView view = createView(group);
            views.put(group, view);
            if (o instanceof List<?>) {
                List<Map.Entry<String, AgentChannel>> channels = (List<Map.Entry<String, AgentChannel>>) o;
                HashMap<String,AgentInfo> nameToInfo = new HashMap<>();
                channels.stream().forEach(e -> {
                    AgentChannel ch = e.getValue();
                    nameToInfo.putIfAbsent(ch.getAgentName(), ch.getAgent());
                });
                view.addChannels(new ArrayList<>(nameToInfo.values()), channels);
            }
            return view;
        }
    }    
    
    final protected boolean hasGroup(String group) {
        return views.containsKey(group);
    }
    
    /**
     * @param parent Full or partial group (x0/.../xN), or {@code null} to get top-level subgroups.
     * @return List of group segments at the next level (x[N+1]).
     *         {@code null} if {@code parent} is neither a valid group nor a partial group;
     *         empty array if {@code parent} is a valid group but not a partial group (no subgroups);
     *         contains "" if {@code parent} is both a valid group and a partial group (has subgroups).
     */
    final protected String[] getSubgroups(String parent) {
        TreeSet<String> gg = subgroups.get(parent);
        if (gg == null) {
            return hasGroup(parent) ? new String[0] : null;
        } else {
            return gg.toArray(new String[0]);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /**
     * Updates {@code subgroups}.
     * 
     * @param group Group to add.
     */
    private void addToSubgroups(String group) {
        
        String[] ss = getPath(group);
        int n = ss.length;
        String parent = null;
        for (int i=0; i<n; i++) {
            TreeSet<String> sg = subgroups.get(parent);
            if (sg == null) {
                sg = new TreeSet<>();
                if (hasGroup(parent)) {
                    sg.add("");
                }
                subgroups.put(parent, sg);
            }
            sg.add(ss[i]);
            parent = parent == null ? ss[i] : parent +"/"+ ss[i];
        }
        
        TreeSet<String> set = subgroups.get(group);
        if (set != null) {
            set.add("");
        }
        
        if (n > depth) {
            depth = n;
        }
    }
    
    /**
     * Updates {@code subgroups}.
     * 
     * @param group Group to add.
     * @return True if the tree depth has increased as a result of this call.
     */
    private void removeFromSubgroups(String group) {
        
        String[] ss = getPath(group);
        int n = ss.length;
        String parent = null;
        for (int i=0; i<n; i++) {
            TreeSet<String> sg = subgroups.get(parent);
            sg.remove(ss[i]);
            if (sg.size() == 1 && sg.contains("")) {
                subgroups.remove(parent);
            }
            parent = parent == null ? ss[i] : parent +"/"+ ss[i];
        }
        
        TreeSet<String> set = subgroups.get(group);
        if (set != null) {
            set.remove("");
        }
        
        int newDepth = 0;
        for (String g : views.keySet()) {
            String[] gg = getPath(g);
            if (gg.length > newDepth) {
                newDepth = gg.length;
            }
        }
        depth = newDepth;
    }

    
// -- Per-group filter : -------------------------------------------------------

    /**
     * Filter to be associated with a group view.
     */
    protected class Filter extends AbstractChannelsFilter {

        private final String group;
        
        public Filter(String group) {
            this.group = group;
        }

        @Override
        public String getName() {
            String name = filter.getName();
            if (name != null && group != null && !group.isEmpty()) {
                name = name +":"+ group;
            }
            return name;
        }

        @Override
        public List<String> getAgents() {
            return filter.getAgents();
        }

        @Override
        public List<String> getOriginChannels() {
            return filter.getOriginChannels();
        }

        @Override
        public List<String> getDisplayChannels() {
            return filterGroup(filter.getDisplayChannels());
        }

        @Override
        public String getOriginPath(String displayPath) {
            return filter.getOriginPath(mergeGroup(group, displayPath));
        }

        @Override
        public List<String> getDisplayPaths(AgentChannel channel) {
            return filterGroup(filter.getDisplayPaths(channel));
        }

        @Override
        public List<String> getDisplayPaths(String originPath) {
            return filterGroup(filter.getDisplayPaths(originPath));
        }

        @Override
        public List<String> getFields(boolean compact) {
            return filter.getFields(compact);
        }
        
        private List<String> filterGroup(List<String> in) {
            if (in == null) {
                return null;
            } else {
                return in.stream().map(c -> separateGroup(c)).filter(ss -> ss != null).map(ss -> ss[1]).collect(Collectors.toList());
            }
        }

    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * JavaBean that contains information required to re-create this view in its current state.
     */
    static public class Descriptor extends AbstractMonitorView.Descriptor {

        private HashMap<String, AbstractMonitorView.Descriptor> views;
        private String[] selectionHistory;
        private int depth;
        private AbstractMonitorView.Descriptor defaultView;

        public AbstractMonitorView.Descriptor getDefaultView() {
            return defaultView;
        }

        public void setDefaultView(AbstractMonitorView.Descriptor defaultView) {
            this.defaultView = defaultView;
        }

        public int getDepth() {
            return depth;
        }

        public void setDepth(int depth) {
            this.depth = depth;
        }

        public String[] getSelectionHistory() {
            return selectionHistory;
        }

        public void setSelectionHistory(String[] selectionHistory) {
            this.selectionHistory = selectionHistory;
        }

        public HashMap<String, AbstractMonitorView.Descriptor> getViews() {
            return views;
        }

        public void setViews(HashMap<String, AbstractMonitorView.Descriptor> views) {
            this.views = views;
        }

        @Override
        public Descriptor clone() {
            Descriptor desc = (Descriptor) super.clone();
            if (desc.views != null) {
                desc.views = new HashMap<>(desc.views);
                desc.views.entrySet().forEach(e -> e.setValue(e.getValue().clone()));
            }
            if (desc.defaultView != null) {
                desc.defaultView = desc.defaultView.clone();
            }
            return desc;
        }
        
    }
    
    /**
     * Returns a descriptor that contains information required to re-create this view in its current state.
     * @return View descriptor.
     */
    @Override
    public Descriptor save() {
        Descriptor desc = descriptor.clone();
        if (views.isEmpty()) {
            desc.setViews(null);
        } else {
            HashMap<String, AbstractMonitorView.Descriptor> dd = new HashMap<>();
            desc.setViews(dd);
            for (Map.Entry<String,Object> e : views.entrySet()) {
                if (e.getValue() instanceof AbstractMonitorView) {
                    dd.put(e.getKey(), ((AbstractMonitorView)e.getValue()).save());
                }
            }
        }
        if (selectionHistory != null && !selectionHistory.isEmpty()) {
            desc.selectionHistory = selectionHistory.toArray(new String[0]);
        } else if (currentGroup != null) {
            desc.selectionHistory = new String[] {currentGroup};
        } else {
            desc.selectionHistory = null;
        }
        return desc;
    }
    
    /**
     * Restores this view to the state described by the provided descriptor, to the extent possible.
     * @param descriptor View descriptor.
     */
    @Override
    public void restore(PersistableMonitorView.Descriptor descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor.clone();
            depth = this.descriptor.depth;
            HashMap<String, AbstractMonitorView.Descriptor> dd = this.descriptor.getViews();
            if (dd != null) {
                for (Map.Entry<String, AbstractMonitorView.Descriptor> e : dd.entrySet()) {
                    Object v = views.get(e.getKey());
                    if (v instanceof AbstractMonitorView) {
                        ((AbstractMonitorView)v).restore(e.getValue());
                    }
                }
            }
            if (this.descriptor.selectionHistory != null && this.descriptor.selectionHistory.length > 0) {
                selectionHistory = new LinkedList<>(Arrays.asList(this.descriptor.selectionHistory));
                selectGroup(this.descriptor.selectionHistory[0]);
            }
        }
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }
    
    
// -- Testing : ----------------------------------------------------------------
    
    @Create(category = "AgentChannelsFilter",
            name = "Explicit group tester",
            path = "Test/Explicit group",
            description = "")
    static public AbstractChannelsFilter test1() {
        return new AbstractChannelsFilter() {
            @Override
            public String getDisplayPath(AgentChannel channel) {
                String type = channel.get(AgentChannel.Key.DATA_TYPE);
                String path = channel.getPath();
                if (type == null) {
                    return path;
                } else {
                    switch (type) {
                        case "CONFIGURATION":
                            return path.replaceFirst("/", "//");
                        case "STATE":
                            return "state/"+ channel.getAgentName() +"//"+ channel.getLocalPath();
                        default:
                            return path;
                    }
                }
            }
            
        };
    }
    

}
