package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.util.List;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.base.filter.ConfigurationFilter;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.services.persist.Persistable;

/**
 * Special {@code MonitorView} for displaying and editing subsystem configurations.
 *
 * @author onoprien
 */
public class ConfigurationView extends ComboGroupView {

// -- Fields : -----------------------------------------------------------------
    
    static public final String PAGE_CREATOR_PATH = "Built-In/Miscellaneous/Configuration";
    static public final String PAGE_CREATOR_PATH_WHOLECAMERA = "Built-In/Whole Camera/Configuration";
    
    private JPanel root;
    private final Action onSave, onDrop;

// -- Life cycle : -------------------------------------------------------------
    
    public ConfigurationView() {
        super(0, null);
        
        onSave = new AbstractAction("Save") {
            @Override
            public void actionPerformed(ActionEvent e) {
                String[] target = getTarget();
                if (target != null) {
                    CommandService.getService().send(target[0], "saveChangesForCategories", target[1]);
                }
            }
        };
        onDrop = new AbstractAction("Drop") {
            @Override
            public void actionPerformed(ActionEvent e) {
                String[] target = getTarget();
                if (target != null) {
                    CommandService.getService().send(target[0], "dropChangesForCategories", target[1]);
                }
            }
        };
    }
    
    
// -- Decorate ComboGroupView : ------------------------------------------------

    @Override
    public JComponent getPanel() {
        if (root == null) {
            root = new JPanel(new BorderLayout());
            JComponent viewPanel = super.getPanel();
            viewPanel.setBorder(BorderFactory.createEtchedBorder());
            root.add(viewPanel, BorderLayout.CENTER);
            Box buttonPanel = Box.createHorizontalBox();
            buttonPanel.add(Box.createHorizontalGlue());
            JButton saveButton = new JButton(onSave);
            buttonPanel.add(saveButton);
            buttonPanel.add(Box.createRigidArea(Const.HDIM));
            JButton dropButton = new JButton(onDrop);
            buttonPanel.add(dropButton);
            root.add(buttonPanel, BorderLayout.SOUTH);
        }
        return root;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /**
     * Gets command target.
     * @return [agent,category].
     */
    private String[] getTarget() {
        try {
            String[] gg = getPath(currentGroup);
            switch (gg.length) {
                case 1:
                    String agent = getFilter().getAgents().get(0);
                    return new String[] {agent, gg[0]};
                case 2:
                    return gg;
                default:
                    return null;
            }
        } catch (NullPointerException | ClassCastException | IndexOutOfBoundsException x) {
            return null;
        }
    }
    

// -- Creator for configuration MonitorPage : ----------------------------------

    @Create(category = MonitorPage.CATEGORY,
            name = "Configuration",
            path = PAGE_CREATOR_PATH,
            description = "Configurations for one or more subsystems, organized by category.")
    static public MonitorPage makeConfigPage(
            @Par(def = Par.NULL, desc = "Page name. If not specified, the page will be named after the selected subsystem.") String name,
            @Par(def = Par.NULL, desc = "List of subsystem names. If null, configurations for all subsystems are displayed.") List<String> subsystems)
    {
        if (subsystems != null && subsystems.isEmpty()) subsystems = null;
        name = name == null || name.trim().isEmpty() ? (subsystems == null || subsystems.size() != 1 ? "Config" : subsystems.get(0)) : name;
        ConfigurationView view = new ConfigurationView();
        view.getDescriptor().setLabels(subsystems == null ? new String[] {"Subsystem", "Category"} : new String[] {"Category"});
        return new MonitorPage(view, new ConfigurationFilter(null, subsystems), name) {
            @Override // making page uneditable
            public MonitorPage edit(String title, Component parent) {
                return null; // FIXME: why isn't it asking for re-creation?
            }
            @Override
            public void restore(Persistable.Descriptor descriptor) {
                try {
                    getView().restore(((MonitorPage.Descriptor)descriptor).getView());
                } catch (NullPointerException | ClassCastException x) {
                }
            }
        };
    }
    
    @Create(category = MonitorPage.CATEGORY,
            name = "Configuration",
            path = PAGE_CREATOR_PATH_WHOLECAMERA,
            description = "Configurations for the whole camera, organized by subsystem and category.")
    static public MonitorPage makeConfigPage() {
        return makeConfigPage(null, null);
    }
    
}
