package org.lsst.ccs.gconsole.plugins.alert;

import java.awt.event.ActionEvent;
import java.lang.reflect.InvocationTargetException;
import java.util.*;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.gconsole.agent.command.CommandHandle;
import org.lsst.ccs.gconsole.agent.command.CommandSender;
import org.lsst.ccs.gconsole.agent.command.CommandTask;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.plugins.tracer.LsstTracerPlugin;
import org.lsst.ccs.gconsole.plugins.tracer.Tracer;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.alert.AlertService;

/**
 * Graphical console plugin that handles alerts received from CCS subsystems.
 *
 * @author onoprien
 */
@Plugin(name="LSST Alert Notification Plugin",
        id="alert",
        description="LSST CCS alert notification service.")
public class LsstAlertPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    static private final String OPT_AUTO_START_VIEWER = "autoStartViewer";
    static private final String OPT_AUTO_START_TRACER = "autoStartTracer";
    static final String OPT_HISTORY = "history";
    static final String OPT_SOUND = "sound";
    static final String OPT_TOFRONT = "toFront";
    static final String OPT_SELECT = "selectLast";
    static final String OPT_IGNOREMUTED = "ignoreMuted";
    static final String OPT_IGNOREACK = "ignoreAck";
    
    private AlertService alertService;
    private CommandSender commandSender;
    
    private Action viewerAction;
    private volatile AlertViewer viewer; // modified on EDT only

    
// -- Lifecycle : --------------------------------------------------------------

    @Override
    public void initialize() {
        
        alertService = getConsole().getAgentService(AlertService.class);
        commandSender = CommandService.getService().getSender();
        commandSender.setCommandHandle(CommandHandle.NONE);
        
        // Define plugin settings and preferences
        
        getServices().addProperty(OPT_AUTO_START_VIEWER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, "Startup", "${"+ OPT_AUTO_START_VIEWER +"} Start viewer on console sturtup.");
        getServices().addProperty(OPT_AUTO_START_TRACER, false);
        getServices().addPreference(new String[] {"LSST","Alerts"}, "Startup", "${"+ OPT_AUTO_START_TRACER +"} Start message viewer on console sturtup.");
        
        getServices().addProperty(OPT_HISTORY, false);
        getServices().addProperty(OPT_SOUND, true);
        getServices().addProperty(OPT_TOFRONT, true);
        getServices().addProperty(OPT_SELECT, true);
        getServices().addProperty(OPT_IGNOREACK, true);
        getServices().addProperty(OPT_IGNOREMUTED, true);
        
        // Create menus for launching alert viewer and tracer
        
        viewerAction = new AbstractAction("Viewer") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ((Boolean) getValue(Action.SELECTED_KEY)) {
                    onNewAlertViewer(null);
                } else {
                    stopAlertViewer();
                }
                
            }
        };
        viewerAction.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(viewerAction, "400: CCS Tools :-1:4", "Alerts:1");
        
        Action act  = new AbstractAction("Messages") {
            @Override
            public void actionPerformed(ActionEvent e) {
                onNewAlertTracer();
            }
        };
        getServices().addMenu(act, "400: CCS Tools :-1:4", "Alerts:2");
        
        ImageIcon icon = new ImageIcon(getClass().getResource("alert_16.png"));
        act  = new AbstractAction("Alerts", icon) {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (viewer == null) {
                    onNewAlertViewer(null);
                } else {
                    viewer.toFront();
                }
            }
        };
        act.putValue(Action.SHORT_DESCRIPTION, "Show alert viewer");
        getConsole().insertToolbarAction(act, 1);
    }

    @Override
    public void start() {
        if ((boolean) getServices().getProperty(OPT_AUTO_START_VIEWER)) {
            onNewAlertViewer(null);
        }
        if ((boolean) getServices().getProperty(OPT_AUTO_START_TRACER)) {
            onNewAlertTracer();
        }
    }
  
    @Override
    public void shutdown() {
        // At the moment, nothing meaningfull can be done on plugin shutdown, since it might be called from JVM shutdown hook.
        // That's OK - standard tools are never shut down.
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    public void onNewAlertViewer(AlertViewer.Descriptor config) {
        if (SwingUtilities.isEventDispatchThread()) {
            if (viewer == null) {
                viewerAction.putValue(Action.SELECTED_KEY, true);
                viewer = new AlertViewer(this, config);
                alertService.addListener(viewer);
            }
        } else {
            SwingUtilities.invokeLater(() -> onNewAlertViewer(config));
        }
    }
    
    public void stopAlertViewer() {
        if (SwingUtilities.isEventDispatchThread()) {
            if (viewer != null) {
                viewerAction.putValue(Action.SELECTED_KEY, false);
                alertService.removeListener(viewer);
                viewer.destroy();
                viewer = null;
            }
        } else {
            SwingUtilities.invokeLater(this::stopAlertViewer);
        }
    }
    
    public void onNewAlertTracer() { // FIXME
        Tracer tracer = new Tracer();
        tracer.getDescriptor().setName("Alerts");
        tracer.setFilter(filteredMessage -> {
            BusMessage bm = filteredMessage.getMessage();
            if (bm instanceof StatusRaisedAlert) {
                StatusRaisedAlert am = (StatusRaisedAlert) bm;
                Alert alert = am.getRaisedAlert();
                String id = alert.getAlertId();
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getCCSTimeStamp().getUTCInstant())).append(" : ");
                sb.append(am.getRaisedAlertState());
                sb.append(" from ").append(am.getOriginAgentInfo().getName()).append(".\n");
                sb.append("ID: ").append(id).append(".  Description:\n");
                sb.append(alert.getDescription()).append("\n");
                filteredMessage.setText(sb.toString());
                return filteredMessage;
            } else if (bm instanceof StatusClearedAlert) {
                StatusClearedAlert am = (StatusClearedAlert) bm;
                StringBuilder sb = new StringBuilder();
                sb.append(AlertViewer.formatTimeStamp(am.getCCSTimeStamp().getUTCInstant())).append(" : ");
                sb.append(am.getOriginAgentInfo().getName()).append(" cleared alerts ");
                for (String id : am.getClearAlertIds()) {
                    sb.append(id).append(", ");
                }
                sb.delete(sb.length()-2, sb.length()).append(".\n");
                filteredMessage.setText(sb.toString());
                return filteredMessage;
            } else {
                return null;
            }
        });
        LsstTracerPlugin tracerPlugin = (LsstTracerPlugin) getConsole().getConsoleLookup().lookup(LsstTracerPlugin.class);
        tracerPlugin.openTracer(tracer);
    }
    
    public void clearAlerts(String subsystem, AlertState maxSeverity, String... ids) {
        if (getConsole().getMessagingAccess().getAgentPresenceManager().agentExists(subsystem)) {
            CommandTask task;
            if (ids == null) {
                if (maxSeverity == AlertState.ALARM) {
                    task = commandSender.send(subsystem, "clearAllAlerts", "--" + AgentCommandDictionaryService.withLockOption.getName());
                } else {
                    task = commandSender.send(subsystem, "clearAllWarnings");
                }
            } else {
                if (maxSeverity == AlertState.ALARM) {
                    String[] args = new String[ids.length + 1];
                    System.arraycopy(ids, 0, args, 1, ids.length);
                    args[0] = "--" + AgentCommandDictionaryService.withLockOption.getName();
                    task = commandSender.sendEncoded(subsystem, "clearAlerts", args);
                } else {
                    task = commandSender.sendEncoded(subsystem, "clearWarnings", ids);
                }
            }
            try {
                Object result = task.getResult();
                if (result instanceof Exception x) {
                    getConsole().error("Failed to clear alerts.", x);
                } else if (result instanceof String[] cleared) {
                    String message;
                    if (cleared.length == 0) {
                        message = "No alerts have been cleared.";
                    } else if (ids == null) {
                        StringBuilder sb = new StringBuilder("<html><b>Cleared:</b>").append(System.lineSeparator());
                        sb.append(String.join(System.lineSeparator(), new TreeSet<>(Arrays.asList(cleared))));
                        message = sb.toString();
                    } else if (cleared.length == ids.length) {
                        StringBuilder sb = new StringBuilder("All requested alerts have been cleared:").append(System.lineSeparator());
                        sb.append(String.join(System.lineSeparator(), new TreeSet<>(Arrays.asList(ids))));
                        message = sb.toString();
                    } else {
                        SortedSet<String> yes = new TreeSet<>(Arrays.asList(cleared));
                        SortedSet<String> no = new TreeSet<>(Arrays.asList(ids));
                        no.removeAll(yes);
                        StringBuilder sb = new StringBuilder("<html><b>Cleared:</b>").append(System.lineSeparator());
                        yes.forEach(id -> sb.append(id).append(System.lineSeparator()));
                        sb.append("<html><p><b>Unable to clear:</b>").append(System.lineSeparator());
                        no.forEach(id -> sb.append(id).append(System.lineSeparator()));
                        message = sb.toString();
                    }
                    SwingUtilities.invokeLater(() -> JOptionPane.showMessageDialog(viewer == null ? null : viewer.getGUI(), message, subsystem, JOptionPane.PLAIN_MESSAGE));
                } else {
                    getConsole().error("Failed to clear alerts: " + Objects.toString(result));
                }
            } catch (InterruptedException x) {
            }
        } else {
            if (ids == null) {
                alertService.clearAllAlertsForSource(subsystem);
            } else {
                alertService.clearAlertsForSource(subsystem, ids);
            }
        }
    }
    
    public void ackAlerts(String message, String subsystem, String[] toAck, String[] toNack) {
        if (getConsole().getMessagingAccess().getAgentPresenceManager().agentExists(subsystem)) {
            if (toAck.length != 0) {
                String[] args = new String[toAck.length + 1];
                System.arraycopy(toAck, 0, args, 1, toAck.length);
                args[0] = message;
                CommandTask task = commandSender.sendEncoded(subsystem, "acknowledgeAlerts", args);
                processAckCommand(task, "acknowledge", subsystem, toAck);
            }
            if (toNack.length != 0) {
                CommandTask task = commandSender.sendEncoded(subsystem, "unacknowledgeAlerts", toNack);
                processAckCommand(task, "unacknowledge", subsystem, toAck);
            }
        }
    }
    
    private void processAckCommand(CommandTask task, String command, String subsystem, String[] ids) {
        try {
            Object result = task.getResult();
            if (result instanceof Exception x) {
                getConsole().error("Failed to " + command + " alerts. ", x);
            } else if (result instanceof String[] acked) {
                TreeSet<String> request = new TreeSet<>(Arrays.asList(ids));
                request.removeAll(new TreeSet<>(Arrays.asList(acked)));
                if (!request.isEmpty()) {
                    String out = "Failed to " + command + " alerts: " + System.lineSeparator() + String.join(System.lineSeparator(), request);
                    SwingUtilities.invokeLater(() -> JOptionPane.showMessageDialog(viewer == null ? null : viewer.getGUI(), out, subsystem, JOptionPane.PLAIN_MESSAGE));
                }
            } else {
                getConsole().error("Failed to " + command + " alerts. " + Objects.toString(result));
            }
        } catch (InterruptedException x) {
        }
    }
    

// -- Saving/restoring : -------------------------------------------------------
    
    @Override
    public ComponentDescriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        desc.setHistory((boolean) getServices().getProperty(OPT_HISTORY));
        desc.setMute((boolean) getServices().getProperty(OPT_SOUND));
        desc.setToFront((boolean) getServices().getProperty(OPT_TOFRONT));
        desc.setSelectLast((boolean) getServices().getProperty(OPT_SELECT));
        desc.setIgnoreAck((boolean) getServices().getProperty(OPT_IGNOREACK));
        desc.setIgnoreMuted((boolean) getServices().getProperty(OPT_IGNOREMUTED));
        if (viewer != null) {
            if (SwingUtilities.isEventDispatchThread()) {
                desc.setViewerConfig(viewer.save());
            } else {
                try {
                    SwingUtilities.invokeAndWait(() -> {
                        if (viewer != null) {
                            desc.setViewerConfig(viewer.save());
                        }
                    });
                } catch (InterruptedException|InvocationTargetException x) {
                }
            }
        }
        return desc;
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        
        if (!(storageBean instanceof Descriptor)) {
            throw new IllegalArgumentException("Illegal descriptor type: "+ storageBean.getClassName());
        }
        Descriptor desc = (Descriptor) storageBean;
        
        getServices().setProperty(OPT_HISTORY, desc.isHistory());
        getServices().setProperty(OPT_SOUND, desc.isMute());
        getServices().setProperty(OPT_TOFRONT, desc.isToFront());
        getServices().setProperty(OPT_SELECT, desc.isSelectLast());
        getServices().setProperty(OPT_IGNOREACK, desc.isIgnoreAck());
        getServices().setProperty(OPT_IGNOREMUTED, desc.isIgnoreMuted());
        
        AlertViewer.Descriptor viewerConfig = desc.getViewerConfig();
        if (viewerConfig == null) {
            if (viewer != null) {
                stopAlertViewer();
            }
        } else {
            stopAlertViewer();
            onNewAlertViewer(viewerConfig);
        }
        return true;
    }
    
    static public class Descriptor extends ComponentDescriptor {

        private boolean selectLast;
        private boolean toFront;
        private boolean mute;
        private boolean history;
        private boolean ignoreAck;
        private boolean ignoreMuted;
        private AlertViewer.Descriptor viewerConfig;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public boolean isSelectLast() {
            return selectLast;
        }

        public void setSelectLast(boolean selectLast) {
            this.selectLast = selectLast;
        }

        public boolean isToFront() {
            return toFront;
        }

        public void setToFront(boolean toFront) {
            this.toFront = toFront;
        }

        public boolean isMute() {
            return mute;
        }

        public void setMute(boolean mute) {
            this.mute = mute;
        }

        public boolean isHistory() {
            return history;
        }

        public void setHistory(boolean history) {
            this.history = history;
        }

        public boolean isIgnoreAck() {
            return ignoreAck;
        }

        public void setIgnoreAck(boolean ignoreAck) {
            this.ignoreAck = ignoreAck;
        }

        public boolean isIgnoreMuted() {
            return ignoreMuted;
        }

        public void setIgnoreMuted(boolean ignoreMuted) {
            this.ignoreMuted = ignoreMuted;
        }

        public AlertViewer.Descriptor getViewerConfig() {
            return viewerConfig;
        }

        public void setViewerConfig(AlertViewer.Descriptor viewerConfig) {
            this.viewerConfig = viewerConfig;
        }
        
    }
        
}
