package org.lsst.ccs.gconsole.base;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.io.Serializable;
import java.util.*;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultCaret;
import javax.swing.text.Position;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyleContext;
import javax.swing.text.StyledDocument;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.gconsole.services.persist.Savable;

/**
 * Component that displays textual records.
 * All access to this class should be on AWT Event Dispatching Thread.
 *
 * @author onoprien
 */
public class OutputPane extends JScrollPane implements Savable {

// -- Fields : -----------------------------------------------------------------
    
    private final TextPane textPane;
    private final StyledDocument document;
    private final LinkedList<Position> records = new LinkedList<>(); // starts of records
    
    private final Style attBase, attPlane;
    private final HashMap<Color,Style> color2style = new HashMap<>(4);
    private String recordTerminator = "\n";
    private int lineLength;
    
    private Descriptor descriptor;

// -- Life cycle : -------------------------------------------------------------
    
    public OutputPane() {
        this(null);
    }
    
    public OutputPane(Descriptor desc) {
        
        descriptor = desc == null ? new Descriptor() : desc.clone();
        
        textPane = new TextPane();
        setViewportView(textPane);
        document = textPane.getStyledDocument();
        
        recordTerminator = descriptor.isSkipLine() ? "\n\n" : "\n";
        
        // Styles

        Style def = StyleContext.getDefaultStyleContext().getStyle(StyleContext.DEFAULT_STYLE);
        attBase = document.addStyle("attBase", def);
        StyleConstants.setFontSize(attBase, descriptor.getFontSize());
        attPlane = document.addStyle("attPlane", attBase);
        
        // Max line length
        
        lineLength = descriptor.getMaxLineLength();
    }
    
    
// -- Setters : ----------------------------------------------------------------
    
    private void setFontSize(int size) {
        setButtonSize(this, size);
        StyleConstants.setFontSize(attBase, size);
        descriptor.setFontSize(size);
    }
    
    public void setSkipLine(boolean skipLine) {
        recordTerminator = skipLine ? "\n\n" : "\n";
        descriptor.setSkipLine(skipLine);
        textPane.actSkipLine.putValue(Action.SELECTED_KEY, skipLine);
    }
    
    public void setLineLength(int maxLength) {
        lineLength = maxLength;
        descriptor.setMaxLineLength(lineLength);
    }
    
// -- Adding records : ---------------------------------------------------------
    
    public void println(Object record, Color color) {
        
        JScrollBar bar = getVerticalScrollBar();
        boolean adjusting = bar.getValueIsAdjusting();
        boolean bottom = bar.getValue() + bar.getVisibleAmount() >= bar.getMaximum();
        
        try {
            int start = document.getEndPosition().getOffset()-1;
            String s = record.toString().replaceFirst("\\s++$", "");
            if (lineLength > 0) {
                String[] ss = split(s);
                SimpleAttributeSet attButton = new SimpleAttributeSet(attBase);
                StyleConstants.setAlignment(attButton, StyleConstants.ALIGN_CENTER);
                Button button = new Button(ss);
                StyleConstants.setComponent(attButton, button);
                document.insertString(start, " ", attButton);
                document.insertString(start + 1, ss[0] + recordTerminator, getStyle(color));
                Position beforeRecord = document.createPosition(start);
                button.before = beforeRecord;
                button.addActionListener(e -> button.act());
                records.add(beforeRecord);
            } else {
                document.insertString(start, s + recordTerminator, getStyle(color));
                records.add(document.createPosition(start));
            }
            
            if (!adjusting) {
                if (bottom) {
                    while (records.size() > descriptor.maxRecords) {
                        records.removeFirst();
                        document.remove(0, records.getFirst().getOffset());
                    }
                    scrollToBottom();
                } else {
                    Position top = null;
                    while (records.size() > descriptor.maxRecords) {
                        int topOffset;
                        if (top == null) {
                            topOffset = textPane.viewToModel(getViewport().getViewPosition());
                            top = document.createPosition(topOffset);
                        } else {
                            topOffset = top.getOffset();
                        }
                        int firstOffset = records.getFirst().getOffset();
                        if (firstOffset < topOffset) {
                            records.removeFirst();
                            document.remove(0, records.getFirst().getOffset());
                        } else {
                            int max = descriptor.getMaxRecordsOverflow();
                            if (max == 0) max = descriptor.getMaxRecords() * 10;
                            if (records.size() > max) {
                                while (records.size() > descriptor.maxRecords) {
                                    records.removeFirst();
                                    document.remove(0, records.getFirst().getOffset());
                                }
                                top = null;
                                scrollToBottom();
                            } else {
                                break;
                            }
                        }
                    }
                    if (top != null) {
                        scrollToPosition(top);
                    }
                }
            }
        } catch (BadLocationException x) {
            throw new RuntimeException(x);
        }
    }
    
    public void println(Object record) {
        println(record, null);
    }
    
    
// -- Text pane class : --------------------------------------------------------
    
    final class TextPane extends JTextPane implements HasPopupItems {
        
        private final Action actClear, actFontLarger, actFontSmaller, actSkipLine, actWrap, actCollapse;
        
        TextPane() {
        
            // JTextPane settings
            
            setEditable(false);
//            setCaretPosition(0);
            try {
                ((DefaultCaret) getCaret()).setUpdatePolicy(DefaultCaret.NEVER_UPDATE);
            } catch (ClassCastException x) {
            }
        
            // Actions
            
            KeyStroke ks;
            String name;

            name = "Clear";
            ks = KeyStroke.getKeyStroke(KeyEvent.VK_C, KeyEvent.ALT_DOWN_MASK);
            actClear = new AbstractAction(name) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    try {
                        document.remove(0, document.getLength());
                    } catch (BadLocationException x) {
                    }
                    records.clear();
                }
            };
            actClear.putValue(Action.ACCELERATOR_KEY, ks);
            getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(ks, name);
            getActionMap().put(name, actClear);

            name = "Larger";
            ks = KeyStroke.getKeyStroke(KeyEvent.VK_EQUALS, KeyEvent.ALT_DOWN_MASK);
            actFontLarger = new AbstractAction(name) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    int fs = StyleConstants.getFontSize(attBase);
                    setFontSize(fs + 1);
                }
            };
            actFontLarger.putValue(Action.ACCELERATOR_KEY, ks);
            getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(ks, name);
            getActionMap().put(name, actFontLarger);

            name = "Smaller";
            ks = KeyStroke.getKeyStroke(KeyEvent.VK_MINUS, KeyEvent.ALT_DOWN_MASK);
            actFontSmaller = new AbstractAction(name) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    int fs = StyleConstants.getFontSize(attBase);
                    if (fs > 6) {
                        setFontSize(fs - 1);
                    }
                }
            };
            actFontSmaller.putValue(Action.ACCELERATOR_KEY, ks);
            getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(ks, name);
            getActionMap().put(name, actFontSmaller);

            name = "Extra line";
            actSkipLine = new AbstractAction(name) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setSkipLine((Boolean) actSkipLine.getValue(Action.SELECTED_KEY));
                }
            };
            actSkipLine.putValue(Action.SELECTED_KEY, descriptor.isSkipLine());
            getActionMap().put(name, actSkipLine);

            name = "Wrap";
            actWrap = new AbstractAction(name) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    descriptor.setNowrap(!(Boolean) actWrap.getValue(Action.SELECTED_KEY));
                }
            };
            actWrap.putValue(Action.SELECTED_KEY, !descriptor.isNowrap());
            getActionMap().put(name, actWrap);

            name = "Collapse";
            actCollapse = new AbstractAction(name) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setLineLength((Boolean) actCollapse.getValue(Action.SELECTED_KEY) ? 100 : 0);
                }
            };
            actCollapse.putValue(Action.SELECTED_KEY, lineLength > 0);
            getActionMap().put(name, actCollapse);
            
        }

        @Override
        public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component cmpnt, Point point) {
            menu.add(new JMenuItem(actClear));
            JMenu fontMenu = new JMenu("Font size");
            menu.add(fontMenu);
            fontMenu.add(new JMenuItem(actFontLarger));
            fontMenu.add(new JMenuItem(actFontSmaller));
            menu.add(new JCheckBoxMenuItem(actSkipLine));
            menu.add(new JCheckBoxMenuItem(actWrap));
            menu.add(new JCheckBoxMenuItem(actCollapse));
            return menu;
        }

        @Override
        public boolean getScrollableTracksViewportWidth() {
            if (descriptor.isNowrap()) {
                return getUI().getPreferredSize(this).width <= getParent().getSize().width;
            } else {
                return super.getScrollableTracksViewportWidth();
            }
        }

    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void setButtonSize(Component comp, int size) {
        if (comp instanceof JButton) {
            comp.setPreferredSize(new Dimension(size, size));
//            comp.setPreferredSize(new Dimension(comp.getPreferredSize().width, size));
        } else if (comp instanceof Container) {
            Component[] cc = ((Container)comp).getComponents();
            for (Component c : cc) {
                setButtonSize(c, size);
            }
        }
    }
    
    private void scrollToBottom() {
        SwingUtilities.invokeLater(() -> {
            JScrollBar bar = getVerticalScrollBar();
            bar.setValue(bar.getMaximum());
        });
    }
    
    private void scrollToPosition(Position p) {
        try {
            Rectangle r = textPane.modelToView(p.getOffset());
            textPane.scrollRectToVisible(r);
        } catch (BadLocationException x) {
        }
    }
    
    private Style getStyle(Color color) {
        if (color == null) return attPlane;
        Style style = color2style.get(color);
        if (style == null) {
            style = document.addStyle(Integer.toHexString(color.getRGB()), attBase);
            StyleConstants.setForeground(style, color);
            color2style.put(color, style);
        }
        return style;
    }
    
    private final class Button extends JButton {
        
        Position before;
        int length;
        String hidden;
        
        Button(String[] ss) {
            super(ss[1] == null ? "-" : "+");
            hidden = ss[1];
            length = ss[0].length();
            int fs = StyleConstants.getFontSize(attBase);
            setPreferredSize(new Dimension(fs, fs));
            setAlignmentY(.8f);
            setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
            setMargin(new Insets(0, 0, 0, 0));
            setFocusPainted(false);
        }
        
        void act() {
            try {
                int offset = before.getOffset() + 1;
                if (hidden == null) {
                    String s = document.getText(offset, length);
                    String[] ss = split(s);
                    if (ss[1] != null) {
                        setText("+");
                        hidden = ss[1];
                        int newLength = ss[0].length();
                        document.remove(offset + newLength, length - newLength);
                        length = newLength;
                    }
                } else {
                    document.insertString(offset + length, hidden, document.getCharacterElement(offset+2).getAttributes());
                    length = length + hidden.length();
                    hidden = null;
                    setText("-");
                }
            } catch (BadLocationException x) {
            }
        }
    }
    
    private String[] split(String s) {
        int i = s.indexOf("\n");
        if (i == -1) {
            if (s.length() > lineLength) {
                return new String[] {s.substring(0, lineLength), s.substring(lineLength)};
            } else {
                return new String[] {s, null};
            }
        } else if (i > lineLength) {
            return new String[]{s.substring(0, lineLength), s.substring(lineLength)};
        } else {
            return new String[]{s.substring(0, i), s.substring(i + 1)};
        }
    }

    
// -- Saving/restoring : -------------------------------------------------------
    
    @Override
    public void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
        } else if (descriptor == null) {
            this.descriptor = new Descriptor();
        }
        setFontSize(this.descriptor.getFontSize());
        setSkipLine(this.descriptor.isSkipLine());
    }

    @Override
    public Descriptor save() {
        return descriptor.clone();
    }
    
    static public class Descriptor implements Serializable, Cloneable {

        private boolean nowrap;
        private boolean skipLine;
        private int maxLineLength;
        private int maxRecords = 300;
        private int maxRecordsOverflow = 3000;
        private int fontSize = 15;

        public int getMaxLineLength() {
            return maxLineLength;
        }

        public void setMaxLineLength(int maxLineLength) {
            this.maxLineLength = maxLineLength;
        }

        public boolean isSkipLine() {
            return skipLine;
        }

        public void setSkipLine(boolean skipLine) {
            this.skipLine = skipLine;
        }

        public int getMaxRecordsOverflow() {
            return maxRecordsOverflow;
        }

        public void setMaxRecordsOverflow(int maxRecordsOverflow) {
            this.maxRecordsOverflow = maxRecordsOverflow;
        }

        public boolean isNowrap() {
            return nowrap;
        }

        public void setNowrap(boolean nowrap) {
            this.nowrap = nowrap;
        }

        public int getMaxRecords() {
            return maxRecords;
        }

        public void setMaxRecords(int maxRecords) {
            this.maxRecords = maxRecords;
        }

        public int getFontSize() {
            return fontSize;
        }

        public void setFontSize(int fontSize) {
            this.fontSize = fontSize;
        }
        
        @Override
        public Descriptor clone() {
            try {
                return (Descriptor) super.clone();
            } catch (CloneNotSupportedException x) {
                throw new RuntimeException(); // never
            }
        }
        
    }
    
    
// -- Testing : ----------------------------------------------------------------
    
    static public void main(String... args) {
        SwingUtilities.invokeLater(() -> {
            JFrame frame = new JFrame("OutputPane test");
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            OutputPane pane = new OutputPane();
            frame.add(pane, BorderLayout.CENTER);
            Box buttonBox = Box.createHorizontalBox();
            frame.add(buttonBox, BorderLayout.SOUTH);
            
            buttonBox.add(new JButton(new AbstractAction("Add one record") {
                private int line = 0;
                @Override
                public void actionPerformed(ActionEvent e) {
                    pane.println("-" + line++ +"-- Line rtgertgretghrtehrth tryhtyhtyh tyjtyjyu \njuyjyu \niunfbweprfui \niofjeo");
                }
            }));
            
            buttonBox.add(new JButton(new AbstractAction("Add red record") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    pane.println("--- Line rtgertgretghrtehrth", Color.RED);
                }
            }));

            buttonBox.add(Box.createHorizontalGlue());
            frame.pack();
            frame.setVisible(true);
        });

    }

//    static public void main(String... args) {
//        DefaultStyledDocument doc = new DefaultStyledDocument();
//        Position start = doc.getStartPosition();
//        Position end = doc.getEndPosition();
//        try {
//            System.out.println(doc.getText(0, doc.getLength()) +", start: "+ start.getOffset() +", end: "+ end.getOffset());
//            doc.insertString(0, "01234", null);
//            System.out.println(doc.getText(0, doc.getLength()) +", start: "+ start.getOffset() +", end: "+ end.getOffset());
//            Position p = doc.createPosition(5);
//            doc.insertString(5, "5", null);
//            System.out.println(doc.getText(0, doc.getLength()) +", start: "+ start.getOffset() +", end: "+ end.getOffset());
//            System.out.println(p.getOffset());
//        } catch (BadLocationException x) {
//
//        }
//    }
    
}
