package org.lsst.ccs.gconsole.base;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.MouseEvent;
import java.util.*;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.base.panel.PanelType;
import org.lsst.ccs.gconsole.plugins.trending.TrendPage;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.gconsole.util.swing.InputFilteringPanel;
import org.lsst.ccs.gconsole.util.swing.MouseAdapterDoubleClick;


/**
 * Service that provides "glass pane" functionality, blocking mouse input to data panels.
 *
 * @author onoprien
 */
public class InputBlocker {

// -- Fields : -----------------------------------------------------------------
    
    // Icons: selected(X|Blocked|Active) * 4 + others(X|Blocked|Active|Mixed)
    static private final Icon[] icons = {
        new ImageIcon(InputBlocker.class.getResource("block_X_X.png"), "no panels"),                        // 0
        new ImageIcon(InputBlocker.class.getResource("block_X_B.png"), "no selection, others blocked"),     // 1
        new ImageIcon(InputBlocker.class.getResource("block_X_A.png"), "no selection, others active"),      // 2
        new ImageIcon(InputBlocker.class.getResource("block_X_M.png"), "no selection, others mixed"),        //3
        new ImageIcon(InputBlocker.class.getResource("block_B_X.png"), "selection blocked, no others"),      //4
        new ImageIcon(InputBlocker.class.getResource("block_B_B.png"), "selection blocked, others blocked"), //5
        new ImageIcon(InputBlocker.class.getResource("block_B_A.png"), "selection blocked, others active"),  //6
        new ImageIcon(InputBlocker.class.getResource("block_B_M.png"), "selection blocked, others mixed"),   //7
        new ImageIcon(InputBlocker.class.getResource("block_A_X.png"), "selection active, no others"),       //8
        new ImageIcon(InputBlocker.class.getResource("block_A_B.png"), "selection active, others blocked"),  //9
        new ImageIcon(InputBlocker.class.getResource("block_A_A.png"), "selection active, others active"),  //10
        new ImageIcon(InputBlocker.class.getResource("block_A_M.png"), "selection active, others mixed"),   //11
    };
    static private final String WRAPPER_KEY = "_InputBlocker_";
    
    static private JLabel label;
    static private Action blockSelected, enableSelected, blockAll, enableAll;
    
    static private boolean updateScheduled;
    static private final InputFilteringPanel.Listener filterPanelListener = e -> {
        if (!updateScheduled) {
            updateScheduled = true;
            SwingUtilities.invokeLater(InputBlocker::updateLabel);
        }
    };
    
// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Initializes this service.
     * Can be called on any thread.
     */
    static public void initialize() {
        ThreadUtil.invokeLater(() -> {
            
            label = new JLabel(icons[0]);
            Console.getConsole().insertStatusBarComponent(label, 0);
            label.setToolTipText("<html>Safe mode blocks input to data panels:<br>- click to toggle selected;<br>- double-click to toggle all.</html>");
            label.addMouseListener(new MouseAdapterDoubleClick() {
                @Override
                public void mouseReleased(MouseEvent e) {
                }
                @Override
                public void mousePressed(MouseEvent e) {
                }
                @Override
                public void mouseClickedFiltered(MouseEvent e) {
                    if (e.getButton() == MouseEvent.BUTTON1) {
                        switch (e.getClickCount()) {
                            case 1:
                                toggleBlockSelected();
                                break;
                            case 2:
                                toggleBlockAll();
                                break;
                        }
                    }
                }
            });

            blockSelected = new AbstractAction("Block input to selected data panel") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setBlockSelected(true);
                }
            };
            enableSelected = new AbstractAction("Enable input to selected data panel") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setBlockSelected(false);
                }
            };
            blockAll = new AbstractAction("Block input to all data panels") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setBlockAll(true);
                }
            };
            enableAll = new AbstractAction("Enable input to all data panels") {
                @Override
                public void actionPerformed(ActionEvent e) {
                    setBlockAll(false);
                }
            };
            JPopupMenu popupMenu = new JPopupMenu();
            popupMenu.add(blockSelected);
            popupMenu.add(enableSelected);
            popupMenu.add(blockAll);
            popupMenu.add(enableAll);
            label.setComponentPopupMenu(popupMenu);

            Console.getConsole().getPanelManager().addListener(e -> {
                Panel key = e.getStandardKey();
                if (key == Panel.SELECTED || key == Panel.OPEN) {
                    SwingUtilities.invokeLater(() -> {
                        updateLabel();
                    });
                }
            });
        });
    }
    
    /**
     * Wraps the provided data panel into an input-filtering component.
     * 
     * @param content Panel to be wrapped.
     * @param blocked True if filtering should be immediately activated.
     * @return Newly created wrapper.
     */
    static public JComponent wrap(JComponent content, boolean blocked) {
        Glass glass = new Glass(content);
        glass.setBlocked(blocked);
        content.putClientProperty(WRAPPER_KEY, glass);
        glass.addListener(filterPanelListener);
        return glass;
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    static public boolean isBlocked(Component component) {
        Glass glass = getGlass(component);
        return glass == null ? false : glass.isBlocked();
    }
    
    
// -- Actions : ----------------------------------------------------------------
    
    static public void setBlockSelected(boolean block) {
        ThreadUtil.invokeLater(() -> {
            PanelManager panMan = Console.getConsole().getPanelManager();
            Component selectedDataPanel = panMan.getSelectedPanel(PanelType.DATA);
            if (selectedDataPanel != null ) {
                setBlocked(selectedDataPanel, block);
            }
        });
    }
    
    static public void setBlockAll(boolean block) {
        ThreadUtil.invokeLater(() -> {
            PanelManager panMan = Console.getConsole().getPanelManager();
            panMan.getPanels(Panel.TYPE, PanelType.DATA).forEach(component -> {
                setBlocked(component, block);
            });
        });
    }
    
    static public void toggleBlockSelected() {
        ThreadUtil.invokeLater(() -> {
            PanelManager panMan = Console.getConsole().getPanelManager();
            Component selectedDataPanel = panMan.getSelectedPanel(PanelType.DATA);
            if (selectedDataPanel != null) {
                boolean block = isBlocked(selectedDataPanel);
                setBlocked(selectedDataPanel, !block);
            }
        });
    }
    
    static public void toggleBlockAll() {
        ThreadUtil.invokeLater(() -> {
            PanelManager panMan = Console.getConsole().getPanelManager();
            Component selectedDataPanel = panMan.getSelectedPanel(PanelType.DATA);
            List<? extends Component> dataPanels = panMan.getPanels(Panel.TYPE, PanelType.DATA);
            boolean block;
            if (selectedDataPanel == null || selectedDataPanel instanceof TrendPage) {
                block = dataPanels.stream().anyMatch(c -> isBlocked(c));
            } else {
                block = isBlocked(selectedDataPanel);
            }
            dataPanels.forEach(c -> setBlocked(c, !block));
        });
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    static private int getState() {
        PanelManager panMan = Console.getConsole().getPanelManager();
        Component selectedDataPanel = panMan.getSelectedPanel(PanelType.DATA);
        if (selectedDataPanel instanceof TrendPage) {
            selectedDataPanel = null;
        }
        boolean hasBlocked = false;
        boolean hasActive = false;
        for (Component c : panMan.getPanels(Panel.TYPE, PanelType.DATA)) {
            if (c != selectedDataPanel && !(c instanceof TrendPage)) {
                boolean blocked = isBlocked(c);
                hasBlocked = hasBlocked || blocked;
                hasActive = hasActive || !blocked;
                if (hasBlocked && hasActive) break;
            }
        }
        int state = 4 * (selectedDataPanel == null ? 0 : ( isBlocked(selectedDataPanel) ? 1 : 2 ) );
        if (hasBlocked) {
            state += hasActive ? 3 : 1;
        } else {
            state += hasActive ? 2 : 0;
        }
        return state;
    }
    
    static private void updateLabel() {
        updateScheduled = false;
        int state = getState();
        label.setIcon(icons[state]);
        
        blockSelected.setEnabled(state > 7);
        enableSelected.setEnabled(state > 3 && state < 8);
        blockAll.setEnabled(state == 2 || state == 3 || state > 5);
        enableAll.setEnabled(state == 1 || (state > 2 && state < 8) || state == 9 || state == 11);
        
        boolean noActivePanels = state == 0 || state == 1 || state == 4 || state == 5;
        if (!(state == 1 || state == 5)) {
            SelfDestructor.setSafe(noActivePanels);
        }
    }
    
    static private Glass getGlass(Component component) {
        try {
            return (Glass) ((JComponent) component).getClientProperty(WRAPPER_KEY);
        } catch (NullPointerException | ClassCastException x) {
            return null;
        }
    }
    
    static private void setBlocked(Component component, boolean blocked) {
        Glass glass = getGlass(component);
        if (glass != null) {
            glass.setBlocked(blocked);
        }
    }
    
    
// -- Wrapper panel class : ----------------------------------------------------
    
    static private class Glass extends InputFilteringPanel {
        
        public Glass(JComponent content) {
            super(content);
        }

        @Override
        protected void onBlock(MouseEvent e) {
            Object[] options = {"Enable this panel", "Enable all panels", "Keep disabled"};
            int input = JOptionPane.showOptionDialog(this, 
                    "Input through this panel has been disabled to prevent accidental actions. Would you like to re-enable it?",
                    "Safe mode", 
                    JOptionPane.DEFAULT_OPTION, JOptionPane.QUESTION_MESSAGE,
                    icons[getState()], 
                    options, "Keep disabled");
            switch(input) {
                case 0:
                    setBlocked(false);
                    break;
                case 1:
                    setBlockAll(false);
                    break;
            }
        }

    }

}
