package org.lsst.ccs.subsystem.monitor.ui.tree;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.io.OutputStream;
import java.util.*;
import javax.swing.AbstractCellEditor;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeCellEditor;
import javax.swing.tree.TreePath;
import org.freehep.application.studio.Studio;
import org.lsst.ccs.gconsole.plugins.monitor.LsstMonitorPlugin;
import org.lsst.ccs.gconsole.plugins.trending.TrendingService;
import org.lsst.ccs.subsystem.monitor.ui.CommandSender;

/**
 * Panel that displays monitoring data as a tree of tables.
 * All access to instances of this class, including construction, should happen on the EDT.
 *
 * @author onoprien
 */
public class DataTree {

// -- Fields : -----------------------------------------------------------------
    
    private static final int NFIELDS = Field.values().length;
    private static final Color COLOR_GOOD = new Color(160, 255, 160),
                               COLOR_WARN = new Color(255, 255, 100),
                               COLOR_ERR = new Color(255, 160, 160),
                               COLOR_OFF = new Color(160, 200, 255),
                               COLOR_POPUP = new Color(255, 255, 160);
    static final Font FONT_PLAIN = new Font("Helvetica", Font.PLAIN, 12);
    static final String MULTI_VALUE = "_m_";
    
    private final CommandSender sender;
    
    private String name;
    private final Tree tree;
    
    private DataNode dataRoot;
    private final HashMap<String,DataNodeLeaf> path2data = new HashMap<>();
    
// -- Life cycle : -------------------------------------------------------------
     
    public DataTree(CommandSender sender) {
        this.sender = sender;
        tree = new Tree();
        tree.setRootVisible(false);
        tree.setShowsRootHandles(true);
        tree.setRowHeight(0);
        tree.setCellRenderer(new TreeRenderer(false));
        tree.setCellEditor(new TreeEditor());
        tree.addMouseMotionListener(new MouseMotionListener() {
            @Override
            public void mouseMoved(MouseEvent e) {
                if (tree.getRowForLocation(e.getX(), e.getY()) != -1) {
                    tree.startEditingAtPath(tree.getPathForLocation(e.getX(), e.getY()));
                }
            }
            @Override
            public void mouseDragged(MouseEvent e) {}
        });
        tree.setEditable(true);
    }
    
    
// -- Updates : ----------------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">    
    public void setName(String name) {
        this.name = name;
    }
    
    public String getName() {
        return name;
    }

    public void setChannels(LinkedHashMap<String, Channel> channels) {
        
        buildDataTree(channels);
        DefaultMutableTreeNode root = buildTreeModel(dataRoot, true);
        tree.setModel(new DefaultTreeModel(root));

        List<DataNodeLeaf> leaves = dataRoot.appendLeaves(new ArrayList<>());
        leaves.forEach(leaf -> {
            tree.expandPath(new TreePath(leaf.parent.parent.modelNode.getPath()));
        });
    }

    public void addChannel(String path, Channel channel) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void removeChannel(String path) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void update(String path, Channel channel) {
        DataNodeLeaf leaf = path2data.get(path);
        if (leaf != null) {
            leaf.update(channel, null);
        }
    }

    public void update(String path) {
        update(path, (Channel)null);
    }

    public void update(String path, Field field) {
        DataNodeLeaf leaf = path2data.get(path);
        if (leaf != null) {
            leaf.update(null, field);
        }
    }

    public void update(String path, EnumSet<Field> fields) {
        update(path);
    }

    public JComponent getView() {
        return tree;
    }
// </editor-fold>    
    
// -- Customizing tree : -------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">    
    class Tree extends JTree implements MonitorDisplay {
        
        private String expansionState;
        
        @Override
        public void saveData(OutputStream out, String mimeType) {
            if (dataRoot != null) {
                List<DataNodeLeaf> leaves = dataRoot.appendLeaves(new ArrayList<>());
                ArrayList<Channel> channels = new ArrayList<>(leaves.size());
                leaves.forEach(leaf -> channels.add(leaf.channel));
                LsstMonitorPlugin.saveData(out, mimeType, channels);
            }
        }
        
        private boolean isDescendant(TreePath path1, TreePath path2) {
            int count1 = path1.getPathCount();
            int count2 = path2.getPathCount();
            if (count1 <= count2) {
                return false;
            }
            while (count1 != count2) {
                path1 = path1.getParentPath();
                count1--;
            }
            return path1.equals(path2);
        }

        void saveExpansionState(int row) {
            TreePath rowPath = getPathForRow(row);
            StringBuilder sb = new StringBuilder();
            int rowCount = getRowCount();
            for (int i = row; i < rowCount; i++) {
                TreePath path = getPathForRow(i);
                if (i == row || isDescendant(path, rowPath)) {
                    if (isExpanded(path)) {
                        sb.append(",").append(i - row);
                    }
                } else {
                    break;
                }
            }
            expansionState = sb.toString();
        }

        void restoreExpanstionState(int row) {
            StringTokenizer stok = new StringTokenizer(expansionState, ",");
            while (stok.hasMoreTokens()) {
                int token = row + Integer.parseInt(stok.nextToken());
                tree.expandRow(token);
            }
        }

    }
    
    class TreeRenderer extends DefaultTreeCellRenderer {
        
        private final NodePanel nodePanel = new NodePanel();
        private final boolean editable;
        
        TreeRenderer(boolean editable) {
            this.editable = editable;
        }
        
        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            DefaultMutableTreeNode treeNode = (DefaultMutableTreeNode) value;
            Object payload = treeNode.getUserObject();
            if (payload instanceof JComponent) {
                return (JComponent) payload;
            } else if (payload instanceof TableHolder) {
                return ((TableHolder)payload).getComponent(editable);
            } else if (payload instanceof DataNode) {
                nodePanel.setNode((DataNode) payload);
                return nodePanel;
            } else {
                return super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            }
        }

    }
    
    class TreeEditor extends AbstractCellEditor implements TreeCellEditor {

        private final TreeRenderer renderer;
        private Object editorValue;
        private Component component;

        public TreeEditor() {
            renderer = new TreeRenderer(true);
        }

        @Override
        public Component getTreeCellEditorComponent(JTree tree, Object value, boolean isSelected, boolean expanded, boolean leaf, int row) {
            editorValue = value;
            return renderer.getTreeCellRendererComponent(tree, value, isSelected, expanded, leaf, row, true);
        }

        /**
         * At this point the component is added to the tree (undocumented!) but 
         * tree's internal cleanup might not yet be done.
         */
        @Override
        public boolean shouldSelectCell(EventObject anEvent) {
            editorValue = null;
            if (anEvent instanceof MouseEvent) {
                redirect((MouseEvent) anEvent);
            }
            return false;
        }

        private void redirect(final MouseEvent anEvent) {
            SwingUtilities.invokeLater(() -> {
                try {
                    MouseEvent ev = SwingUtilities.convertMouseEvent(anEvent.getComponent(), anEvent, component);
                    component.dispatchEvent(ev);
                } catch (Exception x) {} // FIXME
            });
        }

        @Override
        public Object getCellEditorValue() {
            return editorValue;
        }
    }
    
    final class NodePanel extends Box {

        private final JLabel label = new JLabel();
        private JCheckBox compact, flatten, flip;
        private DataNode node;

        NodePanel() {
            super(BoxLayout.X_AXIS);
            add(label);
            compact = new JCheckBox();
            compact.setToolTipText("Compact");
            compact.setOpaque(false);
            compact.addActionListener(e -> {
                if (node != null) {
                    node.setCompact(compact.isSelected());
                }
            });
            add(compact);
            flatten = new JCheckBox();
            flatten.setToolTipText("Flatten");
            flatten.setOpaque(false);
            flatten.addActionListener(e -> {
                if (node != null) {
                    boolean selected = flatten.isSelected();
                    flip.setEnabled(selected);
                    node.setFlatten(selected);
                }
            });
            add(flatten);
            flip = new JCheckBox();
            flip.setToolTipText("Flip");
            flip.setOpaque(false);
            flip.addActionListener(e -> {
                if (node != null) {
                    node.setFlipped(flip.isSelected());
                }
            });
            add(flip);
            add(Box.createRigidArea(new Dimension(100,0)));
            add(Box.createHorizontalGlue());

        }
        
        void setNode(DataNode node) {
            this.node = null;
            if (node != null) {
                label.setText(" "+ node.name +" ");
                compact.setSelected(node.compact);
                compact.setEnabled(node.modelNode.getLeafCount() > 0);
                flatten.setSelected(node.flatten);
                flip.setSelected(node.flip);
                flatten.setEnabled(node.depth == 2 || node.depth == 3);
                flip.setEnabled(node.flatten);
                this.node = node;
            }
        }

    }
    
    class TableHolder {
        
        private final JPanel renderPanel, editPanel;
        
        TableHolder(AnyTableModel model) {
            JTable table = createTable(model);
            renderPanel = new JPanel(new BorderLayout());
            renderPanel.setBorder(BorderFactory.createLineBorder(Color.BLACK));
            if (model instanceof TableModelLeaves) renderPanel.add(table.getTableHeader(), BorderLayout.NORTH);
            renderPanel.add(table, BorderLayout.CENTER);
            table = createTable(model);
            editPanel = new JPanel(new BorderLayout());
            editPanel.setBorder(BorderFactory.createLineBorder(Color.BLACK));
            if (model instanceof TableModelLeaves) editPanel.add(table.getTableHeader(), BorderLayout.NORTH);
            editPanel.add(table, BorderLayout.CENTER);
        }
        
        JPanel getComponent(boolean editable) {
            return editable ? editPanel : renderPanel;
        }
        
    }
// </editor-fold>    

// -- Data tree node classes : -------------------------------------------------

// <editor-fold defaultstate="collapsed">    
    class DataNode {
        
        final String name;
        DataNode parent;
        boolean compact;
        boolean flatten;
        boolean flip;
        int depth;
        ArrayList<DataNodeLeaf> leaves;
        ArrayList<DataNode> children;
        
        DefaultMutableTreeNode modelNode;
        
        DataNode(String name) {
            this.name = name;
        }
        
        void setCompact(boolean compact) {
            if (compact == this.compact) return;
            this.compact = compact;
            tree.saveExpansionState(0);
            buildTreeModel(this, true);
            tree.restoreExpanstionState(0);
        }
        
        void setFlatten(boolean flatten) {
            if (flatten == this.flatten) return;
            this.flatten = flatten;
            buildTreeModel(this, true);
            tree.expandPath(new TreePath(modelNode.getPath()));
        }
         
        void setFlipped(boolean flip) {
            if (flip == this.flip) return;
            this.flip = flip;
            tree.saveExpansionState(0);
            buildTreeModel(this, true);
            tree.restoreExpanstionState(0);
       }
       
        DataNode addChild(String name) {
            if (children == null) children = new ArrayList<>(1);
            for (DataNode child : children) {
                if (child.name.equals(name)) {
                    return child;
                }
            }
            DataNode newChild = new DataNode(name);
            newChild.parent = this;
            children.add(newChild);
            return newChild;
        }
        
        DataNodeLeaf addLeaf(String name, Channel channel) {
            if (leaves == null) leaves = new ArrayList<>(1);
            DataNodeLeaf leaf = new DataNodeLeaf(name, channel);
            leaves.add(leaf);
            leaf.parent = this;
            return leaf;
        }
        
        List<DataNodeLeaf> appendLeaves(List<DataNodeLeaf> out) {
            if (out == null) out = new ArrayList<>();
            if (leaves != null) out.addAll(leaves);
            if (children != null) {
                for (DataNode child : children) {
                    child.appendLeaves(out);
                }
            }
            return out;
        }

        @Override
        public String toString() {
            return name;
        }
    }
    
    class DataNodeLeaf {
        
        final String name;
        Channel channel;
        DataNode parent;
        int id;
        AnyTableModel model;

        DataNodeLeaf(String name, Channel channel) {
            this.name = name;
            this.channel = channel;
        }
        
        void update(Channel channel, Field field) {
            if (channel != null) this.channel = channel;
            model.update(this, field);
        }
    }
// </editor-fold>    

// -- Monitoring table classes : -----------------------------------------------
    
// <editor-fold defaultstate="collapsed">        
    JTable createTable(AnyTableModel model) {
        JTable table = new JTable(model);
        
        JTableHeader header = table.getTableHeader();
        header.setReorderingAllowed(false);
//        hdr.setSize(hdr.getWidth(), hdr.getHeight() + 2);
//        if (model instanceof TableModelLeaves) {
////        TableCellRenderer hr = header.getDefaultRenderer();
////        System.out.println("Setting table...");
//            for (int i = 0; i < model.getColumnCount(); i++) {
//                TableColumn column = table.getColumnModel().getColumn(i);
//                column.sizeWidthToFit();
////            int prefWidth = hr.getTableCellRendererComponent(null, column.getHeaderValue(), false, false, 0, 0).getPreferredSize().width;
////            System.out.println(i +" : "+ prefWidth);
////            column.setPreferredWidth(prefWidth);
//            }
//        }





        header.setResizingAllowed(false);
        
        
        
        
        
        table.setDefaultRenderer(Object.class, new TableRenderer());
        table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
        table.setRowSelectionAllowed(false);
        table.setColumnSelectionAllowed(false);
        table.setFont(FONT_PLAIN);
        table.setRowHeight(table.getRowHeight() + 2);
        table.setShowGrid(true);
        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent evt) {
                tableMouseClicked(evt);
            }
        });
        
        return table;
    }
    
    private void tableMouseClicked(MouseEvent evt) {

        int nClick = evt.getClickCount();
        JTable table = (JTable)evt.getSource();
        Point point = evt.getPoint();
        int row = table.rowAtPoint(point);
        int column = table.columnAtPoint(point);
        if (nClick == 1) {
            // FIXME
        } else if (nClick == 2) {
            Object value = table.getValueAt(row, column);
            if (value instanceof ValueHolder) {
                ValueHolder vh = (ValueHolder) value;
                String[] path = {vh.channel.getSubsystemName(), vh.channel.getName()};
                Studio studio = (Studio) Studio.getApplication();
                if (studio == null) return;
                TrendingService trending = (TrendingService) studio.getLookup().lookup(TrendingService.class);
                if (trending == null) return;
                trending.show(path);
            }
        }
    }
    
    class TableRenderer extends DefaultTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {

            setBackground(Color.WHITE);
            setForeground(Color.BLACK);
            Component c;
            String s;
            if (column == 0) {
                super.getTableCellRendererComponent(table, value, false, false, row, column);
                setHorizontalAlignment(SwingConstants.LEFT);
            } else if (row == 0 && table.getModel() instanceof TableModelDeep) {
                super.getTableCellRendererComponent(table, value, false, false, row, column);
                setHorizontalAlignment(SwingConstants.CENTER);
            } else if (value instanceof ValueHolder) {
                ValueHolder vh = ((ValueHolder) value);
                super.getTableCellRendererComponent(table, vh.value, false, false, row, column);
                setBackground(vh.color);
                setHorizontalAlignment(SwingConstants.RIGHT);
            } else if (value instanceof LimitHolder) {
                LimitHolder vh = ((LimitHolder) value);
                super.getTableCellRendererComponent(table, vh.value, false, false, row, column);
                setForeground(vh.color);
                setHorizontalAlignment(SwingConstants.RIGHT);
            } else if (value instanceof AlarmHolder) {
                AlarmHolder ah = ((AlarmHolder) value);
                super.getTableCellRendererComponent(table, ah.getText(), false, false, row, column);
                setHorizontalAlignment(SwingConstants.CENTER);
            } else {
                if (MULTI_VALUE.equals(value)) {
                    value = "";
                    setBackground(Color.LIGHT_GRAY);
                }
                super.getTableCellRendererComponent(table, value, false, false, row, column);
                setHorizontalAlignment(SwingConstants.CENTER);
            }
            return this;
        }
    }
    
    class ValueHolder {
        String value;
        Color color;
        Channel channel;
        ValueHolder(String value, Color color, Channel channel) {
            this.value = value;
            this.color = color;
            this.channel = channel;
        }
    }
    
    class LimitHolder {
        String value;
        Color color;
        Channel channel;
        LimitHolder(String value, Color color, Channel channel) {
            this.value = value;
            this.color = color;
            this.channel = channel;
        }
        @Override
        public String toString() {
            return value;
        }
    }
    
    class AlarmHolder {
        String name;
        AlarmHolder(String name) {
            this.name = name;
        }
        String getText() {return name == null ? "" : "  \u2713";}
    }
    
    private Object getFormattedValue(Channel channel, Field field) {
        String s;
        double d;
        Color c;
        switch (field) {
            case DESCR:
                return channel.getDescription();
            case VALUE:
                d = channel.getValue();
                if (Double.isNaN(d)) {
                    return new ValueHolder("", Color.WHITE, channel);
                } else {
                    if (channel.isOnline()) {
                        c = channel.isGood() ? COLOR_GOOD : COLOR_ERR;
                    } else {
                        c = COLOR_OFF;
                    }
                    return new ValueHolder(String.format(channel.getFormat(), d), c, channel);
                }
            case UNITS:
                return channel.getUnits();
            case LOW:
                d = channel.getLowLimit();
                if (Double.isNaN(d)) {
                    return new LimitHolder("", Color.BLACK, channel);
                } else {
                    c = channel.isLowLimitChanged() ? Color.BLUE : Color.BLACK;
                    return new LimitHolder(String.format(channel.getFormat(), d), c, channel);
                }
            case ALERT_LOW:
                s = channel.getLowAlarm();
                return (s == null || s.isEmpty()) ? new AlarmHolder(null) : new AlarmHolder(s);
            case HIGH:
                d = channel.getHighLimit();
                if (Double.isNaN(d)) {
                    return new LimitHolder("", Color.BLACK, channel);
                } else {
                    c = channel.isHighLimitChanged() ? Color.BLUE : Color.BLACK;
                    return new LimitHolder(String.format(channel.getFormat(), d), c, channel);
                }
            case ALERT_HIGH:
                s = channel.getHighAlarm();
                return (s == null || s.isEmpty()) ? new AlarmHolder(null) : new AlarmHolder(s);
            case NAME:
                return channel.getName();
            default:
                return "";
        }
    }
    
    abstract class AnyTableModel extends AbstractTableModel {
        
        protected final DataNode node;
        protected DefaultMutableTreeNode jNode;
        
        AnyTableModel(DataNode node) {
            this.node = node;
        }
        
        void setJTreeNode(DefaultMutableTreeNode jNode) {
            
        }
        
        abstract void update(DataNodeLeaf leaf, Field field);
    }
    
    class TableModelLeaves extends AnyTableModel {
        
        TableModelLeaves(DataNode node) {
            super(node);
        }

        @Override
        public int getRowCount() {
            return node.leaves.size();
        }

        @Override
        public int getColumnCount() {
            return node.compact ? 2 : NFIELDS;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            DataNodeLeaf leaf = node.leaves.get(rowIndex);
            return columnIndex == 0 ? leaf.name : getFormattedValue(leaf.channel, Field.values()[columnIndex]);
        }

        @Override
        public String getColumnName(int column) {
            Field field = Field.values()[column];
            String out = field.toString();
            switch (field) {
                case NAME:
                    return "     "+ out +"     ";
                case VALUE:
                    return "    "+ out +"   ";
                case UNITS:
                    return " "+ out +" ";
                case LOW:
                    return "  "+ out +" ";
                case HIGH:
                    return " "+ out +" ";
                case ALERT_LOW:
                case ALERT_HIGH:
                    return out;
                case DESCR:
                    return "       "+ out +"       ";
                default:
                    return out;
            }
        }

        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return columnIndex == Field.HIGH.ordinal() || columnIndex == Field.LOW.ordinal();
        }

        @Override
        public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
            if (columnIndex == Field.HIGH.ordinal() || columnIndex == Field.LOW.ordinal()) {
                try {
                    double newValue = Double.parseDouble(aValue.toString());
                    DataNodeLeaf leaf = node.leaves.get(rowIndex);
                    Channel channel = leaf.channel;
                    boolean isHigh = columnIndex == Field.HIGH.ordinal();
                    double oldValue = isHigh ? channel.getHighLimit() : channel.getLowLimit();
                    if (Double.isNaN(oldValue) || (Math.abs(oldValue - newValue) > 0.00001)) { // FIXME
                        String target = isHigh ? "limitHi" : "limitLo";
                        sender.sendCommand(channel.getSubsystemName(), channel.getName(), "change", target, newValue);
                    }
                } catch (NumberFormatException x) {
                    return;
                }
                super.setValueAt(getValueAt(rowIndex, columnIndex), rowIndex, columnIndex);
            } else {
                super.setValueAt(aValue, rowIndex, columnIndex);
            }
        }

        @Override
        void update(DataNodeLeaf leaf, Field field) {
            if (field == null) {
                fireTableRowsUpdated(leaf.id, leaf.id);
            } else {
                fireTableCellUpdated(leaf.id, field.ordinal());
            }
//            getTreeModel().nodeChanged(node.modelNode);
            getTreeModel().nodeChanged(jNode);
        }
        
    }
    
    class TableModelDeep extends AnyTableModel {
        
        private Object[][] data; // [row][column], covers actual table
        private HashMap<String,int[]> path2cell;  // 3-node path to [row,column]
        private int nValueCols;
        private ArrayList<ArrayList<DataNodeLeaf>> rows; // lists of leaves in each row
        
        TableModelDeep(DataNode node) {
            super(node);

            List<DataNodeLeaf> leaves = new ArrayList<>();
            List<DataNodeLeaf> leaves3 = new ArrayList<>();
            
            for (DataNodeLeaf leaf : node.appendLeaves(new ArrayList<>())) {
                String[] path = getPath(node, leaf).split("/");
                if (path.length == 2) {
                    leaves.add(leaf);
                } else if (path.length == 3) {
                    leaves3.add(leaf);
                }
            }
            leaves.addAll(leaves3);

            ArrayList<String> rowNames = new ArrayList<>();
            ArrayList<String> columnNames = new ArrayList<>();            
            path2cell = new LinkedHashMap<>();
            rows = new ArrayList<>(); rows.add(null);
            String currentSection = "";
            int currentRow = 1;
            int currentColumn = 1;
            for (DataNodeLeaf leaf : leaves) {
                String path = getPath(node, leaf);
                String[] sPath = path.split("/");
                String section, row, column;
                if (sPath.length == 2) {
                    section = "";
                    row = node.flip ? sPath[1] : sPath[0];
                    column = node.flip ? sPath[0] : sPath[1];
                } else {
                    section = sPath[0];
                    row = node.flip ? sPath[2] : sPath[1];
                    row = row +"/"+ section;
                    column = node.flip ? sPath[1] : sPath[2];
                }
                if (!section.equals(currentSection)) {
                    rowNames.add(section);
                    rows.add(null);
                    currentRow++;
                    currentSection = section;
                }
                int iRow = rowNames.indexOf(row) + 1;
                if (iRow == 0) {
                    iRow = currentRow++;
                    rowNames.add(row);
                    rows.add(new ArrayList<>());
                }
                int iColumn = columnNames.indexOf(column) + 1;
                if (iColumn == 0) {
                    iColumn = currentColumn++;
                    columnNames.add(column);
                }
                path2cell.put(path, new int[] {iRow, iColumn});
                rows.get(iRow).add(leaf);
            }
            nValueCols = columnNames.size();
            
            int nRows = rows.size();
            int nColumns = node.compact ? nValueCols + 1 : nValueCols + NFIELDS -2;
            data = new Object[nRows][];
            for (int ir=0; ir<nRows; ir++) {
                data[ir] = new Object[nColumns];
                if (ir == 0) {  // title row
                    data[0][0] = "";
                    for (int ic=0; ic<nValueCols; ic++) {
                        data[0][ic+1] = columnNames.get(ic);
                    }
                    if (!node.compact) {
                        for (int ic=2; ic < NFIELDS-1; ic++) {
                            data[0][ic + nValueCols -1] = Field.values()[ic];
                        }
                    }
                } else if (rows.get(ir) == null) { // section title row
                    data[ir][0] = "<html><b> "+ rowNames.get(ir-1) +"</b>";
                    for (int ic=1; ic < nColumns; ic++) {
                        data[ir][ic] = "";
                    }
                } else { // normal row
                    data[ir][0] = rowNames.get(ir-1).split("/")[0];
                    if (!node.compact) {
                        for (int i = 2; i < NFIELDS - 1; i++) {
                            Field f = Field.values()[i];
                            Object newContent = null;
                            for (DataNodeLeaf dl : rows.get(ir)) {
                                Object c = getFormattedValue(dl.channel, f);
                                if (newContent == null) {
                                    newContent = c;
                                } else if (!newContent.equals(c)) {
                                    newContent = MULTI_VALUE;
                                    break;
                                }
                            }
                            data[ir][fieldToColumn(f)] = newContent;
                        }
                    }
                }
            }
            
            for (DataNodeLeaf leaf : leaves) {
                int[] cell = path2cell.get(getPath(node, leaf));
                data[cell[0]][cell[1]] = getFormattedValue(leaf.channel, Field.VALUE);
            }

        }

        @Override
        public int getRowCount() {
            return data.length;
        }

        @Override
        public int getColumnCount() {
            return node.compact ? nValueCols+1 : nValueCols + NFIELDS - 2;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            return data[rowIndex][columnIndex];
        }

        @Override
        void update(DataNodeLeaf leaf, Field field) {
            int[] cell = path2cell.get(getPath(node, leaf));
            if (field == null && !node.compact) {  // updating all fields in an extended table
                data[cell[0]][cell[1]] = getFormattedValue(leaf.channel, Field.VALUE);
                for (int i=2; i<NFIELDS-1; i++) {
                    Field f = Field.values()[i];
                    if (f.isUpdatable()) {
                        Object newContent = null;
                        for (DataNodeLeaf dl : rows.get(cell[0])) {
                            Object c = getFormattedValue(dl.channel, f);
                            if (newContent == null) {
                                newContent = c;
                            } else {
                                if (!newContent.equals(c)) {
                                    newContent = MULTI_VALUE;
                                    break;
                                }
                            }
                        }
                        data[cell[0]][fieldToColumn(f)] = newContent;
                    }
                }
                fireTableRowsUpdated(cell[0], cell[0]);
            } else if (field == Field.VALUE || (field == null && node.compact)) { // updating value field only
                data[cell[0]][cell[1]] = getFormattedValue(leaf.channel, Field.VALUE);
                fireTableCellUpdated(cell[0], cell[1]);
            } else if (field != Field.VALUE && !node.compact) { // updating single non-value field
                Object newContent = null;
                for (DataNodeLeaf dl : rows.get(cell[0])) {
                    Object c = getFormattedValue(dl.channel, field);
                    if (newContent == null) {
                        newContent = c;
                    } else if (!newContent.equals(c)) {
                        newContent = MULTI_VALUE;
                        break;
                    }
                }
                cell[1] = fieldToColumn(field);
                data[cell[0]][cell[1]] = newContent;
                fireTableCellUpdated(cell[0], cell[1]);
            }
//            ((DefaultTreeModel)tree.getModel()).nodeChanged(node.modelNode);
            ((DefaultTreeModel)tree.getModel()).nodeChanged(jNode);
        }
        
        private int fieldToColumn(Field field) {
            return field.ordinal() + nValueCols - 1;
        }
        
        private Field columnToField(int columnIndex) {
            return Field.values()[columnIndex - nValueCols + 1];
        }

    }
// </editor-fold>    
    
// -- Local methods : ----------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    static private String[] getPath(String displayPath) {
        int i = displayPath.indexOf("//");
        if (i != -1) displayPath = displayPath.substring(i+2);
        return displayPath.split("/");
    }
    
    static private String getPath(DataNode node, DataNodeLeaf leaf) {
        StringBuilder sb = new StringBuilder(leaf.name);
        DataNode parent = leaf.parent;
        while (parent != null) {
            if (parent == node) {
                return sb.toString();
            } else {
                sb.insert(0,"/").insert(0,parent.name);
                parent = parent.parent;
            }
        }
        return null;
    }
    
    /**  Sets dataRoot and path2data */
    private void buildDataTree(Map<String, Channel> channels) {
        path2data.clear();
        dataRoot = new DataNode("root");
        for (Map.Entry<String, Channel> e : channels.entrySet()) {
            String[] path = getPath(e.getKey());
            DataNode parent = dataRoot;
            for (int level=0; level<path.length-1; level++) {
                parent = parent.addChild(path[level]);
            }
            DataNodeLeaf leaf = parent.addLeaf(path[path.length-1], e.getValue());
            path2data.put(e.getKey(), leaf);
        }
        for (DataNodeLeaf leaf : path2data.values()) {
            int depth = 0;
            DataNode node = leaf.parent;
            while (node != null) {
                if (node.depth < ++depth) {
                    node.depth = depth;
                }
                node = node.parent;
            }
        }
    } 
    
    /** Builds JTree model branch under given root */
    private DefaultMutableTreeNode buildTreeModel(DataNode dataNode, boolean start) {
        
        DefaultMutableTreeNode parent = null;
        DefaultMutableTreeNode node;
        if (start) {
            node = dataNode.modelNode;
            if (node == null) {
                node = new DefaultMutableTreeNode(dataNode);
                dataNode.modelNode = node;
            } else {
                parent = (DefaultMutableTreeNode) node.getParent(); 
                node.removeAllChildren();
            }
        } else {
            node = new DefaultMutableTreeNode(dataNode);
            dataNode.modelNode = node;
        }
        
        if (dataNode.leaves != null) {
            node.add(createLeavesNode(dataNode));
        }
        
        if (dataNode.children != null) {
            if (dataNode.flatten) {
                node.add(createDeepNode(dataNode));
            } else {
                for (DataNode child : dataNode.children) {
                    node.add(buildTreeModel(child, false));
                }
            }
        }
        
        if (parent != null) {
            getTreeModel().nodeStructureChanged(parent);
        }
        return node;
    }
    
    private DefaultMutableTreeNode createLeavesNode(DataNode node) {
        TableModelLeaves model = new TableModelLeaves(node);
        for (int i=0; i < node.leaves.size(); i++) {
            DataNodeLeaf leaf = node.leaves.get(i);
            leaf.id = i;
            leaf.model = model;
        }
        DefaultMutableTreeNode treeModelNode = new DefaultMutableTreeNode(new TableHolder(model));
        model.jNode = treeModelNode;
        return treeModelNode;
    }
    
    private DefaultMutableTreeNode createDeepNode(DataNode node) {
        TableModelDeep model = new TableModelDeep(node);
        node.appendLeaves(new ArrayList<>()).forEach(leaf -> leaf.model = model);
        DefaultMutableTreeNode treeModelNode = new DefaultMutableTreeNode(new TableHolder(model));
        model.jNode = treeModelNode;
        return treeModelNode;
    }
    
    private DefaultTreeModel getTreeModel() {
        return (DefaultTreeModel) tree.getModel();
    }
// </editor-fold>
}
