package org.lsst.ccs.subsystem.monitor.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.monitor.data.MonMainState;

/**
 ******************************************************************************
 **
 **  Implements the generic monitoring panel.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class MonMainPanel extends JPanel {

    final static Color RED = new Color(150, 0, 0), GREEN = new Color(0, 150, 0);
    final private CommandSender gui;
    private int nSubsys; 
    private String[] name;
    private double[] period;
    private JLabel[] lblStateValue;
    private JTextField[] tfPeriod;


    public MonMainPanel(CommandSender gui) {

        this.gui = gui;
    }

        
    public void setSubsystems(String[] names) {

        if (nSubsys != names.length) {
            nSubsys = names.length;
            name = new String[nSubsys];
            System.arraycopy(names, 0, name, 0, nSubsys);
            period = new double[nSubsys];
            lblStateValue = new JLabel[nSubsys];
            tfPeriod = new JTextField[nSubsys];
        }
        initComponents();
        for (int j = 0; j < nSubsys; j++) {
            (new UpdateStatus(j, new MonMainState(0))).run();
            (new DisableSystem(j)).run();
        }
    }


    private void initComponents() {

        setLayout(new GridBagLayout());
        GridBagConstraints gc = new GridBagConstraints();
        gc.insets = new Insets(5, 5, 5, 5);
        gc.gridx = 0;
        gc.gridy = 0;
        Dimension dimState = new Dimension(0, 0);
        JLabel[] lblState = new JLabel[nSubsys];
        for (int j = 0; j < nSubsys; j++) {
            JPanel pnlSubsys = new JPanel();
            JPanel pnlState = new JPanel();
            JPanel pnlPeriod = new JPanel();

            JLabel label = new JLabel(name[j] + " state: ");
            Dimension ds = label.getPreferredSize();
            dimState.height = ds.height;
            dimState.width = Math.max(dimState.width, ds.width);
            lblState[j] = label;
            pnlState.add(label);

            label = new JLabel("...");
            Dimension dsv = label.getPreferredSize();
            dsv.width = 70;
            label.setPreferredSize(dsv);
            label.setMinimumSize(dsv);
            lblStateValue[j] = label;
            pnlState.add(label);

            pnlPeriod.add(new JLabel("Update Period: "));
            JTextField textf = new JTextField();
            Dimension dp = textf.getPreferredSize();
            dp.width = 50;
            textf.setHorizontalAlignment(JTextField.CENTER);
            textf.setMinimumSize(dp);
            textf.setPreferredSize(dp);
            textf.setName(String.valueOf(j));
            textf.addActionListener(new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    tfPeriodAction(evt);
                }
            });
            tfPeriod[j] = textf;
            pnlPeriod.add(textf);
            pnlPeriod.add(new JLabel("secs"));

            pnlSubsys.setLayout(new GridBagLayout());
            GridBagConstraints gbc = new GridBagConstraints();
            gbc.insets = new Insets(5, 5, 5, 5);
            gbc.gridy = 0;
            gbc.anchor = GridBagConstraints.WEST;
            gbc.gridx = 0;
            pnlSubsys.add(pnlState, gbc);
            gbc.anchor = GridBagConstraints.EAST;
            gbc.gridx = 1;
            pnlSubsys.add(pnlPeriod, gbc);

            add(pnlSubsys, gc);
            gc.gridy++;
        }
        for (int j = 0; j < nSubsys; j++) {
            lblState[j].setPreferredSize(dimState);
            lblState[j].setMinimumSize(dimState);
        }
    }


    private void tfPeriodAction(ActionEvent evt) {
        int id = Integer.valueOf(((JTextField)evt.getSource()).getName());
        try {
            double value = Double.valueOf(tfPeriod[id].getText());
            gui.sendCommand(name[id], null, "setTickMillis", (int)(1000 * value));
            tfPeriod[id].setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tfPeriod[id].setText(String.valueOf(period[id]));
        }
    }


    public void updateControlPanel(int id, MonMainState ms) {
        SwingUtilities.invokeLater(new UpdateStatus(id, ms));
    }


    public void disableSystem(int id) {
        SwingUtilities.invokeLater(new DisableSystem(id));
    }


    private class UpdateStatus implements Runnable {

        private final int id;
        private final MonMainState ms;

        UpdateStatus(int id, MonMainState ms) {
            this.id = id;
            this.ms = ms;
        }

        @Override
        public void run() {
            lblStateValue[id].setText("RUNNING");
            lblStateValue[id].setForeground(GREEN);
            period[id] = ms.getTickMillis() / 1000.0;
            tfPeriod[id].setEnabled(true);
            tfPeriod[id].setText(String.valueOf(period[id]));
        }

    }


    private class DisableSystem implements Runnable {

        private final int id;

        DisableSystem(int id) {
            this.id = id;
        }

        @Override
        public void run() {
            lblStateValue[id].setText("STOPPED");
            lblStateValue[id].setForeground(RED);
            tfPeriod[id].setEnabled(false);
        } 
    }

    private static final long serialVersionUID = 1L;
}
