package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.Color;
import java.util.EnumSet;
import org.lsst.ccs.bus.messages.BusMessage;

/**
 * A bus message that has been processed by tracer filters.
 * Provides access to the original {@link BusMessage} object and additional flags
 * and properties added by filters.
 *
 * @author onoprien
 */
public interface FilteredMessage {
    
    /**
     * Flags that can trigger various actions by the tracer and/or the graphical console.
     * The exact action may depend on the console configuration and on the message properties.
     */
    enum Flag {
        /** Ignore messages until WAKE is received. */ SLEEP,
        /** Stop ignoring messages. */ WАКЕ,
        /** Stop scrolling. */ PAUSE,
        /** Resume scrolling. */ RESUME,
        /** Request user attention. */ ATTENTION,
        /** Put the console in WARNING state. */ WARNING,
        /** Put the console in ALARM state. */ ALARM,
        /** Clear the console alert state. */ CLEAR,
        /** Submit local WARNING to the graphical console alert service. */ LOCAL_WARNING,
        /** Submit local ALARM to the graphical console alert service. */ LOCAL_ALARM,
        /** Clear the local alert state. */ LOCAL_CLEAR,
        /** Mark the message. */ MARK
    }
    
    /**
     * Returns the wrapped bus message, usually the original message received over the buses.
     * @return The wrapped bus message object.
     */
    BusMessage getBusMessage();
    
    /**
     * Sets the wrapped bus message.
     * @param busMessage the bus message object to be wrapped by this filtered message.
     * @return the bus message object previously wrapped by this filtered message.
     */
    BusMessage setBusMessage(BusMessage busMessage);
    
    /**
     * Returns the text message associated with this filtered message.
     * @return The text message associated with this filtered message, or {@code null} if there is no associated message.
     * @deprecated Use {@code getText()} instead.
     */
    @Deprecated
    default String getMessage() {
        return getText();
    }
    
    /**
     * Associates a text message with this filtered message.
     * @param message The text message to be associated.
     * @return The previously associated message, or {@code null} if there was no associated message.
     * @deprecated Use {@code setText(...)} instead.
     */
    @Deprecated
    default String setMessage(String message) {
        return setText(message);
    }
    
    /**
     * Returns the text message associated with this filtered message.
     * @return The text message associated with this filtered message, or {@code null} if there is no associated message.
     */
    default String getText() {
        return null;
    }
    
    /**
     * Associates a text message with this filtered message.
     * @param message The text message to be associated.
     * @return The previously associated message, or {@code null} if there was no associated message.
     */
    default String setText(String message) {
        return null;
    }
    
    /**
     * Returns the flags associated with this message.
     * @return Set of flags associated with this message, or {@code null} if there are no flags.
     */
    default EnumSet<Flag> getFlags() {
        return null;
    }
    
    /**
     * Associates one or more flags with this message.
     * @param action The first flag.
     * @param actions More flags, if any.
     */
    default void addFlag(Flag action, Flag... actions) {
    }
    
    /**
     * Removes all flags from this message.
     */
    default void clearFlags() {
    }
    
    /**
     * Returns the color to be used for printing this message.
     * @return the color associated with this message, or {@code null} if there is no associated color.
     */
    default Color getColor() {
        return null;
    }
    
    /**
     * Sets the color to be used for printing this message.
     * @param color the color associated with this message.
     * @return the color previously associated with this message, or {@code null} if there was no associated color.
     */
    default Color setColor(Color color) {
        return null;
    }
    
    /**
     * Returns the property of this message identified by the specified key.
     * @param key The key.
     * @return The property identified by the specified key, or {@code null} if there is no such property.
     */
    default Object getProperty(String key) {
        return null;
    }

    /**
     * Sets the property of this message identified by the specified key.
     * @param key The key.
     * @param value The value to be set.
     * @return The previous value, or {@code null} if there was no such property.
     */
    default Object setProperty(String key, Object value) {
        return null;
    }
    
}
