package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.util.*;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.event.TreeModelListener;
import javax.swing.table.JTableHeader;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.treetable.TreeTableModel;

/**
 * Table of monitoring data that uses a tree with collapsible nodes to display channel names.
 * <p>
 * <font color=red>
 * While this class has been tested to function correctly in many scenarios, it relies on a hacky {@code JXTreeTable}.
 * I ran into problems trying to use it with multiple views, for example. For now, do not use it directly.
 * </font>
 *
 * @author onoprien
 */
public class TreeTable extends MonitorTable implements TreeTableModel {

// -- Fields : -----------------------------------------------------------------
    
    private final List<MonitorField> columns;
    private final DefaultTreeModel treeModel;
    private final Node root = new Node("");
    
    private boolean isInitialized = false;

// -- Life cycle : -------------------------------------------------------------
    
    public TreeTable(Map<String,ChannelHandle> channels, List<MonitorField> fields) {
        
        columns = trimAbsentFields(fields, channels.values(), null);
        
        // Build tree
        
        for (Map.Entry<String,ChannelHandle> e : channels.entrySet()) {
            String[] path = e.getKey().split("/+");
            Node node = root;
            for (String nodeName : e.getKey().split("/+")) {
                node = addChild(node, nodeName);
            }
            node.setChannelHandle(e.getValue());
        }
        treeModel = new DefaultTreeModel(root);
        
        // Assign row numbers to nodes
        
        nRows = setRowNumbers(-1, root);
        
        // Fill table cells
        
        nColumns = columns.size()+1;
        cells = new ArrayList<>(nRows);
        for (int i=0; i<nRows; i++) {
            cells.add(new ArrayList<>(nColumns));
        }
        
        
        Enumeration<TreeNode> nodes = root.depthFirstEnumeration();
        while (nodes.hasMoreElements()) {
            Node node = (Node) nodes.nextElement();
            if (node == root) break;
            int row = node.getRow();
            List<ChannelHandle> handles;
            if (node.isLeaf()) {
                handles = Collections.singletonList(node.getChannelHandle());
            } else {
                handles = new ArrayList<>();
                Enumeration<TreeNode> nn = node.depthFirstEnumeration();
                while (nn.hasMoreElements()) {
                    Node n = (Node) nn.nextElement();
                    if (n.isLeaf()) {
                        handles.add(n.getChannelHandle());
                    }
                }
                ((ArrayList)handles).trimToSize();
            }
            cells.get(row).add(new MonitorCell(Collections.emptyList(), MonitorField.NULL, new FormattedValue(node.toString(), SwingConstants.LEFT)));
            for (MonitorField field : columns) {
                MonitorCell cell = new MonitorCell(handles, field);
                format.format(cell);
                cells.get(row).add(cell);
            }
        }
        
    }
    
    private Node addChild(Node parent, String name) {
        Enumeration<TreeNode> e = parent.children();
        Node child = null;
        while (e.hasMoreElements()) {
            Node n = (Node) e.nextElement();
            if (name.equals(n.toString())) {
                child = n;
                break;
            }
        }
        if (child == null) {
            child = new Node(name);
            parent.add(child);
        }
        return child;
    }
    
    private int setRowNumbers(int start, Node node) {
        node.row = start++;
        Enumeration<TreeNode> e = node.children();
        while (e.hasMoreElements()) {
            start = setRowNumbers(start, (Node) e.nextElement());
        }
        return start;
    }
    
    public void init() {
        if (isInitialized) return;
        Enumeration<TreeNode> nodes = root.depthFirstEnumeration();
        while (nodes.hasMoreElements()) {
            Node node = (Node) nodes.nextElement();
            ChannelHandle ch = node.getChannelHandle();
            if (ch != null) ch.setTarget(node);
        }
        isInitialized = true;
    }
    
    @Override
    public void destroy() {
        if (!isInitialized) return;
        Enumeration<TreeNode> nodes = root.depthFirstEnumeration();
        while (nodes.hasMoreElements()) {
            Node node = (Node) nodes.nextElement();
            ChannelHandle ch = node.getChannelHandle();
            if (ch != null) ch.setTarget(null);
        }
        isInitialized = false;
    }

    @Override
    public JTable getTable() {
        JTable table = super.getTable();
        JTableHeader header = table.getTableHeader();
        header.setResizingAllowed(true);
        return table;
    }

    @Override
    public JTable makeTable() {
        init();
        return new Table(this);
    }

    
// -- Setters : ----------------------------------------------------------------

// -- Implements TreeTableModel : ----------------------------------------------
    
    @Override
    public String getColumnName(int column) {
        return column == 0 ? "Channel" : columns.get(column-1).getTitle();
    }

    @Override
    public int getHierarchicalColumn() {
        return 0;
    }

    @Override
    public Object getValueAt(Object node, int column) {
        return column == 0 ? node : getValueAt(((Node)node).row, column);
    }

    @Override
    public boolean isCellEditable(Object node, int column) {
        return column == 0 ? false : isCellEditable(((Node)node).getRow(), column);
    }

    @Override
    public void setValueAt(Object value, Object node, int column) {
        if (column != 0) {
            setValueAt(value, ((Node)node).row, column);
        }
    }


// -- Delegate TreeModel to treeModel : ----------------------------------------

    @Override
    public Object getRoot() {
        return treeModel.getRoot();
    }

    @Override
    public Object getChild(Object parent, int index) {
        return treeModel.getChild(parent, index);
    }

    @Override
    public int getChildCount(Object parent) {
        return treeModel.getChildCount(parent);
    }

    @Override
    public boolean isLeaf(Object node) {
        return treeModel.isLeaf(node);
    }

    @Override
    public void valueForPathChanged(TreePath path, Object newValue) {
        treeModel.valueForPathChanged(path, newValue);
    }

    @Override
    public int getIndexOfChild(Object parent, Object child) {
        return treeModel.getIndexOfChild(parent, child);
    }

    @Override
    public void addTreeModelListener(TreeModelListener l) {
        treeModel.addTreeModelListener(l);
    }

    @Override
    public void removeTreeModelListener(TreeModelListener l) {
        treeModel.removeTreeModelListener(l);
    }
    
    
// -- Tree of channels : -------------------------------------------------------
    

    private class Node extends DefaultMutableTreeNode implements Updatable {
        
        int row;
        ChannelHandle channelHandle;
        
        Node(String name) {
            super(name);
        }

        public int getRow() {
            return row;
        }

        public void setRow(int row) {
            this.row = row;
        }

        public ChannelHandle getChannelHandle() {
            return channelHandle;
        }

        public void setChannelHandle(ChannelHandle channelHandle) {
            this.channelHandle = channelHandle;
        }

        @Override
        public List<MonitorField> getAffectedFields(List<String> attributes) {
            return MonitorField.getAffectedFields(attributes, columns);
        }

        @Override
        public void update(ChannelHandle channelHandle, List<MonitorField> fields) {
            int topChangedRow = Integer.MAX_VALUE;
            for (MonitorField field : fields) {
                int col = columns.indexOf(field);
                if (col == -1) break;
                Node node = this;
                while (node != root && format.format(cells.get(node.row).get(col+1))) {
                    if (node.row < topChangedRow) topChangedRow = row;
                    node = (Node) node.getParent();
                }
            }
            if (topChangedRow < Integer.MAX_VALUE) {
                TreeTable.this.fireTableRowsUpdated(topChangedRow, row);
                fireChangeEvent(CELLS_EVENT);
            }
        }

        @Override
        public void update(ChannelHandle channelHandle) {
            update(channelHandle, columns);
        }
        
    }
    
    
// -- Customized JXTreeTable (hack around hack around DefaultTableCellRenderer "color memory") : ---
    
    private static class Table extends JXTreeTable {
        Table(TreeTable model) {
            super(model);
        }
        @Override
        protected void resetDefaultTableCellRendererColors(Component renderer, int row, int column) {
        }
    }


// -- Saving/restoring : -------------------------------------------------------
    
//    public static class Descriptor implements Serializable {
//
//        private String expansion;
//
//        public String getExpansion() {
//            return expansion;
//        }
//
//        public void setExpansion(String expansion) {
//            this.expansion = expansion;
//        }
//        
//    }
//    
//    public Descriptor save() {
//        
//    }
//    
//    public void restore(Descriptor desc) {
//        
//    }
    

}
