package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Dimension;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.gconsole.agent.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;

/**
 * Monitoring data view that displays basic info for the monitored subsystems and provides monitoring controls.
 * For each subsystem, the panel displays:<ul>
 * <li>Time stamp of the last update.
 * <li>Monitoring frequency field.
 * <li>Subsystem name is color-coded to reflect configuration state.
 * <li>Save and reset buttons.
 * </ul>
 *
 * @author onoprien
 */
@ConsoleLookup(id="org.lsst.ccs.gconsole.plugins.monitor.MonitorView",
               name="Configuration View",
               path="Built-In/Configuration",
               description="Monitoring data view that displays configuration state of currently connected subsystems. The view provides controls for saving and resetting configuration, as well as for changing monitoring frequency.")
public class ConfigView extends AbstractMonitorView1 {

// -- Fields : -----------------------------------------------------------------
    
    static private final String STATE_PATH = "/state/ConfigurationState";
    static private final String TIME_PATH = "/runtimeInfo/current time";
    
    private List<String> agentList;
    
    private final AgentDataALL ALL;
    private final TreeMap<String,AgentData> data = new TreeMap<>();
    private AgentData[] model;
    
    private final GUI panel;

// -- Life cycle : -------------------------------------------------------------
    
    public ConfigView() {
        
        ALL = new AgentDataALL();
        
        model = new AgentData[] {ALL};
        
        filter = new AbstractChannelsFilter() {
            @Override
            public List<String> getAgents() {
                return agentList;
            }
            @Override
            public List<String> getOriginChannels() {
                return Arrays.asList(STATE_PATH, TIME_PATH);
            }
        };
        
        panel = new GUI();    
    }
    
    
// -- Implementing MonitorView : -----------------------------------------------

    @Override
    public JComponent getPanel() {
        return panel;
    }

    @Override
    public AgentChannelsFilter getFilter() {
        return filter;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        if (filter == null) {
            agentList = null;
        } else {
            List<String> aa = filter.getAgents();
            agentList = aa == null ? null : new ArrayList<>(aa);
        }
    }

// -- Updating : ---------------------------------------------------------------
    
    @Override
    protected void addChannels(AgentInfo agent, Map<String, AgentChannel> channels) {
        String agentName = agent.getName();
        AgentData ad = data.get(agentName);
        if (ad == null) {
            ad = new AgentData();
            data.put(agentName, ad);
        }
        for (Map.Entry<String, AgentChannel> e : channels.entrySet()) {
            if (e.getKey().endsWith(STATE_PATH)) {
                ad.configState = e.getValue();
            } else if (e.getKey().endsWith(TIME_PATH)) {
                ad.time = e.getValue();
            }
        }
        ALL.time = ad.getTime();
        updateModel();
        panel.reset();
    }

    @Override
    protected void removeChannels(AgentInfo agent, List<String> paths) {
        String agentName = agent.getName();
        AgentData ad = data.remove(agentName);
        updateModel();
        if (ad != null) panel.reset();
    }

    @Override
    protected void updateChannels(AgentInfo agent, Map<String, Map.Entry<AgentChannel, List<String>>> channels) {
        String agentName = agent.getName();
        AgentData ad = data.get(agentName);
        if (ad != null) {
             boolean stateChanged = channels.containsKey(agentName + STATE_PATH);
             if (stateChanged) {
                 ALL.state = ConfigurationState.CONFIGURED.toString();
                 for (int i=1; i<model.length; i++) {
                     ALL.state = combineStates(ALL.state, model[i].getState());
                 }
             }
             ALL.time = ad.getTime();
             panel.update(ad, stateChanged);
        }
    }

    
// -- GUI panel class : --------------------------------------------------------
    
    private final class GUI extends JPanel {
        
        private final JComboBox<AgentData> agentCombo;
        private final JLabel timeLabel;
//        private final JFormattedTextField freqField;
        private final JButton saveButton;
        private final JButton resetButton;
        
        GUI() {
            setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            add(Box.createVerticalGlue());
            Box box = Box.createHorizontalBox();
            add(box);
            box.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
            
            agentCombo = new JComboBox(model);
            agentCombo.setMaximumSize(new Dimension(Integer.MAX_VALUE, agentCombo.getPreferredSize().height));
            agentCombo.addActionListener(e -> {
                AgentData ad = (AgentData) agentCombo.getSelectedItem();
                if (ad == null) return;
                onSelection(ad);
            });
            box.add(agentCombo);
            box.add(Box.createRigidArea(Const.HDIM));
            
            box.add(new JLabel("Last updated: "));
            timeLabel = new JLabel("Never");
            box.add(timeLabel);
//            box.add(Box.createRigidArea(Const.HDIM));
//            
//            box.add(new JLabel("Check every "));
//            freqField = new JFormattedTextField(new DecimalFormat("##0") );
//            freqField.setColumns(4);
//            freqField.setMaximumSize(freqField.getPreferredSize());
//            freqField.setMinimumSize(freqField.getPreferredSize());
//            box.add(freqField);
//            box.add(new JLabel(" seconds."));
            box.add(Box.createRigidArea(new Dimension(3*Const.HSPACE, Const.VSPACE)));
            box.add(Box.createHorizontalGlue());
            
            saveButton = new JButton("Save");
            saveButton.addActionListener(e -> {
                AgentData selectedAgent = (AgentData) agentCombo.getSelectedItem();
                if (selectedAgent != null && selectedAgent != ALL && ConfigurationState.DIRTY.toString().equals(selectedAgent.getState())) {
//                    Console.getConsole().sendCommand(selectedAgent.name +"/saveLimits");
                    Console.getConsole().sendCommand(selectedAgent.getName() +"/saveAllChanges");
                }
            });
            box.add(saveButton);
            box.add(Box.createRigidArea(Const.HDIM));
            resetButton = new JButton("Reset");
            resetButton.addActionListener(e -> {
                AgentData selectedAgent = (AgentData) agentCombo.getSelectedItem();
                if (selectedAgent != null && selectedAgent != ALL && ConfigurationState.DIRTY.toString().equals(selectedAgent.getState())) {
//                    Console.getConsole().sendCommand(selectedAgent.name +"/dropLimitChanges");
                    Console.getConsole().sendCommand(selectedAgent.getName() +"/dropAllChanges");
                }
            });
            box.add(resetButton);
            
            box.setMaximumSize(new Dimension(Integer.MAX_VALUE, box.getPreferredSize().height));
            add(box);

        }
        
        void reset() {
            AgentData selected = (AgentData) agentCombo.getSelectedItem();
            if (selected == null) selected = ALL;
            agentCombo.removeAllItems();
            agentCombo.setModel(new DefaultComboBoxModel<>(model) );
            agentCombo.setSelectedItem(selected);
        }
        
        void update(AgentData item, boolean stateChanged) {
            AgentData selected = (AgentData) agentCombo.getSelectedItem();
            if (selected == item || selected == ALL) {
                if (stateChanged) {
                    onSelection(selected);
                    agentCombo.repaint();
                } else {
                    timeLabel.setText(formatTime(item.getTime()));
                }
            }
        }
        
        private void onSelection(AgentData item) {
            timeLabel.setText(formatTime(item.getTime()));
            Boolean b = ConfigurationState.DIRTY.toString().equals(item.getState());
            saveButton.setEnabled(b && item != ALL);
            resetButton.setEnabled(b && item != ALL);
        }
        
    }

// -- Local methods : ----------------------------------------------------------
    
    private String combineStates(String state1, String state2) {
        String dirty = ConfigurationState.DIRTY.toString();
        return dirty.equals(state1) || dirty.equals(state2) ? dirty : ConfigurationState.CONFIGURED.toString();
    }
    
    private void updateModel() {
        model = new AgentData[data.size() + 1];
        model[0] = ALL;
        int i = 1;
        for (AgentData item : data.values()) {
            model[i++] = item;
        }
    }
    
    private String formatTime(Instant time) {
        try {
            return String.format("%1$tF %1$tT.", ZonedDateTime.ofInstant(time, ZoneId.systemDefault()));
        } catch (IllegalFormatException | DateTimeException | NullPointerException x) {
            return "unknown.";
        }
    }

    
// -- Local classes : ----------------------------------------------------------
    
    private class AgentData {
        
        AgentChannel configState;
        AgentChannel time;
        
        String getState() {
            if (configState == null) {
                return ConfigurationState.CONFIGURED.toString();
            } else {
                Object v = configState.get();
                return v == null ? ConfigurationState.CONFIGURED.toString() : v.toString();
            }
        }
        
        Instant getTime() {
            return time == null ? null : time.get();
        }
        
        String getName() {
            return time == null ? configState.getAgentName() : time.getAgentName();
        }

        @Override
        public String toString() {
            return ConfigurationState.DIRTY.toString().equals(getState()) ? "<html><font color=\"blue\">"+ getName() : getName();
        }
        
    }
    
    private class AgentDataALL extends AgentData {
        
        String state = ConfigurationState.CONFIGURED.toString();
        Instant time;

        @Override
        String getName() {
            return "ALL";
        }

        @Override
        Instant getTime() {
            return time;
        }

        @Override
        String getState() {
            return state;
        }
        
    }
    
}
