package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;
import org.lsst.ccs.gconsole.agent.AgentStatusListener;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.util.ThreadUtil;


/**
 * LSST Command Browser Plugin.
 * The original version of the command dictionary tool was written by Etienne Marin-Matholaz.
 * 
 * @author onoprien
 */
@Plugin(name="LSST Commands Browser Plugin",
        id="command",
        description="LSST Commands Browser allows listing available commands for a subsystem and executing them.",
        shortDescription="LSST Remote Subsystem Commands Browser")
public class LsstCommandBrowserPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    static public String[] DEFAULT_TYPES = new String[] {"WORKER", "SERVICE", "MCM", "OCS_BRIDGE", "LOCK_MANAGER"};
            
    private final ArrayList<Browser> browsers = new ArrayList<>(0);
    
    private final AgentStatusListener agentConnectionListener;

// -- Life cycle : -------------------------------------------------------------
    
    public LsstCommandBrowserPlugin() {
        
        agentConnectionListener = new AgentStatusListener() {
            
            @Override
            public void connect(AgentStatusEvent event) {
                AgentInfo agent = event.getSource();
                AgentInfo.AgentType type = agent.getType();
                if (!(AgentInfo.AgentType.CONSOLE.equals(type) || AgentInfo.AgentType.LISTENER.equals(type))) {
                    Action action = new AbstractAction(agent.getName()) {
                        @Override
                        public void actionPerformed(ActionEvent e) {
                            String agentName = agent.getName();
                            createBrowser("Commands:" + agentName, agentName);
                        }
                    };
                    Console.getConsole().addMenu(action, null, "400: CCS Tools :-1:5", "Command Browser:100:100", "Subsystems:1");
                }
            }

            @Override
            public void disconnect(AgentStatusEvent event) {
                AgentInfo agent = event.getSource();
                Console.getConsole().removeMenu(" CCS Tools ", "Command Browser", "Subsystems", agent.getName());
            }
            
        };
    }

    @Override
    public void initialize() {
        Action newDictionaryAction = new AbstractAction("Browse...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                BrowserFull.Descriptor descriptor = new BrowserFull.Descriptor();
                descriptor.setName("Commands");
                descriptor.setFilterTypes(DEFAULT_TYPES);
                LsstCommandBrowserPlugin.this.createBrowser(descriptor);
            }
        };
        getServices().addMenu(newDictionaryAction, "400: CCS Tools :-1:5", "Command Browser:0");
    }

    @Override
    public void start() {
        Console.getConsole().getStatusAggregator().addListener(agentConnectionListener, null, Collections.emptyList());
    }

    @Override
    public void stop() {
        Console.getConsole().getStatusAggregator().removeListener(agentConnectionListener);
    }

    
// -- Creating browsers : ------------------------------------------------------

    /**
     * Creates and displays a generic (unfiltered, default settings) command browser page.
     * May be called on any thread.
     * 
     * @param name Page title.
     * @param subsystem Subsystem name, or {@code null} if the page should be able to display commands for all subsystems.
     */
    public void createBrowser(String name, String subsystem) {
        Browser.Descriptor descriptor;
        if (subsystem == null) {
            descriptor = new BrowserFull.Descriptor();
        } else {
            descriptor = new BrowserOneSubsystem.Descriptor();
            descriptor.setAgent(subsystem);
        }
        descriptor.setName(name);
        createBrowser(descriptor);
    }
    
    /**
     * Creates and displays a command dictionary page customized by the provided descriptor.
     * May be called on any thread.
     * 
     * @param descriptor Descriptor of the page to be created.
     */
    public void createBrowser(Serializable descriptor) {
        ThreadUtil.invokeLater(() -> {
            Browser browser;
            if (descriptor instanceof BrowserFull.Descriptor) {
                browser = new BrowserFull();
            } else if (descriptor instanceof BrowserOneSubsystem.Descriptor) {
                browser = new BrowserOneSubsystem();
            } else {
                getConsole().getLogger().error("Unknown type of command browser descriptor: "+ descriptor);
                return;
            }
            browser.restore(descriptor);
            showBrowser(browser);
        });        
    }
    
    /**
     * Registers and displays a command browser page.
     * Should be called on EDT.
     * 
     * @param browser Dictionary to be displayed.
     */
    public void showBrowser(Browser browser) {
        Map<Object, Object> par = new HashMap<>();
        par.put(Panel.TITLE, browser.getName());
        Consumer<JComponent> onClose = c -> {
            Iterator<Browser> it = browsers.iterator();
            while(it.hasNext()) {
                Browser b = it.next();
                if (c == b.getPanel()) {
                    b.shutdown();
                    it.remove();
                    break;
                }
            }
            browsers.trimToSize();
        };
        par.put(Panel.ON_CLOSE, onClose);
        getConsole().getPanelManager().open(browser.getPanel(), par);
        browsers.add(browser);
        browsers.trimToSize();        
    }
    
    
// -- Saving/restoring : -------------------------------------------------------

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        if (! (storageBean instanceof Descriptor)) return true;
        Descriptor desc = (Descriptor) storageBean;
        if (!browsers.isEmpty()) {
            ArrayList<Browser> copy = new ArrayList<>(browsers);
            for (Browser d : copy) {
                JComponent panel = d.getPanel();
                if (panel != null) {
                    getConsole().getPanelManager().close(panel);
                }
            }
            browsers.clear();
        }
        Serializable[] dd = desc.getBrowsers();
        if (dd != null) {
            for (Serializable d : dd) {
                createBrowser(d);
            }
        }
        return true;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        if (!browsers.isEmpty()) {
            Serializable[] descriptors = new Serializable[browsers.size()];
            for (int i = 0; i < descriptors.length; i++) {
                descriptors[i] = browsers.get(i).save();
            }
            desc.setBrowsers(descriptors);
        }
        return desc;
    }

    static public class Descriptor extends ComponentDescriptor {

        private Serializable[] browsers;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public Serializable[] getBrowsers() {
            return browsers;
        }

        public void setBrowsers(Serializable[] browsers) {
            this.browsers = browsers;
        }
        
    }




    
}
