package org.lsst.ccs.subsystem.vacuum.constants;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.subsystem.vacuum.data.VacuumException;

/**
 *  Definitions for the vacuum system switches.
 *
 *  @author The CCS team
 */
public class Switches {

    public static final int
        SW_CRYO_TURBO_PUMP  = 0,
        SW_CRYO_ION_PUMP1   = 1,
        SW_CRYO_ION_PUMP2   = 2,
        SW_CRYO_ION_PUMP3   = 3,
        SW_CRYO_ION_PUMP4   = 4,
        SW_CRYO_ION_PUMP5   = 5,
        SW_CRYO_ION_PUMP6   = 6,
        SW_CRYO_GATE_VALVE  = 7,
        SW_CRYO_SCROLL_PUMP = 8,
        SW_HX_TURBO_PUMP    = 9,
        SW_HX_ION_PUMP1     = 10,
        SW_HX_ION_PUMP2     = 11,
        SW_HX_GATE_VALVE    = 12,
        SW_HX_SCROLL_PUMP   = 13,
        SW_INST_FTPP_VALVE  = 14,
        SW_INST_FTH_VALVE   = 15,
        SW_INST_L3H_VALVE   = 16,
        SW_INST_L3LF_VALVE  = 17,
        SW_INST_SCROLL_PUMP = 18,
        SW_CRYO_VAC_CC      = 19,
        SW_CRYO_TURBO_CC    = 20,
        SW_HX_VAC_CC        = 21,
        SW_HX_TURBO_CC      = 22,
        SW_INST_FLINE_VALVE = 23,
        SW_CRYO_FLINE_VALVE = 24,
        SW_HX_FLINE_VALVE   = 25,
        SW_CRYO_TURBO_VENT_VALVE  = 26,
        SW_HX_TURBO_VENT_VALVE    = 27,
        NUM_SWITCHES        = 28;


    private static final Map<String, Integer> nameMap = new LinkedHashMap<>();
    static {
        nameMap.put("CryoTurboPump", SW_CRYO_TURBO_PUMP);
        nameMap.put("CryoIonPump1", SW_CRYO_ION_PUMP1);
        nameMap.put("CryoIonPump2", SW_CRYO_ION_PUMP2);
        nameMap.put("CryoIonPump3", SW_CRYO_ION_PUMP3);
        nameMap.put("CryoIonPump4", SW_CRYO_ION_PUMP4);
        nameMap.put("CryoIonPump5", SW_CRYO_ION_PUMP5);
        nameMap.put("CryoIonPump6", SW_CRYO_ION_PUMP6);
        nameMap.put("CryoGateValve", SW_CRYO_GATE_VALVE);
        nameMap.put("CryoScrollPump", SW_CRYO_SCROLL_PUMP);
        nameMap.put("CryoFlineValve", SW_CRYO_FLINE_VALVE);
        nameMap.put("HxTurboPump", SW_HX_TURBO_PUMP);
        nameMap.put("HxIonPump1", SW_HX_ION_PUMP1);
        nameMap.put("HxIonPump2", SW_HX_ION_PUMP2);
        nameMap.put("HxGateValve", SW_HX_GATE_VALVE);
        nameMap.put("HxScrollPump", SW_HX_SCROLL_PUMP);
        nameMap.put("HxFlineValve", SW_HX_FLINE_VALVE);
        nameMap.put("InstFppValve", SW_INST_FTPP_VALVE);
        nameMap.put("InstFthValve", SW_INST_FTH_VALVE);
        nameMap.put("InstL3hValve", SW_INST_L3H_VALVE);
        nameMap.put("InstL3lfValve", SW_INST_L3LF_VALVE);
        nameMap.put("InstScrollPump", SW_INST_SCROLL_PUMP);
        nameMap.put("InstFlineValve", SW_INST_FLINE_VALVE);
        nameMap.put("CryoVacCC", SW_CRYO_VAC_CC);
        nameMap.put("CryoTurboCC", SW_CRYO_TURBO_CC);
        nameMap.put("HxVacCC", SW_HX_VAC_CC); 
        nameMap.put("HxTurboCC", SW_HX_TURBO_CC);
        nameMap.put("CryoTurboVentValve", SW_CRYO_TURBO_VENT_VALVE);
        nameMap.put("HxTurboVentValve", SW_HX_TURBO_VENT_VALVE);
    }

    private static final Map<Integer, String> idMap = new LinkedHashMap<>();
    static {
        for (Map.Entry e : nameMap.entrySet()) {
            idMap.put((Integer)e.getValue(), (String)e.getKey());
        }
    }

    private static final int[] deviceList = new int[NUM_SWITCHES];
    static {
        deviceList[SW_CRYO_TURBO_PUMP] = Devices.DEVC_CRYO_TURBO_PUMP;
        deviceList[SW_CRYO_ION_PUMP1] = Devices.DEVC_ION_PUMP;
        deviceList[SW_CRYO_ION_PUMP2] = Devices.DEVC_ION_PUMP;
        deviceList[SW_CRYO_ION_PUMP3] = Devices.DEVC_ION_PUMP;
        deviceList[SW_CRYO_ION_PUMP4] = Devices.DEVC_ION_PUMP;
        deviceList[SW_CRYO_ION_PUMP5] = Devices.DEVC_ION_PUMP;
        deviceList[SW_CRYO_ION_PUMP6] = Devices.DEVC_ION_PUMP;
        deviceList[SW_CRYO_GATE_VALVE] = Devices.DEVC_PLUTO;
        deviceList[SW_CRYO_SCROLL_PUMP] = Devices.DEVC_PDU;
        deviceList[SW_CRYO_FLINE_VALVE] = Devices.DEVC_MAQ20;
        deviceList[SW_HX_TURBO_PUMP] = Devices.DEVC_HX_TURBO_PUMP;
        deviceList[SW_HX_ION_PUMP1] = Devices.DEVC_ION_PUMP;
        deviceList[SW_HX_ION_PUMP2] = Devices.DEVC_ION_PUMP;
        deviceList[SW_HX_GATE_VALVE] = Devices.DEVC_PLUTO;
        deviceList[SW_HX_SCROLL_PUMP] = Devices.DEVC_PDU;
        deviceList[SW_HX_FLINE_VALVE] = Devices.DEVC_MAQ20;
        deviceList[SW_INST_FTPP_VALVE] = Devices.DEVC_MAQ20;
        deviceList[SW_INST_FTH_VALVE] = Devices.DEVC_MAQ20;
        deviceList[SW_INST_L3H_VALVE] = Devices.DEVC_MAQ20;
        deviceList[SW_INST_L3LF_VALVE] = Devices.DEVC_MAQ20;
        deviceList[SW_INST_SCROLL_PUMP] = Devices.DEVC_PDU;
        deviceList[SW_INST_FLINE_VALVE] = Devices.DEVC_MAQ20;
        deviceList[SW_CRYO_VAC_CC] = Devices.DEVC_CRYO_VAC_MKS;
        deviceList[SW_CRYO_TURBO_CC] = Devices.DEVC_CRYO_TURBO_MKS;
        deviceList[SW_HX_VAC_CC] = Devices.DEVC_HX_VAC_MKS;
        deviceList[SW_HX_TURBO_CC] = Devices.DEVC_HX_TURBO_MKS;
        deviceList[SW_CRYO_TURBO_VENT_VALVE] = Devices.DEVC_CRYO_TURBO_PUMP;
        deviceList[SW_HX_TURBO_VENT_VALVE] = Devices.DEVC_HX_TURBO_PUMP;
    }

    /**
     *  Gets the list of switch names.
     * 
     *  @return  The list of names
     */
    public static List<String> getNames()
    {
        return new ArrayList<>(nameMap.keySet());
    }

    /**
     *  Gets a switch name from its ID.
     * 
     *  @param  id  The switch id
     *  @return  The switch name
     */
    public static String getName(int id)
    {
        return idMap.get(id);
    }

    /**
     *  Gets a switch ID from its name.
     * 
     *  @param  name  The switch name
     *  @return  The switch ID
     *  @throws  VacuumException
     */
    public static int getId(String name) throws VacuumException
    {
        Integer id = nameMap.get(name);
        if (id == null) {
            throw new VacuumException("Invalid valve name: " + name);
        }
        return id;
    }

    /**
     *  Gets a device ID for a switch.
     * 
     *  @param  id  The switch ID
     *  @return  The device ID
     */
    public static int getDevice(int id)
    {
        return deviceList[id];
    }

}
