package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestHeaters;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Implements a temperature controller for the volume purge cabinet.
 *
 *  @author Owen Saxton
 */
public class VpcControl implements HasLifecycle {

    private static final String VPC = "Vpc";
    private static final Map<Integer, Integer> heaterChans = new HashMap<>();
    static {
        heaterChans.put(PurgeTestHeaters.HEATER_VPC1_ID, BfrDevice.CHAN_HEATER_1);
        heaterChans.put(PurgeTestHeaters.HEATER_VPC2_ID, BfrDevice.CHAN_HEATER_2);
    }

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskServices;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, Channel> allChannels = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.TREE)
    private BfrDevice bfrDevc;
    
    @ConfigurationParameter(category=VPC, maxLength=PurgeTestHeaters.NUM_VPC_HEATERS, isFinal=true)
    private volatile double[] deltaTemps = new double[0];
    @ConfigurationParameter(category=VPC, isFinal=true)
    private volatile Integer updateTime = Integer.MAX_VALUE; // The update time interval (msec)

    private String[] refTempChans;     // Reference temperature channels to use
    private String[] ctrlTempChans;    // Controltemperature channels to use

    private final List<Channel> refTempChanList = new ArrayList<>();   // Reference temperature channels to use
    private final List<Channel> ctrlTempChanList = new ArrayList<>();  // Control temperature channels to use

    private static final Logger LOG = Logger.getLogger(VpcControl.class.getName());
    private boolean active = false;    // True if loop is active
    private boolean loopFailedRef = false, loopFailedCtrl = false, loopFailedHtr = false;
    private long lastLoopTime;


    /**
     *  Sets up the VPC temperature control loop timer task.
     */
    @Override
    public void build() {
        AgentPeriodicTask apt = new AgentPeriodicTask("vpc-loop-" + name,
                                                      () -> iterateLoop()).withPeriod(Duration.ofMillis(500));
        periodicTaskServices.scheduleAgentPeriodicTask(apt);
    }


    /**
     *  Initializes the parameters.
     */
    @Override
    public void init()
    {
        if (bfrDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "BfrDevice component", "is missing");
        }
        if (refTempChans == null || refTempChans.length == 0) {
            ErrorUtils.reportConfigError(LOG, name, "refTempChans", "is missing or empty");
        }
        else {
            for (String cName : refTempChans) {
                Channel cmpt = allChannels.get(cName);
                if (cmpt != null) {
                    refTempChanList.add(cmpt);
                }
                else {
                    ErrorUtils.reportConfigError(LOG, name, "refTempChans", "contains non-Channel item");
                }
            }
        }
        if (ctrlTempChans == null || ctrlTempChans.length == 0) {
            ErrorUtils.reportConfigError(LOG, name, "ctrlTempChans", "is missing or empty");
        }
        else {
            for (String cName : ctrlTempChans) {
                Channel cmpt = allChannels.get(cName);
                if (cmpt != null) {
                    ctrlTempChanList.add(cmpt);
                }
                else {
                    ErrorUtils.reportConfigError(LOG, name, "ctrlTempChans", "contains non-Channel item");
                }
            }
        }
        if (deltaTemps.length != PurgeTestHeaters.NUM_VPC_HEATERS) {
            ErrorUtils.reportConfigError(LOG, name, "deltaTemps", "must be of length " + PurgeTestHeaters.NUM_VPC_HEATERS);
        }
        if (updateTime == Integer.MAX_VALUE) {
            ErrorUtils.reportConfigError(LOG, name, "updateTime", "is missing");
        }
    }


    /**
     *  Sets a target delta temperature.
     *
     *  @param  heater  The VPC heater ID
     *  @param  temp    The temperature to set
     *  @throws  UtilityException
     */
    public void setDeltaTemp(int heater, double temp) throws UtilityException
    {
        deltaTemps[heater] = temp;
    }


    /**
     *  Gets a target delta temperature.
     *
     *  @param  heater  The VPC heater ID
     *  @return  The set temperature
     */
    public double getDeltaTemp(int heater)
    {
        return deltaTemps[heater];
   }


    /**
     *  Starts the control loop.
     *
     *  @throws  UtilityException
     */
    public void startLoop() throws UtilityException
    {
        active = true;
    }


    /**
     *  Stops the control loop.
     *
     *  @throws  UtilityException
     */
    public void stopLoop() throws UtilityException
    {
        active = false;
    }


    /**
     *  Gets the control loop state.
     *
     *  @return  Whether the control loop is active
     */
    public boolean isLoopActive()
    {
        return active;
    }


    /**
     *  Timer routine for control loop iteration.
     */
    private void iterateLoop()
    {
        if (!active) return;
        long time = System.currentTimeMillis();
        if (time < lastLoopTime + updateTime) return;
        lastLoopTime = time;
        double refTemp = 0.0, ctrlTemp = 0.0;
        int count = 0;
        for (Channel tempChan : refTempChanList) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                refTemp += value;
                count++;
            }
        }
        if (count > 0) {
            refTemp /= count;
            loopFailedRef = false;
        }
        else {
            if (!loopFailedRef) {
                LOG.severe("Control loop iteration failed: no valid reference temperature values available");
                loopFailedRef = true;
            }
            return;
        }
        count = 0;
        for (Channel tempChan : ctrlTempChanList) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                ctrlTemp += value;
                count++;
            }
        }
        if (count > 0) {
            ctrlTemp /= count;
            loopFailedCtrl = false;
        }
        else {
            if (!loopFailedCtrl) {
                LOG.severe("Control loop iteration failed: no valid control temperature values available");
                loopFailedCtrl = true;
            }
            return;
        }
        if (!setHeaters(refTemp - ctrlTemp)) {
            if (!loopFailedHtr) {
                LOG.severe("Control loop iteration failed: heaters are offline");
                loopFailedHtr = true;
            }
            return;
        }
        loopFailedHtr = false;
    }


    /**
     *  Sets the heater state
     *
     *  @param  deltaTemp  The measured temperature difference
     *  @return  Whether heaters are online
     */
    private boolean setHeaters(double deltaTemp)
    {
        for (int htrId = 0; htrId < PurgeTestHeaters.NUM_VPC_HEATERS; htrId++) {
            int htrChan = heaterChans.get(htrId);
            Boolean htrOn = bfrDevc.isSwitchOn(htrChan);
            if (htrOn == null) return false;
            if (deltaTemp > deltaTemps[htrId]) {
                if (!htrOn) {
                    bfrDevc.switchOn(htrChan);
                }
            }
            else if (deltaTemp <= 0.0) {
                if (htrOn) {
                    bfrDevc.switchOff(htrChan);
                }
            }
        }
        return true;
    }

}
