package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.utility.constants.FanState;
import org.lsst.ccs.subsystem.utility.constants.UTTestFans;
import org.lsst.ccs.subsystem.utility.constants.UTTestSwitches;
import org.lsst.ccs.subsystem.utility.constants.UTTestValves;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.constants.UtilityAgentProperties;
import org.lsst.ccs.subsystem.utility.constants.ValveState;
import org.lsst.ccs.subsystem.utility.data.UTTestState;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Implements the utility trunk test subsystem.
 *
 *  @author Owen Saxton
 */
public class UTTestMain extends Subsystem implements HasLifecycle {

    private static final int[] devNums = new int[UTTestSwitches.NUM_SWITCHES];
    static {
        devNums[UTTestSwitches.SW_BFR_24V_PDU] = UTTestSwitches.DEVC_BFR;
        devNums[UTTestSwitches.SW_BFR_48V_PDU] = UTTestSwitches.DEVC_BFR;
        devNums[UTTestSwitches.SW_PDU_24V_MAIN] = UTTestSwitches.DEVC_PDU_24V;
        devNums[UTTestSwitches.SW_PDU_ELEX] = UTTestSwitches.DEVC_PDU_24V;
        devNums[UTTestSwitches.SW_PDU_48V_MAIN] = UTTestSwitches.DEVC_PDU_48V;
        devNums[UTTestSwitches.SW_PDU_UT_FAN] = UTTestSwitches.DEVC_PDU_48V;
    }
    private static final int[] swNums = new int[UTTestSwitches.NUM_SWITCHES];
    static {
        swNums[UTTestSwitches.SW_BFR_24V_PDU] = BfrDevice.CHAN_24V_PDU;
        swNums[UTTestSwitches.SW_BFR_48V_PDU] = BfrDevice.CHAN_48V_PDU;
        swNums[UTTestSwitches.SW_PDU_24V_MAIN] = -1;
        swNums[UTTestSwitches.SW_PDU_ELEX] = Pdu24VDevice.CHAN_ELECTRONICS;
        swNums[UTTestSwitches.SW_PDU_48V_MAIN] = -1;
        swNums[UTTestSwitches.SW_PDU_UT_FAN] = Pdu48VDevice.CHAN_UT_FAN;
    }
    private static final Map<Integer, String> typeMap = new HashMap<>();
    static {
        typeMap.put(UTTestSwitches.DEVC_BFR, "BFR");
        typeMap.put(UTTestSwitches.DEVC_PDU_24V, "24V PDU");
        typeMap.put(UTTestSwitches.DEVC_PDU_48V, "48V PDU");
    }

    private static final Logger LOG = Logger.getLogger(UTTestMain.class.getName());

    @LookupName
    String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<SwitchControl> swDeviceList = new ArrayList<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, FanPIControl> fanDeviceMap = new LinkedHashMap<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private UTMaq20Device utControl;

    private String utFan;

    private final SwitchControl[] swDevices = new SwitchControl[UTTestSwitches.NUM_DEVICES];
    private final UTTestState utTestState = new UTTestState();
    private final FanPIControl[] fanControl = new FanPIControl[UTTestFans.NUM_FANS];


    /**
     *  Constructor.
     */
    public UTTestMain() {
        super("uttest", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create and schedule an AgentPeriodicTask to update and publish the utility trunk system state
        AgentPeriodicTask pt = new AgentPeriodicTask("uttest-state",
                                                     () -> updateState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent's type.
        propertiesService.setAgentProperty(UtilityAgentProperties.UT_TEST_TYPE, getClass().getCanonicalName());

        // Check power devices and control channels
        for (SwitchControl sd : swDeviceList) {
            swDevices[sd.getSwitchDevice()] = sd;
        }
        for (int j = 0; j < swDevices.length; j++) {
            if (swDevices[j] == null) {
                ErrorUtils.reportConfigError(LOG, name, typeMap.get(j), "has not been defined");
            }
        }

        // Check fan controllers
        if (utFan != null) {
            fanControl[UTTestFans.FAN_UT_ID] = fanDeviceMap.get(utFan);
        }
        if (fanControl[UTTestFans.FAN_UT_ID] == null) {
            ErrorUtils.reportConfigError(LOG, name, "utFan", "not specified or not defined");
        }
        if (utControl == null) {
            ErrorUtils.reportConfigError(LOG, name, "UTMaq20Device", "has not been defined");
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        //updateState();
        //publishState();        // For any GUIs
        LOG.info("Utility trunk test subsystem started");
    }


    /**
     *  Gets the subsystem state.
     *
     *  @return  The full PDU system state
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public UTTestState getFullState()
    {
        utTestState.setTickMillis(getTickPeriod());
        return utTestState;
    }


    /**
     *  Gets the list of switch names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of switch names")
    public List<String> getSwitchNames()
    {
        return new ArrayList<>(UTTestSwitches.switchNameToId.keySet());
    }


    /**
     *  Turns a switch on or off (using its name).
     *
     *  @param  name  The switch name.
     *  @param  on  Whether to turn on or off
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setSwitchOn(@Argument(description="The switch name") String name,
                            @Argument(description="Whether to turn on") boolean on) throws UtilityException
    {
        try {
            Integer sw = UTTestSwitches.switchNameToId.get(name);
            if (sw == null) {
                throw new UtilityException("Invalid switch name: " + name);
            }
            synchronized (utTestState) {
                SwitchState state = on ? SwitchState.ON : SwitchState.OFF;
                if (state == utTestState.getSwitchBaseState(sw)) return;
                utTestState.setSwitchBaseState(sw, state);
                if (utTestState.getSwitchState(sw) == SwitchState.OFFLINE) return;
                utTestState.setSwitchState(sw, state);
                setSwitchOn(sw, on);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Gets the list of fan names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of fan names")
    public List<String> getFanNames()
    {
        return new ArrayList<>(UTTestFans.fanNameToId.keySet());
    }


    /**
     *  Sets a fan state (using its name).
     *
     *  @param  name   The fan name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan control state")
    public void setFanState(@Argument(description="The fan name") String name,
                            @Argument(description="The control state") FanState state) throws UtilityException
    {
        try {
            int fanId = getFanId(name);
            if (state == FanState.OFFLINE || (state == FanState.TEMP && !UTTestFans.hasTempState[fanId])) {
                throw new UtilityException("Invalid " + name + " state: " + state);
            }
            synchronized (utTestState) {
                if (state == utTestState.getFanBaseState(fanId)) return;
                utTestState.setFanBaseState(fanId, state);
                if (utTestState.getFanState(fanId) == FanState.OFFLINE) return;
                utTestState.setFanState(fanId, state);
                FanPIControl ctrlr = fanControl[fanId];
                if (state == FanState.TEMP) {
                    ctrlr.setTemperature(utTestState.getDeltaTemp(fanId));
                    ctrlr.startLoop(utTestState.getFanSpeed(fanId));
                }
                else {
                    if (UTTestFans.hasTempState[fanId]) {
                        ctrlr.stopLoop();
                    }
                    if (state == FanState.SPEED) {
                        ctrlr.setSpeed(utTestState.getFanSpeed(fanId));
                    }
                }
                setSwitchOn(UTTestFans.fanSwitchMap.get(fanId), state != FanState.OFF);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a fan speed (using its name).
     *
     *  @param  name   The fan name
     *  @param  speed  The fan speed
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan speed")
    public void setFanSpeed(@Argument(description="The fan name") String name,
                            @Argument(description="The fan speed") double speed) throws UtilityException
    {
        try {
            int fanId = getFanId(name);
            synchronized (utTestState) {
                utTestState.setFanSpeed(fanId, speed);
                if (utTestState.getFanState(fanId) == FanState.SPEED) {
                    fanControl[fanId].setSpeed(speed);
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Gets the list of valve names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of valve names")
    public List<String> getValveNames()
    {
        return new ArrayList<>(UTTestValves.valveNameToId.keySet());
    }


    /**
     *  Sets a valve state (using its name).
     *
     *  @param  name   The valve name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a valve control state")
    public void setValveState(@Argument(description="The valve name") String name,
                              @Argument(description="The control state") ValveState state) throws UtilityException
    {
        try {
            int valveId = getValveId(name);
            if (state == ValveState.OFFLINE || state == ValveState.TEMP) {
                throw new UtilityException("Invalid " + name + " state: " + state);
            }
            synchronized (utTestState) {
                if (state == utTestState.getValveBaseState(valveId)) return;
                utTestState.setValveBaseState(valveId, state);
                ValveState currState = utTestState.getValveState(valveId);
                if (currState == ValveState.OFFLINE || currState == ValveState.TEMP) return;
                utTestState.setValveState(valveId, state);
                operateValve(valveId);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a valve position (using its name).
     *
     *  @param  name   The valve name
     *  @param  position  The valve position
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a valve position")
    public void setValvePosition(@Argument(description="The valve name") String name,
                                 @Argument(description="The valve position") double position) throws UtilityException
    {
        try {
            int valveId = getValveId(name);
            synchronized (utTestState) {
                utTestState.setValvePosition(valveId, position);
                if (utTestState.getValveState(valveId) == ValveState.POSN) {
                    operateValve(valveId);
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a fan delta temperature (using its name).
     *
     *  @param  name  The fan name
     *  @param  temp  The delta temperature to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a delta temperature")
    public void setFanDeltaTemp(@Argument(description="The fan name") String name,
                                @Argument(description="The delta temperature") double temp) throws UtilityException
    {
        try {
            int fanId = getFanId(name);
            synchronized (utTestState) {
                utTestState.setDeltaTemp(fanId, temp);
                if (utTestState.getFanState(fanId) == FanState.TEMP) {
                    fanControl[fanId].setTemperature(temp);
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)") int period)
    {
        setTickPeriod(period);
        //updateState();
        publishState();    // Must do this for GUI
    }


    /**
     *  Turns a switch on or off using its ID.
     *
     *  @param  sw  The switch ID.
     *  @param  on  Whether to turn on or off
     *  @throws  UtilityException
     */
    private void setSwitchOn(int sw, boolean on) throws UtilityException
    {
        if (sw < 0 || sw >= UTTestSwitches.NUM_SWITCHES) {
            throw new UtilityException("Invalid switch number: " + sw);
        }
        if (on) {
            swDevices[devNums[sw]].switchOn(swNums[sw]);
        }
        else {
            swDevices[devNums[sw]].switchOff(swNums[sw]);
        }
        //updateState();
        //publishState();
    }


    /**
     *  Operates a coolant valve.
     *
     *  @param  valveId  The valve ID
     *  @throws  UtilityException
     */
    private void operateValve(int valveId) throws UtilityException
    {
        ValveState state = utTestState.getValveState(valveId);
        if (state == ValveState.OFFLINE || state == ValveState.TEMP) return;
        double position = state == ValveState.SHUT ? 0.0 : utTestState.getValvePosition(valveId);
        if (valveId == UTTestValves.VALVE_UT_ID) {
            utControl.setUtValve(position);
        }
        
    }


    /**
     *  Updates the utility trunk system state when devices go offline or online.
     */
    private void updateState()
    {
        synchronized (utTestState) {
            boolean changed = false;
            for (int sw = 0; sw < UTTestSwitches.NUM_PWR_SWITCHES; sw++) {
                Boolean st = swDevices[devNums[sw]].isSwitchOn(swNums[sw]);
                SwitchState ss = st == null ? SwitchState.OFFLINE : st ? SwitchState.ON : SwitchState.OFF;
                if (ss != utTestState.getSwitchState(sw)) {
                    if (ss != SwitchState.OFFLINE) {
                        utTestState.setSwitchBaseState(sw, ss);
                    }
                    utTestState.setSwitchState(sw, ss);
                    changed = true;
                }
            }
            for (int fan = 0; fan < UTTestFans.NUM_FANS; fan++) {
                int sw = UTTestFans.fanSwitchMap.get(fan);
                Boolean st = swDevices[devNums[sw]].isSwitchOn(swNums[sw]);
                FanState fs = st == null ? FanState.OFFLINE : !st ? FanState.OFF : utTestState.getFanBaseState(fan);
                if (fs != utTestState.getFanState(fan)) {
                    if (fs != FanState.OFFLINE) {
                        utTestState.setFanBaseState(fan, fs);
                    }
                    utTestState.setFanState(fan, fs);
                    FanPIControl ctrlr = fanControl[fan];
                    if (ctrlr != null && fs == FanState.SPEED) {
                        ctrlr.setSpeed(utTestState.getFanSpeed(fan));
                    }
                    changed = true;
                }
            }
            for (int valve = 0; valve < UTTestValves.NUM_VALVES; valve++) {
                ValveState vs = !utControl.isOnline() ? ValveState.OFFLINE : utTestState.getValveBaseState(valve);
                if (vs != utTestState.getValveState(valve)) {
                    utTestState.setValveState(valve, vs);
                    try {
                        operateValve(valve);
                    }
                    catch (UtilityException e) {
                        LOG.log(Level.SEVERE, "Error operating valve {0}: {1}", new Object[]{valve, e});
                    }
                    changed = true;
                }
            }
            if (changed) {
                publishState();
            }
        }
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    

    /**
     *  Publishes the state of the system.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    private void publishState()
    {
        utTestState.setTickMillis(getTickPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(UTTestState.KEY, utTestState));
    }    


    /**
     *  Gets a fan ID from its name.
     */
    private int getFanId(String name) throws UtilityException
    {
        Integer id = UTTestFans.fanNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid fan name: " + name);
        }
        return id;
    }


    /**
     *  Gets a valve ID from its name.
     */
    private int getValveId(String name) throws UtilityException
    {
        Integer id = UTTestValves.valveNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid valve name: " + name);
        }
        return id;
    }

}
