package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.utility.constants.FanState;
import org.lsst.ccs.subsystem.utility.constants.HeaterState;
import org.lsst.ccs.subsystem.utility.constants.UTPurgeFans;
import org.lsst.ccs.subsystem.utility.constants.UTPurgeHeaters;
import org.lsst.ccs.subsystem.utility.constants.UTPurgeValves;
import org.lsst.ccs.subsystem.utility.constants.UtilityAgentProperties;
import org.lsst.ccs.subsystem.utility.constants.ValveState;
import org.lsst.ccs.subsystem.utility.constants.VpcControlState;
import org.lsst.ccs.subsystem.utility.data.UTPurgeState;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Implements the utility trunk & purge subsystem.
 *
 *  @author Owen Saxton
 */
public class UTPurgeMain extends Subsystem implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(UTPurgeMain.class.getName());

    @LookupName
    String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<SwitchControl> swDeviceList = new ArrayList<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final Map<String, FanPIControl> fanDeviceMap = new LinkedHashMap<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private VpcControl vpcControl;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private PurgeMaq20Device purgeControl;

    private String mpcFan, vpcFan, utFan;

    private final UTPurgeState purgeState = new UTPurgeState();
    private final FanPIControl[] fanControl = new FanPIControl[UTPurgeFans.NUM_FANS];


    /**
     *  Constructor.
     */
    public UTPurgeMain() {
        super("purgetest", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create and schedule an AgentPeriodicTask to update and publish the purge system state
        AgentPeriodicTask pt = new AgentPeriodicTask("UT-state",
                                                     () -> updateState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Subsystem post-initialization
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent's type.
        propertiesService.setAgentProperty(UtilityAgentProperties.UT_PURGE_TYPE, getClass().getCanonicalName());

        // Check fan controllers
        if (mpcFan != null) {
            fanControl[UTPurgeFans.FAN_MPC_ID] = fanDeviceMap.get(mpcFan);
        }
        if (fanControl[UTPurgeFans.FAN_MPC_ID] == null) {
            ErrorUtils.reportConfigError(LOG, name, "mpcFan", "not specified or not defined");
        }
        if (vpcFan != null) {
            fanControl[UTPurgeFans.FAN_VPC_ID] = fanDeviceMap.get(vpcFan);
        }
        if (fanControl[UTPurgeFans.FAN_VPC_ID] == null) {
            ErrorUtils.reportConfigError(LOG, name, "vpcFan", "not specified or not defined");
        }
        if (utFan != null) {
            fanControl[UTPurgeFans.FAN_UT_ID] = fanDeviceMap.get(utFan);
        }
        if (fanControl[UTPurgeFans.FAN_UT_ID] == null) {
            ErrorUtils.reportConfigError(LOG, name, "utFan", "not specified or not defined");
        }
        if (vpcControl == null) {
            ErrorUtils.reportConfigError(LOG, name, "VpcControl", "has not been defined");
        }
        if (purgeControl == null) {
            ErrorUtils.reportConfigError(LOG, name, "PurgeMaq20Device", "has not been defined");
        }

        // Initialize VPC heater values (temperatures)
        purgeState.setHeaterValue(UTPurgeHeaters.HEATER_VPC1_ID, vpcControl.getDeltaTemp(UTPurgeHeaters.HEATER_VPC1_ID));
        purgeState.setHeaterValue(UTPurgeHeaters.HEATER_VPC2_ID, vpcControl.getDeltaTemp(UTPurgeHeaters.HEATER_VPC2_ID));
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        //updateState();
        //publishState();        // For any GUIs
        LOG.info("Purge test subsystem started");
    }


    /**
     *  Gets the subsystem state.
     *
     *  @return  The full PDU system state
     */
    @Command(type=CommandType.QUERY, description="Get the full state")
    public UTPurgeState getFullState()
    {
        purgeState.setTickMillis(getTickPeriod());
        return purgeState;
    }


    /**
     *  Gets the list of fan names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of fan names")
    public List<String> getFanNames()
    {
        return new ArrayList<>(UTPurgeFans.fanNameToId.keySet());
    }


    /**
     *  Sets a fan state (using its name).
     *
     *  @param  name   The fan name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan control state")
    public void setFanState(@Argument(description="The fan name") String name,
                            @Argument(description="The control state") FanState state) throws UtilityException
    {
        try {
            int fanId = getFanId(name);
            if (state == FanState.OFFLINE || (state == FanState.TEMP && !UTPurgeFans.hasTempState[fanId])) {
                throw new UtilityException("Invalid " + name + " state: " + state);
            }
            synchronized (purgeState) {
                if (state == purgeState.getFanBaseState(fanId)) return;
                purgeState.setFanBaseState(fanId, state);
                if (purgeState.getFanState(fanId) == FanState.OFFLINE) return;
                purgeState.setFanState(fanId, state);
                FanPIControl ctrlr = fanControl[fanId];
                if (state == FanState.TEMP) {
                    ctrlr.setTemperature(purgeState.getDeltaTemp(fanId));
                    ctrlr.startLoop(purgeState.getFanSpeed(fanId));
                }
                else {
                    if (UTPurgeFans.hasTempState[fanId]) {
                        ctrlr.stopLoop();
                    }
                    if (state == FanState.SPEED) {
                        ctrlr.setSpeed(purgeState.getFanSpeed(fanId));
                    }
                }
                //setSwitchOn(UTPurgeFans.fanSwitchMap.get(fanId), state != FanState.OFF);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a fan speed (using its name).
     *
     *  @param  name   The fan name
     *  @param  speed  The fan speed
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan speed")
    public void setFanSpeed(@Argument(description="The fan name") String name,
                            @Argument(description="The fan speed") double speed) throws UtilityException
    {
        try {
            int fanId = getFanId(name);
            synchronized (purgeState) {
                purgeState.setFanSpeed(fanId, speed);
                if (purgeState.getFanState(fanId) == FanState.SPEED) {
                    fanControl[fanId].setSpeed(speed);
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets the VPC control state.
     *
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set the VPC control state")
    public void setVpcState(@Argument(description="The control state") VpcControlState state) throws UtilityException
    {
        try {
            synchronized (purgeState) {
                if (state == purgeState.getVpcState()) return;
                purgeState.setVpcState(state);
                HeaterState h1State = purgeState.getHeaterState(UTPurgeHeaters.HEATER_VPC1_ID);
                HeaterState h2State = purgeState.getHeaterState(UTPurgeHeaters.HEATER_VPC2_ID);
                if (state == VpcControlState.TEMP) {
                    if (h1State != HeaterState.OFFLINE) {
                        purgeState.setHeaterState(UTPurgeHeaters.HEATER_VPC1_ID, HeaterState.TEMP);
                    }
                    if (h2State != HeaterState.OFFLINE) {
                        purgeState.setHeaterState(UTPurgeHeaters.HEATER_VPC2_ID, HeaterState.TEMP);
                    }
                    vpcControl.startLoop();
                }
                else {
                    vpcControl.stopLoop();
                    if (h1State != HeaterState.OFFLINE) {
                        purgeState.setHeaterState(UTPurgeHeaters.HEATER_VPC1_ID,
                                                  purgeState.getHeaterBaseState(UTPurgeHeaters.HEATER_VPC1_ID));
                        operateHeater(UTPurgeHeaters.HEATER_VPC1_ID);
                    }
                    if (h2State != HeaterState.OFFLINE) {
                        purgeState.setHeaterState(UTPurgeHeaters.HEATER_VPC2_ID,
                                                  purgeState.getHeaterBaseState(UTPurgeHeaters.HEATER_VPC2_ID));
                        operateHeater(UTPurgeHeaters.HEATER_VPC2_ID);
                    }
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Gets the list of valve names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of valve names")
    public List<String> getValveNames()
    {
        return new ArrayList<>(UTPurgeValves.valveNameToId.keySet());
    }


    /**
     *  Sets a valve state (using its name).
     *
     *  @param  name   The valve name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a valve control state")
    public void setValveState(@Argument(description="The valve name") String name,
                              @Argument(description="The control state") ValveState state) throws UtilityException
    {
        try {
            int valveId = getValveId(name);
            if (state == ValveState.OFFLINE || state == ValveState.TEMP) {
                throw new UtilityException("Invalid " + name + " state: " + state);
            }
            synchronized (purgeState) {
                if (state == purgeState.getValveBaseState(valveId)) return;
                purgeState.setValveBaseState(valveId, state);
                ValveState currState = purgeState.getValveState(valveId);
                if (currState == ValveState.OFFLINE || currState == ValveState.TEMP) return;
                purgeState.setValveState(valveId, state);
                operateValve(valveId);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a valve position (using its name).
     *
     *  @param  name   The valve name
     *  @param  position  The valve position
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a valve position")
    public void setValvePosition(@Argument(description="The valve name") String name,
                                 @Argument(description="The valve position") double position) throws UtilityException
    {
        try {
            int valveId = getValveId(name);
            synchronized (purgeState) {
                purgeState.setValvePosition(valveId, position);
                if (purgeState.getValveState(valveId) == ValveState.POSN) {
                    operateValve(valveId);
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a fan delta temperature (using its name).
     *
     *  @param  name  The fan name
     *  @param  temp  The delta temperature to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a delta temperature")
    public void setFanDeltaTemp(@Argument(description="The fan name") String name,
                                @Argument(description="The delta temperature") double temp) throws UtilityException
    {
        try {
            int fanId = getFanId(name);
            synchronized (purgeState) {
                purgeState.setDeltaTemp(fanId, temp);
                if (purgeState.getFanState(fanId) == FanState.TEMP) {
                    if (fanId != UTPurgeFans.FAN_VPC_ID) {
                        fanControl[fanId].setTemperature(temp);
                    }
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Gets the list of heater names.
     * 
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of heater names")
    public List<String> getHeaterNames()
    {
        return new ArrayList<>(UTPurgeHeaters.heaterNameToId.keySet());
    }


    /**
     *  Sets a heater control state (using its name).
     *
     *  @param  name   The heater name
     *  @param  state  The state to set
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a heater control state")
    public void setHeaterState(@Argument(description="The heater name") String name,
                               @Argument(description="The control state") HeaterState state) throws UtilityException
    {
        try {
            int htrId = getHeaterId(name);
            if (state == HeaterState.OFFLINE || state == HeaterState.TEMP) {
                throw new UtilityException("Invalid " + name + " state: " + state);
            }
            synchronized (purgeState) {
                if (state == purgeState.getHeaterBaseState(htrId)) return;
                purgeState.setHeaterBaseState(htrId, state);
                HeaterState currState = purgeState.getHeaterState(htrId);
                if (currState == HeaterState.OFFLINE || currState == HeaterState.TEMP) return;
                purgeState.setHeaterState(htrId, state);
                operateHeater(htrId);
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets a heater value (using its name).
     *
     *  @param  name   The heater name
     *  @param  value  The heater value (power duty cycle or delta rtemperature)
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a heater power")
    public void setHeaterValue(@Argument(description="The heater name") String name,
                               @Argument(description="The value") double value) throws UtilityException
    {
        try {
            int htrId = getHeaterId(name);
            synchronized (purgeState) {
                purgeState.setHeaterValue(htrId, value);
                if (htrId == UTPurgeHeaters.HEATER_SIM_ID) {
                    operateHeater(htrId);
                }
                else if (purgeState.getHeaterState(htrId) == HeaterState.TEMP) {
                    vpcControl.setDeltaTemp(htrId, value);
                }
            }
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets the update (tick) period.
     *
     *  @param  period  The update period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the update period")
    public void setUpdatePeriod(@Argument(name="period", description="The tick period (msecs)") int period)
    {
        setTickPeriod(period);
        //updateState();
        publishState();    // Must do this for GUI
    }


    /**
     *  Operates a heater.
     *
     *  @param  htrId  The heater ID
     *  @throws  UtitliyException
     */
    private void operateHeater(int htrId) throws UtilityException
    {
        HeaterState state = purgeState.getHeaterState(htrId);
        if (state == HeaterState.OFFLINE || state == HeaterState.TEMP) return;
        //setSwitchOn(UTPurgeHeaters.heaterSwitchMap.get(htrId), state == HeaterState.ON);
        if (htrId == UTPurgeHeaters.HEATER_SIM_ID && state == HeaterState.ON) {
            purgeControl.setSimHeater(purgeState.getHeaterValue(htrId));
        }
    }


    /**
     *  Operates a coolant valve.
     *
     *  @param  valveId  The valve ID
     *  @throws  UtilityException
     */
    private void operateValve(int valveId) throws UtilityException
    {
        ValveState state = purgeState.getValveState(valveId);
        if (state == ValveState.OFFLINE || state == ValveState.TEMP) return;
        double position = state == ValveState.SHUT ? 0.0 : purgeState.getValvePosition(valveId);
        if (valveId == UTPurgeValves.VALVE_UT_ID) {
            purgeControl.setUtValve(position);
        }
        
    }


    /**
     *  Updates the purge system state when devices go offline or online.
     */
    private void updateState()
    {
        synchronized (purgeState) {
            boolean changed = false;
            for (int fan = 0; fan < UTPurgeFans.NUM_FANS; fan++) {
                int sw = UTPurgeFans.fanSwitchMap.get(fan);
                Boolean st = true; // swDevices[devNums[sw]].isSwitchOn(swNums[sw]);
                FanState fs = st == null ? FanState.OFFLINE : !st ? FanState.OFF : purgeState.getFanBaseState(fan);
                if (fs != purgeState.getFanState(fan)) {
                    if (fs != FanState.OFFLINE) {
                        purgeState.setFanBaseState(fan, fs);
                    }
                    purgeState.setFanState(fan, fs);
                    FanPIControl ctrlr = fanControl[fan];
                    if (ctrlr != null && fs == FanState.SPEED) {
                        ctrlr.setSpeed(purgeState.getFanSpeed(fan));
                    }
                    changed = true;
                }
            }
            for (int htr = 0; htr < UTPurgeHeaters.NUM_HEATERS; htr++) {
                int sw = UTPurgeHeaters.heaterSwitchMap.get(htr);
                Boolean st = true;  // swDevices[devNums[sw]].isSwitchOn(swNums[sw]);
                HeaterState hs = st == null ? HeaterState.OFFLINE :
                                 UTPurgeHeaters.hasAutoState[htr] && purgeState.getVpcState() == VpcControlState.TEMP
                                   ? HeaterState.TEMP : st ? HeaterState.ON : HeaterState.OFF;
                if (hs != purgeState.getHeaterState(htr)) {
                    if (hs != HeaterState.TEMP && hs != HeaterState.OFFLINE) {
                        purgeState.setHeaterBaseState(htr, hs);
                    }
                    purgeState.setHeaterState(htr, hs);
                    try {
                        operateHeater(htr);
                    }
                    catch (UtilityException e) {
                        LOG.log(Level.SEVERE, "Error operating heater {0}: {1}", new Object[]{htr, e});
                    }
                    changed = true;
                }
            }
            for (int valve = 0; valve < UTPurgeValves.NUM_VALVES; valve++) {
                ValveState vs = !purgeControl.isOnline() ? ValveState.OFFLINE :
                                UTPurgeValves.hasTempState[valve] && purgeState.getVpcState() == VpcControlState.TEMP
                                  ? ValveState.TEMP : purgeState.getValveBaseState(valve);
                if (vs != purgeState.getValveState(valve)) {
                    purgeState.setValveState(valve, vs);
                    try {
                        operateValve(valve);
                    }
                    catch (UtilityException e) {
                        LOG.log(Level.SEVERE, "Error operating valve {0}: {1}", new Object[]{valve, e});
                    }
                    changed = true;
                }
            }
            if (changed) {
                publishState();
            }
        }
    }


    /**
     *  Sets the tick period
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }
    

    /**
     *  Publishes the state of the system.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    private void publishState()
    {
        purgeState.setTickMillis(getTickPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(UTPurgeState.KEY, purgeState));
    }    


    /**
     *  Gets a fan ID from its name.
     */
    private int getFanId(String name) throws UtilityException
    {
        Integer id = UTPurgeFans.fanNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid fan name: " + name);
        }
        return id;
    }


    /**
     *  Gets a valve ID from its name.
     */
    private int getValveId(String name) throws UtilityException
    {
        Integer id = UTPurgeValves.valveNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid valve name: " + name);
        }
        return id;
    }


    /**
     *  Gets a heater ID from its name.
     */
    private int getHeaterId(String name) throws UtilityException
    {
        Integer id = UTPurgeHeaters.heaterNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid heater name: " + name);
        }
        return id;
    }

}
