package org.lsst.ccs.subsystem.utility;

import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20AnalogControl;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Controls utility trunk Maq20 device
 * 
 *  @author saxton
 */
public class UTMaq20Device extends Maq20Device {

    public static final int
        CHAN_UT_VALVE   = 0;

    @LookupField (strategy=LookupField.Strategy.ANCESTORS)
    private PurgeTestMain main;
    @LookupField (strategy=LookupField.Strategy.CHILDREN)
    private Maq20AnalogControl maqCtrl;

    private static final Logger LOG = Logger.getLogger(UTMaq20Device.class.getName());
    private int modIndex;


    /**
     *  Basic initialization.
     */
    @Override
    public void initDevice()
    {
        super.initDevice();
        if (maqCtrl == null) {
            ErrorUtils.reportConfigError(LOG, name, "analog control", "not defined");
        }
        modIndex = maqCtrl.getModIndex();
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  Channel value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        if ((type & 0xff) == modIndex) {
            if (hwChan == CHAN_UT_VALVE) {
                value = 62.5 * value - 0.25;
            }
        }
        return value;
    }


    /**
     *  Sets the UT valve position.
     *
     *  @param  posn  Requested position (0 - 1)
     *  @throws  UtilityException
     */
    public void setUtValve(double posn) throws UtilityException
    {
        setValve(CHAN_UT_VALVE, posn);
    }


    /**
     *  Sets a valve position.
     *
     *  @param  chan  The channel number
     *  @param  posn  Requested position (0 - 1)
     *  @throws  UtilityException
     */
    public void setValve(int chan, double posn) throws UtilityException
    {
        double value = 0.016 * Math.min(Math.max(posn, 0.0), 1.0) + 0.004;
        try {
            maqCtrl.writeValue(chan, value);
        }
        catch (DriverException e) {
            throw new UtilityException("Error operating valve (channel " + chan + "): " + e);
        }
    }

}
