package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20PWMControl;
import org.lsst.ccs.subsystem.common.PIController;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Implements a fan speed loop controller for the utility trunk system.
 *
 *  Also allows open-loop speed control if no temperatures are specified.
 *
 *  @author Owen Saxton
 */
public class FanPIControl implements HasLifecycle {

    private static final String PIC = "Pic";

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskServices;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String,Channel> allChannels = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String,Maq20PWMControl> allFans = new HashMap<>();
    
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  gain = Double.NaN;              // loop gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  timeConst = Double.NaN;         // integration time constant (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  smoothTime = Double.NaN;        // input smoothing time (secs)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  awGain = Double.NaN;            // anti-windup gain
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  baseDuty = Double.NaN;          // base duty cycle input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  tolerance = Double.NaN;         // maximum on-target error (%)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  minInput = -10.0;               // minimum input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  maxInput = 10.0;                // maximum input
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile double  minOutput = 1.0;                // minimum PID output (inverse duty cycle)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Double  maxOutput = 10.0;               // maximum PID output (inverse duty cycle)
    @ConfigurationParameter(category=PIC, isFinal=true)
    private volatile Integer updateTime = Integer.MAX_VALUE; // The update time interval (msec)

    private String[] refTempChans;     // Reference temperature channels to use
    private String[] ctrlTempChans;    // Controltemperature channels to use
    private String[] fans;             // The fan controllers to use

    private final List<Channel> refTempChanList = new ArrayList<>();   // Reference temperature channels to use
    private final List<Channel> ctrlTempChanList = new ArrayList<>();  // Control temperature channels to use
    private final Map<String,Maq20PWMControl> fansMap = new HashMap<>();  // Maq20 fan controllers to use

    private static final Logger LOG = Logger.getLogger(FanPIControl.class.getName());
    private PIController pic;          // The PI controller
    private double lastDuty;           // The last duty cycle value set
    private boolean active = false;    // True if loop is active
    private double setTemp;            // Temperature set point
    private boolean loopFailed = false;
    private long lastLoopTime;


    /**
     *  Sets up the fan speed control loop timer task.
     */
    @Override
    public void build() {
        AgentPeriodicTask apt = new AgentPeriodicTask("fan-loop-" + name,
                                                      () -> iterateLoop()).withPeriod(Duration.ofMillis(500));
        periodicTaskServices.scheduleAgentPeriodicTask(apt);
    }


    /**
     *  Initializes the parameters.
     */
    @Override
    public void init()
    {
        if (fans == null || fans.length == 0) {
            ErrorUtils.reportConfigError(LOG, name, "fans", "is missing or empty");
        }
        else {
            for (String cName : fans) {
                Maq20PWMControl cmpt = allFans.get(cName);
                if (cmpt != null) {
                    fansMap.put(cName, cmpt);
                }
                else {
                    ErrorUtils.reportConfigError(LOG, name, "fans", "contains non-Maq20PWMControl item");
                }
            }
        }
        if (refTempChans == null) {
            ErrorUtils.reportConfigError(LOG, name, "refTempChans", "is missing");
        }
        else {
            for (String cName : refTempChans) {
                Channel cmpt = allChannels.get(cName);
                if (cmpt != null) {
                    refTempChanList.add(cmpt);
                }
                else {
                    ErrorUtils.reportConfigError(LOG, name, "refTempChans", "contains non-Channel item");
                }
            }
        }
        if (ctrlTempChans == null) {
            ErrorUtils.reportConfigError(LOG, name, "ctrlTempChans", "is missing");
        }
        else {
            for (String cName : ctrlTempChans) {
                Channel cmpt = allChannels.get(cName);
                if (cmpt != null) {
                    ctrlTempChanList.add(cmpt);
                }
                else {
                    ErrorUtils.reportConfigError(LOG, name, "ctrlTempChans", "contains non-Channel item");
                }
            }
        }
        if (ctrlTempChanList.isEmpty() ^ refTempChanList.isEmpty()) {
            ErrorUtils.reportConfigError(LOG, name, "ctrlTempChans and refTempChans", "are not both non-empty nor both empty");
        }
        if (ctrlTempChanList.isEmpty()) return;  // No loop control requested

        if (gain == null) {
            ErrorUtils.reportConfigError(LOG, name, "gain", "is missing");
        }
        if (timeConst.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "timeConst", "is missing");
        }
        if (smoothTime.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "smoothTime", "is missing");
        }
        if (awGain.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "awGain", "is missing");
        }
        if (baseDuty.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "baseDuty", "is missing");
        }
        if (tolerance.isNaN()) {
            ErrorUtils.reportConfigError(LOG, name, "tolerance", "is missing");
        }
        if (updateTime == Integer.MAX_VALUE) {
            ErrorUtils.reportConfigError(LOG, name, "updateTime", "is missing");
        }
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(baseDuty);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
    }


    /**
     *  Sets the fan speed (or duty cycle).
     *
     *  @param  speed  The speed to set
     */
    public void setSpeed(double speed)
    {
        for (Entry<String,Maq20PWMControl> e : fansMap.entrySet()) {
            try {
                e.getValue().setDutyCycle1(speed);
            }
            catch (Exception ex) {
                LOG.log(Level.SEVERE, "Error setting {0} fan speed: {1}", new Object[]{e.getKey(), ex.getMessage()});
            }
        }
    }


    /**
     *  Sets the target temperature.
     *
     *  @param  temp  The temperature to set
     *  @throws  UtilityException
     */
    public void setTemperature(double temp) throws UtilityException
    {
        checkPicPresent();
        setTemp = temp;
        pic.setSetpoint(setTemp);
    }


    /**
     *  Gets the target temperature.
     *
     *  @return  The set temperature
     */
    public double getTemperature()
    {
        return setTemp;
   }


    /**
     *  Sets the gain.
     *
     *  @param  gain  The gain to set
     *  @throws  UtilityException
     */
    public void setGain(double gain) throws UtilityException
    {
        checkPicPresent();
        this.gain = gain;
        if (pic != null) {  // Can get called during startup before pic exists
            pic.setPID(gain, timeConst);
        }
    }


    /**
     *  Gets the gain.
     *
     *  @return  The gain
     */
    public double getGain()
    {
        return gain;
    }


    /**
     *  Sets the time constant.
     *
     *  @param  time  The time constant to set
     *  @throws  UtilityException
     */
    public void setTimeConstant(double time) throws UtilityException
    {
        checkPicPresent();
        timeConst = time;
        pic.setPID(gain, timeConst);
    }


    /**
     *  Gets the time constant.
     *
     *  @return  The time constant
     */
    public double getTimeConstant()
    {
        return timeConst;
   }


    /**
     *  Starts the control loop.
     *
     *  @param  duty  The initial duty cycle value
     *  @throws  UtilityException
     */
    public void startLoop(double duty) throws UtilityException
    {
        checkPicPresent();
        if (active) return;
        lastDuty = 1.0 / duty;
        startLoop();
    }


    /**
     *  Starts the control loop.
     *
     *  @throws  UtilityException
     */
    public void startLoop() throws UtilityException
    {
        checkPicPresent();
        if (active) return;
        pic.reset();
        pic.setIntegral(lastDuty - baseDuty);
        active = true;
    }


    /**
     *  Stops the control loop.
     *
     *  @throws  UtilityException
     */
    public void stopLoop() throws UtilityException
    {
        checkPicPresent();
        if (!active) return;
        active = false;
        //setSpeed(1.0);  // Maximum fan speed
    }


    /**
     *  Gets the control loop state.
     *
     *  @return  Whether the control loop is active
     */
    public boolean isLoopActive()
    {
        return active;
    }


    /**
     *  Resets the controller.
     * 
     *  @throws  UtilityException
     */
    public void reset() throws UtilityException
    {
        checkPicPresent();
        pic.reset();
    }


    /**
     *  Timer routine for control loop iteration.
     */
    private void iterateLoop()
    {
        if (!active) return;
        long time = System.currentTimeMillis();
        if (time < lastLoopTime + updateTime) return;
        lastLoopTime = time;
        double refTemp = 0.0, ctrlTemp = 0.0;
        int count = 0;
        for (Channel tempChan : refTempChanList) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                refTemp += value;
                count++;
            }
        }
        if (count > 0) {
            refTemp /= count;
        }
        else {
            if (!loopFailed) {
                LOG.severe("Control loop iteration failed: no valid reference temperature values available");
                loopFailed = true;
            }
            return;
        }
        count = 0;
        for (Channel tempChan : ctrlTempChanList) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                ctrlTemp += value;
                count++;
            }
        }
        if (count > 0) {
            ctrlTemp /= count;
        }
        else {
            if (!loopFailed) {
                LOG.severe("Control loop iteration failed: no valid control temperature values available");
                loopFailed = true;
            }
            return;
        }
        lastDuty = pic.performPI(new double[]{ctrlTemp - refTemp}, time / 1000.0);
        setSpeed(1.0 / lastDuty);
        loopFailed = false;
    }


    /**
     *  Checks whether PI controller is present.
     */
    private void checkPicPresent() throws UtilityException
    {
        if (pic == null) {
            throw new UtilityException("Control loop not present");
        }
    }

}
