package org.lsst.ccs.subsystem.utility;

import java.util.ArrayList;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
//import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.utility.constants.MpmAlert;
import org.lsst.ccs.subsystem.utility.constants.ConditionState;
import org.lsst.ccs.subsystem.utility.constants.LatchState;
import org.lsst.ccs.subsystem.utility.constants.PLCState;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.constants.UtilityAgentProperties;
import org.lsst.ccs.subsystem.utility.data.UtilityException;
import org.lsst.ccs.subsystem.utility.data.MpmSysState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * The master protection system
 *
 * @author The LSST CCS Team
 */
public class MpmMain /*extends Subsystem*/ implements HasLifecycle, ClearAlertHandler {

    private static final int[] switchChannels = new int[MpmSysState.NUM_SWITCHES];
    static {
        switchChannels[MpmSysState.SW_BLOCK_COLD_HEAT] = MpmPlutoDevice.SW_BLOCK_COLD_HEAT;
        switchChannels[MpmSysState.SW_BLOCK_COLD_REFG] = MpmPlutoDevice.SW_BLOCK_COLD_REFG;
        switchChannels[MpmSysState.SW_BLOCK_CRYO_HEAT] = MpmPlutoDevice.SW_BLOCK_CRYO_HEAT;
        switchChannels[MpmSysState.SW_BLOCK_CRYO_REFG] = MpmPlutoDevice.SW_BLOCK_CRYO_REFG;
        switchChannels[MpmSysState.SW_BLOCK_COOLANT] = MpmPlutoDevice.SW_BLOCK_COOLANT;
        switchChannels[MpmSysState.SW_BLOCK_REB_POWER] = MpmPlutoDevice.SW_BLOCK_REB_POWER;
        switchChannels[MpmSysState.SW_BLOCK_UT_POWER] = MpmPlutoDevice.SW_BLOCK_UT_POWER;
    }
    private static final Map<Integer, MpmAlert> alertMap = new HashMap<>();
    static {
        alertMap.put(MpmSysState.LATCH_COLD_TEMP_HIGH, MpmAlert.COLD_TEMP_HIGH);
        alertMap.put(MpmSysState.LATCH_COLD_TEMP_LOW, MpmAlert.COLD_TEMP_LOW);
        alertMap.put(MpmSysState.LATCH_CRYO_TEMP_HIGH, MpmAlert.CRYO_TEMP_HIGH);
        alertMap.put(MpmSysState.LATCH_CRYO_TEMP_LOW, MpmAlert.CRYO_TEMP_LOW);
        alertMap.put(MpmSysState.LATCH_CRYO_VACUUM, MpmAlert.CRYO_VACUUM_BAD);
        alertMap.put(MpmSysState.LATCH_HEX_VACUUM, MpmAlert.HEX_VACUUM_BAD);
        alertMap.put(MpmSysState.LATCH_UT_LEAK, MpmAlert.UT_COOLANT_LEAK);
        alertMap.put(MpmSysState.LATCH_UT_LEAK_FAULT, MpmAlert.UT_COOLANT_LEAK);
        alertMap.put(MpmSysState.LATCH_UT_SMOKE, MpmAlert.UT_SMOKE_DETC);
        alertMap.put(MpmSysState.LATCH_UT_SMOKE_FAULT, MpmAlert.UT_SMOKE_DETC);
        alertMap.put(MpmSysState.LATCH_UT_TEMP, MpmAlert.UT_TEMP_HIGH);
    }
    private static final Map<String, Integer> revAlertMap = new HashMap<>();
    static {
        for (int cond : alertMap.keySet()) {
            revAlertMap.put(alertMap.get(cond).getId(), cond);
        }
    }
    private static final String[] plcNames = new String[MpmSysState.NUM_PLCS];
    static {
        plcNames[MpmSysState.PLC_TRUNK] = "Trunk";
        plcNames[MpmSysState.PLC_COLD] = "Cold";
        plcNames[MpmSysState.PLC_CRYO] = "Cryo";
    }

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService apts;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService as;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService aps;
    @LookupField( strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private MpmPlutoDevice plutoDevc;

    // From Groovy file

    // General
    private static final Logger LOG = Logger.getLogger(MpmMain.class.getName());
    private final MpmSysState protState = new MpmSysState();
    private final Device[] switchDevices = new Device[MpmSysState.NUM_SWITCHES];
    private boolean running = false;
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();

    /*
    public Protection() {
        super("protection", AgentInfo.AgentType.WORKER);
    }
    */

    /**
     *  Build phase
     */
    @Override
    public void build() {
        //setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        //Create and schedule an AgentPeriodicTask to update the protection state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("Protection-state",
                                   () -> updateProtState()).withPeriod(Duration.ofMillis(1000));
        apts.scheduleAgentPeriodicTask(pt);
        //ass.registerState(ProtectionState.class, "Protection state", this);
        //ass.updateAgentState(ProtectionState.UNKNOWN);
        for (MpmAlert alert : MpmAlert.values()) {
            activeAlertMap.put(alert.getId(), false);
        }
    }


    /**
     *  Post initialization
     */
    @Override
    public void postInit() {
        //Set a property to define that this Agent is a protection subsystem.
        aps.setAgentProperty(UtilityAgentProperties.MPM_TYPE, MpmMain.class.getCanonicalName());

        if (plutoDevc != null) {
            for (int cond : plutoDevc.getLatchIds()) {
                protState.setLatch(cond, LatchState.CLEAR);
            }
            for (int cond : plutoDevc.getConditionIds()) {
                protState.setCondition(cond, ConditionState.NO);
            }
        }
        else {
            MonitorLogUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }

        for (int sw : switchChannels) {
            protState.setSwitchState(sw, SwitchState.OFFLINE);
            switchDevices[sw] = plutoDevc;
        }
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("Protection subsystem started");
        running = true;
    }


    /**
     *  Gets the state of the protection system.
     *
     *  @return  The protection system state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the protection system state")
    public MpmSysState getSystemState() {
        protState.setTickMillis(getTickPeriod());
        return protState;
    }    


    /**
     *  Gets the list of switch names.
     *
     *  @return  The switch names.
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.QUERY, description="Get switch names")
    public List<String> getSwitchNames() throws UtilityException
    {
        return new ArrayList(SwitchNames.NAME_MAP.keySet());
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a switch")
    public void setSwitchOn(@Argument(description="The switch number") int sw,
                            @Argument(description="Whether to turn on") boolean on) throws UtilityException
    {
        try {
            if (sw < 0 || sw >= MpmSysState.NUM_SWITCHES) {
                throw new UtilityException("Invalid switch number: " + sw);
            }
            setSwitch(sw, on);
        }
        finally {
            publishState();
        }
        
    }


    /**
     *  Turns a named switch on or off.
     *
     *  @param  name  The switch name.
     *  @param  on    Whether to turn on or off
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description="The switch name") String name,
                                 @Argument(description="Whether to turn on") boolean on) throws UtilityException
    {
        Integer sw = SwitchNames.NAME_MAP.get(name);
        try {
            if (sw == null) {
                throw new UtilityException("Invalid switch name: " + name);
            }
            setSwitch(sw, on);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Turns a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  Whether to turn on or off
     *  @throws  UtilityException
     */
    private void setSwitch(int sw, boolean on) throws UtilityException
    {
        SwitchState state = protState.getSwitchState(sw);
        if (state == SwitchState.OFFLINE) return;
        Device swDevice = switchDevices[sw];
        if (swDevice == plutoDevc) {
            plutoDevc.setSwitchOn(switchChannels[sw], on);
        }
    }


    /**
     *  Gets whether a switch is on.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     *  @throws  UtilityException
     */
    private Boolean isSwitchOn(int sw)
    {
        Boolean value = false;
        Device swDevice = switchDevices[sw];
        if (swDevice == plutoDevc) {
            value = plutoDevc.isSwitchOn(switchChannels[sw]);
        }
        return value;
    }


    /**
     *  Clears a latched condition.
     *
     *  @param  cond  The condition number.
     *  @throws  UtilityException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearLatch(@Argument(description="The condition number") int cond) throws UtilityException
    {
        try {
            plutoDevc.clearLatch(cond);
        }
        finally {
            publishState();
        }
    }


    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(description="The tick period (ms)") int value)
    {
        setTickPeriod(value);
        publishState();
    }


    /**
     *  Updates the protection system state periodically.
     *
     *  The protection state consists mainly of the state of the switches (lines) being
     *  controlled, along with whether they can be turned on.
     */
    private void updateProtState()
    {
        if (!running) return;

        boolean changed = false;

        for (int plc = 0; plc < MpmSysState.NUM_PLCS; plc++) {
            PLCState oldState = protState.getPlcState(plc);
            Boolean active = plutoDevc.isPlcActive(plc);
            PLCState newState = active == null ? PLCState.OFFLINE : active ? PLCState.ALIVE : PLCState.DEAD;
            if (newState != oldState) {
                changed = true;
                protState.setPlcState(plc, newState);
                String desc = plcNames[plc] + " protection PLC ";
                if (newState == PLCState.ALIVE) {
                    lowerAlert(MpmAlert.PROT_PLC_NOT_ALIVE, desc + "is alive");
                }
                else {
                    raiseAlert(MpmAlert.PROT_PLC_NOT_ALIVE, desc + (newState == PLCState.DEAD ? "has died" : "is offline"));
                }
            }
        }

        for (int sw = 0; sw < MpmSysState.NUM_SWITCHES; sw++) {
            SwitchState oldState = protState.getSwitchState(sw);
            Boolean isOn = isSwitchOn(sw);
            SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
            if (state != oldState) {
                protState.setSwitchState(sw, state);
                changed = true;
            }
        }

        for (int cond = 0; cond < MpmSysState.NUM_LATCHES; cond++) {
            Boolean active = plutoDevc.isLatchActive(cond);
            Boolean latched = plutoDevc.isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = protState.getLatch(cond); 
            if (state != oldState) {
                protState.setLatch(cond, state);
                MpmAlert alert = alertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlert(alert, "Protection PLC error condition set");
                }
                else if (state != LatchState.OFFLINE && oldState == LatchState.ACTIVE) {
                    lowerAlert(alert, "Protection PLC error condition cleared");
                }
                changed = true;
            }
        }

        for (int cond = 0; cond < MpmSysState.NUM_CONDITIONS; cond++) {
            Boolean active = plutoDevc.isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != protState.getCondition(cond)) {
                protState.setCondition(cond, state);
                changed = true;
            }
        }

        if (changed) {
            publishState();
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The protection alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlert(MpmAlert alert, String cond)
    {
        as.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
        activeAlertMap.put(alert.getId(), true);
    }


     /**
     *  Checks whether an alert has been raised.
     *
     *  @param  alert  The protection alert to check
     *  @return  Whether it has been raised
     */
    private boolean isAlertRaised(MpmAlert alert)
    {
        return activeAlertMap.get(alert.getId()).equals(Boolean.TRUE);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The protection alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(MpmAlert alert, String cond)
    {
        as.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
        activeAlertMap.put(alert.getId(), false);
    }


    /**
     *  Clears an alert.
     * 
     *  @param  alert  The alert
     *  @return  Action code
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert)
    {
        Boolean active = activeAlertMap.get(alert.getAlertId());
        if (active != null && !active) {
            Integer cond = revAlertMap.get(alert.getAlertId());
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (UtilityException e) {
                    LOG.severe("Error clearing latched PLC condition: " + e);
                }
            }
        }
        return active == null ? ClearAlertCode.UNKNOWN_ALERT
                              : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }


    /**
     *  Publishes the state of the protection system.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        protState.setTickMillis(getTickPeriod());
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(MpmSysState.KEY, protState));
    }


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        apts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)apts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
