package org.lsst.ccs.subsystem.utility;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.lambda.Cpfe1000;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a Lambda CPFE1000 power supply system.
 *
 *  @author Owen Saxton
 */
public class CpfeDevice extends Device {

   /**
    *  Constants.
    */
    public static final int
        CHAN_STATUS = 0,
        CHAN_TEMPERATURE = 1,
        CHAN_VOLTAGE = 2,
        CHAN_CURRENT = 3,
        NUM_CHANS = 4;

   /**
    *  Private lookup maps.
    */
    private static final Map<String, Cpfe1000.ConnType> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", Cpfe1000.ConnType.SERIALUSB);
        cTypeMap.put("FTDI", Cpfe1000.ConnType.FTDIUSB);
    }

   /**
    *  Data fields.
    */
    private String connType;     // Connection type string
    private String devcId;       // Device identification
    private int address = 0xa0;  // I2C bus address

    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final Cpfe1000 cpfe = new Cpfe1000();  // Associated CPFE object
    private Cpfe1000.ConnType cType;


   /**
    *  Performs basic initialization.
    */
    @Override
    protected void initDevice()
    {
        if (connType == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "connType", "is missing");
        }
        cType = cTypeMap.get(connType.toUpperCase());
        if (cType == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "connType", "is invalid");
        }
        if (devcId == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "devcId", "is missing");
        }
        fullName = "CPFE1000 (" + devcId + ":" + address + ")";
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        try {
            cpfe.open(cType, devcId, address);
            setOnline(true);
            initSensors();
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
        try {
            cpfe.close();
        }
        catch (DriverException e) {
        }
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            MonitorLogUtils.reportError(sLog, name, "hwchan", hwChan);
        }
        return new int[]{0, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        try {
            switch (hwChan) {

            case CHAN_STATUS:
                value = cpfe.readStatus(); break;

            case CHAN_TEMPERATURE:
                value = cpfe.readTemperature(); break;

            case CHAN_VOLTAGE:
                value = cpfe.readVoltage(); break;

            case CHAN_CURRENT:
                value = cpfe.readCurrent(); break;
            }
        }
        catch (DriverException e) {
            sLog.error("Error reading CPFE1000 data: " + e);
            setOnline(false);
        }

        return value;
    }

}
