package org.lsst.ccs.subsystem.teststand.limits;

import java.util.List;

/**
 * A simple algorithm which will adjust limits as they head towards the target,
 * but not if they move away from the target. This algorithm does not apply
 * any time or slope limits.
 *
 * @author tonyj
 */
class RatchetTransitionAlgorithm implements TransitionAlgorithm {

    private LimitAlgorithm limitAlgorithm;
    private double lastValue;
    private boolean goingUp;
    private LimitsInterface limits;

    @Override
    public void init(List<Double> parameters, LimitAlgorithm limitAlgorithm, double initialValue, LimitsInterface limits, long startTime) {
        if (!parameters.isEmpty()) {
            throw new IllegalArgumentException("RatchetTransitionAlgorithm does not take any parameters");
        }
        this.limitAlgorithm = limitAlgorithm;
        this.lastValue = initialValue;
        this.goingUp = limitAlgorithm.getTargetValue() > initialValue;
        this.limits = limits;
    }

    @Override
    public boolean isAtTarget(double currentValue, long time) {
        return currentValue > limitAlgorithm.getTargetLimits().getWarnLow() && currentValue < limitAlgorithm.getTargetLimits().getWarnHigh();
    }

    @Override
    public void adjustLimits(double currentValue, long time) {
        if (goingUp && currentValue > lastValue) {
            lastValue = currentValue;
            limitAlgorithm.adjust(currentValue, limits);
        } else if (!goingUp && currentValue < lastValue) {
            lastValue = currentValue;
            limitAlgorithm.adjust(currentValue, limits);
        }
    }

    @Override
    public void completeTransition(long time) {
        limitAlgorithm.getTargetLimits().copyTo(limits);
    }

    @Override
    public LimitsInterface getLimits() {
        return limits;
    }

}
