package org.lsst.ccs.subsystem.teststand;

import org.lsst.ccs.Agent;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.gpio.GPIODriver;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.drivers.gpio.GPIODriverUno1483;
import org.lsst.ccs.subsystem.teststand.data.UnoGPIOLines;
import org.lsst.ccs.subsystem.teststand.states.UniblitzShutterState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Class to operate an Uniblitz Shutter Controller (Uniblitz VMM-D4 or 
 *  VCM-D1) via a GIPO output line.  This controller provides no feedback 
 *  signal, so published shutter state reflects only the GPIO control line.
 *
 *  @author Al Eisner
 */
public class UniblitzShutter implements HasLifecycle {

    @LookupName
    private String name;

    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService stateService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @ConfigurationParameter(isFinal = true, description = "GPIO output line",
                            category = "General")
    private volatile UnoGPIOLines.Output gpioLine;

    private GPIODriver gpio;
    private GPIODriver.GPIOChannel channel;
    private Logger log;
    private boolean initialized = false;

   /**
    *  Check validity of configured output line. 
    *  If valid, set line to closed-shutter state.
    */
    @Override
    public void init() {

        stateService.registerState(UniblitzShutterState.class, 
                                   name + " Shutter State", this);

        log = subsys.getLogger();
        if (gpioLine == null) {
            MonitorLogUtils.reportConfigError(log, name, "gpioLine", "not set");
        }
        try {
            gpio = new GPIODriverUno1483();
            channel = gpio.getChannel(gpioLine.getLine());
            if (channel.getDirection() != GPIODriver.GPIOChannel.Direction.OUT) {
                MonitorLogUtils.reportConfigError(log, name, "gpioLine",
                                                  "not a writable channel");
            }
            channel.lock();          // A precaution against interference
            initialized = true;
            channel.write(false);
            publishState(false);
	} 
        catch (DriverException ex) {
            log.error(name + "Failure to initialize GPIO channel", ex);
        }
    }

   /**
    *  Open or close a shutter
    *
    *  @param  state  <true|false> for <opened|closed>
    *  @throws DriverException
    */
    public void setShutter(boolean state) throws DriverException {
        if (!initialized) {
	    throw new RuntimeException(name + ": GPIO line not initialized");
        }
        channel.write(state);
        publishState(state);
    }

   /**
    *  Publish requested shutter position as a State
    *
    *  @param  state  <true|false> for <opened|closed>
    */

    private void publishState(boolean state)
    {
        UniblitzShutterState enumState = UniblitzShutterState.getEnum(state);
        stateService.updateAgentComponentState(this, enumState);
        log.info(name + ":  digital control line set to state " 
                 + enumState.toString());
    }

   /**
    *  Command to open or close a shutter
    *
    *  @param  state  <true|false> for <open|close>
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="setShutterPosition",
             description="Open or close Uniblitz shutters on one controller")
    public void setShutterPosition(@Argument(name="<boolean>",
     description="<true|false> for <open|close>") boolean state)
     throws DriverException
    {
        setShutter(state);
    }

}
