package org.lsst.ccs.subsystem.teststand;

import hep.aida.util.XMLUtils;
import hep.aida.*;
import java.util.Map;
import java.util.HashMap;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.devices.picoammeters.KeysightB2981a;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 * Device class for communicating with the Keysight driver
 *
 * @author Homer Neal
 */
public class KeysightDevice extends Device implements BiasDevice, PhotoDiodeDevice {

    public static final int
        CHAN_VOLTAGE = 0,
        CHAN_CURRENT = 1,
        NUM_CHANS = 2;
    private static final Map<String, KeysightB2981a.ConnType> connTypeMap = new HashMap<>();
    static {
        connTypeMap.put("SERIAL", KeysightB2981a.ConnType.SERIAL);
        connTypeMap.put("NET", KeysightB2981a.ConnType.NET);
    }

    @ConfigurationParameter(isFinal = true,
                            description = "connection type (NET or SERIAL)")
    private volatile String connType;
    @ConfigurationParameter(isFinal = true, description = "port Id")
    private volatile String devName;
    @ConfigurationParameter(isFinal = true,
                            description = "baud rate (0 for default)")
    private volatile int baudRate;


    private double[] runBias = new double[TSConfig.MAXSTATES]; // value overriden by groovy configuration

    private final KeysightB2981a pico = new KeysightB2981a();
    private KeysightB2981a.ConnType connTypeE;

    TSConfig cfg = null;
    TSState.pwrstates kstate = TSState.pwrstates.NOTCONFIGURED;

    IDataPointSetFactory dpsf = null;
    private double lastCurrent;
    private double lastVoltage;
    private int plotidx = 0;
    private boolean doPDPlot = false;

    /**
     * Performs initial configuration.
     */
    @Override
    protected void initDevice() {
        if (connType == null) {
            MonitorLogUtils.reportConfigError(log, name, "connType", "not specified");
        }
        connTypeE = connTypeMap.get(connType.toUpperCase());
        if (connTypeE == null) {
            MonitorLogUtils.reportConfigError(log, name, "connType", "is invalid");
        }
        if (devName == null) {
            MonitorLogUtils.reportConfigError(log, name, "devName", "not specified");
        }
        if (baudRate == 0) {
            log.info("Using default baud rate (" + KeysightB2981a.DEFAULT_BAUD + ")");
        }
        fullName = "Keysight KeysightB2981a (" + devName + ")";
    }

    /**
     * Performs full initialization - opens the connection
     */
    @Override
    protected void initialize() {
        try {
            pico.open(connTypeE, devName, baudRate); // open connection
            initSensors();
            kstate = TSState.pwrstates.OK;
            log.info("Connected to " + fullName);
            setOnline(true);
        } catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
        }
        inited = true;
    }

    /**
     * Closes the connection.
     */
    @Override
    protected void close() {
        try {
            pico.close();
            kstate = TSState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("Error disconnecting from " + fullName + ": " + e);
        }
    }

    /**
     * Checks a channel's parameters for validity.
     * 
     * @param name Channel name
     * @param hwChan Hardware channel number
     * @param type Channel type
     * @param subtype Channel subtype
     * @return Encoded type and subtype as an array
     * @throws Exception if parameter invalid
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception {
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            MonitorLogUtils.reportError(log, name, "hwChan", hwChan);
        }
        return new int[]{0, 0};
    }

    /**
     * Reads a channel.
     * 
     * @param chan
     * @param type
     * @return
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        //log.debug("KeysightDevice readChannel called! chan=" + chan + " type=" + type);
        if (chan == CHAN_VOLTAGE) {
            value = readVoltage();
            if (value < -998. || value > 9.8E36) {
                value = Double.NaN;
            }
//                kstate = TSState.pwrstates.ON;
        }
        else if (chan == CHAN_CURRENT) {
            value = readCurrent();
            if (value < -998. || value > 9.8E36) {
                value = Double.NaN;
            }
//                kstate = TSState.pwrstates.ON;
        }
        return value;
    }

    /**
     * Sets the configuration
     * @param cfg 
     */
    @Override
    public void setCfg(TSConfig cfg) {
        this.cfg = cfg;
    }

    /**
     * Sets the data point set factory address.
     * 
     * @param dpsf The data point set factory
     */
    @Override
    public void setDPSF(IDataPointSetFactory dpsf) {
        this.dpsf = dpsf;
    }

    /**
     * Sets ARM count
     * 
     * @param count The arm count
     * @throws DriverException
     */
    @Command(name = "setarmcount", description = "Set the ARM count")
    public void setArmCount(@Argument(name = "count", description = "ARM count") int count) throws DriverException {
        pico.setArmCount(count);
    }

    /**
     * Sets TRIGger count
     * 
     * @param value The count value
     * @throws DriverException
     */
    @Command(name = "settrigcount", description = "Set the trigger count")
    public void setTrigCount(@Argument(name = "value", description = "trigger count") int value) throws DriverException {
        pico.setTrigCount(value);
    }

    /**
     * Sets the Display ON(true)/OFF(false)
     *
     * @param dstate
     * @throws DriverException
     */
    @Override
    @Command(name = "setdisplay", description = "Set the display")
    public void setDisplay(@Argument(name = "OnOff", description = "true(ON)/false(OFF)") boolean dstate) throws DriverException {
        pico.setDisplay(dstate);
    }

    /**
     * Clears the data buffer
     *
     * @throws DriverException
     */
    @Command(name = "clrbuff", description = "Clears the Keysight read buffer")
    public void clearBuffer() throws DriverException {
        pico.clrBuff();
    }


    /**
     * Gets a set of current readings
     *
     * @param nreads
     * @param nplc
     * @throws DriverException
     * @return
     */
    @Command(type = CommandType.QUERY, name = "readCurrents", description = "Read a set of currents")
    public double[][] readCurrents(@Argument(name = "nreads", description = "#reads (armcount)") int nreads,
                                   @Argument(name = "nplc", description = "#Pulse Line Counts btwn reads (0.01->60.)") double nplc)
                      throws DriverException {
        return pico.readCurrents(nreads, nplc);
    }

    /**
     * Start the accumulation of readings into the device's internal buffer
     *
     * @param nreads
     * @param nplc
     * @return The expected time (secs)
     */
    @Override
    @Command(type = CommandType.QUERY, name = "accumBuffer", description = "start the buffer accumulation")
    public double accumBuffer(
            @Argument(name = "nreads", description = "#reads (armcount)") int nreads,
            @Argument(name = "nplc", description = "#Pulse Line Counts btwn reads (0.01->60.)") double nplc) {
        return accumBuffer(nreads, nplc, true);
    }
    /**
     * Start the accumulation of readings into the device's internal buffer
     *
     * @param nreads
     * @param nplc
     * @param nextrig
     * @param pause
     * @return The expected time (secs)
     */
    @Override
    @Command(type = CommandType.QUERY, name = "accumBuffer", description = "start the buffer accumulation")
    public double accumBufferEx(
            @Argument(name = "nreads", description = "#reads (armcount)") int nreads,
            @Argument(name = "nplc", description = "#Pulse Line Counts btwn reads (0.01->60.)") double nplc,
            @Argument(name = "nextrig", description = "the number of extra triggers") int nextrig,
            @Argument(name = "pause", description = "pause in seconds between triggers") double pause) {
        if ( cfg != null ) {
            cfg.setPDcnt(nreads);
            cfg.setPDnplc(nplc);
        }
        try {
            pico.accumBuffer(nreads, nplc, nextrig, pause);
        } catch (DriverException e) {
            log.error("Failed to prep device for doing accumulations.", e);
        }
        return nplc / 60.0 * nreads;
    }

    /**
     * Start the accumulation of readings into the device's internal buffer
     *
     * @param nreads
     * @param nplc
     * @param wait
     * @return The expected time (secs)
     */
    @Override
    @Command(type = CommandType.QUERY, name = "accumBuffer", description = "start the buffer accumulation")
    public double accumBuffer(
            @Argument(name = "nreads", description = "#reads (armcount)") int nreads,
            @Argument(name = "nplc", description = "#Pulse Line Counts btwn reads (0.01->60.)") double nplc,
            @Argument(name = "wait", description = "to wait (true) or not to (false)") boolean wait) {
        if ( cfg != null ) {
            cfg.setPDcnt(nreads);
            cfg.setPDnplc(nplc);
        }
        try {
            pico.accumBuffer(nreads, nplc);
        } catch (DriverException e) {
            log.error("Failed to prep device for doing accumulations.", e);
        }
        return nplc / 60.0 * nreads;
    }

    /**
     * Waits for accumulation to finish.
     * 
     * @param timeout
     * @throws DriverException
     */
    @Override
    @Command(type = CommandType.QUERY, name = "waitAccum", description = "wait for accumulation completion")
    public void waitAccum(@Argument(description = "Timeout (sec)") double timeout) throws DriverException {
        pico.waitAccum(timeout);
    }

    /**
     * Reads the data buffer
     * 
     * @return
     * @throws DriverException
     */
    @Override
    @Command(type = CommandType.QUERY, name = "readBuffer", description = "read the buffer")
    public double[][] readBuffer() throws DriverException {
        waitAccum(60.0);  // 1 minute should be enough
        double[][] buff = pico.readBuffer();

        if (isDoPDPlot()) {
            log.error("doing new aidaplots");
//        AidaPlots plots = new AidaPlots();
            KeyValueDataList plots = new KeyValueDataList();
            log.debug("declaring pdthst");
            plotidx = (plotidx % 3) + 1;
            IDataPointSet pdthst = dpsf.create("CurrentHist" + plotidx, "Current Reading vs. time(s) - Date:" + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                    System.currentTimeMillis())), 2);

            log.debug("filling pdthst");

            // Fill the two dimensional IDataPointSet
            for (int i = 0; i < buff[0].length; i++) {
                pdthst.addPoint();
//            pdthst.point(i).coordinate(0).setValue((double) i * nplc);
                pdthst.point(i).coordinate(0).setValue(buff[1][i]);
                pdthst.point(i).coordinate(0).setErrorPlus(0.);
                pdthst.point(i).coordinate(1).setValue(buff[0][i]);
                pdthst.point(i).coordinate(1).setErrorPlus(0.);
                pdthst.point(i).coordinate(1).setErrorMinus(0.);
            }

            try {
                log.debug("doing new addplots");

                //Fill the AidaPlot object with their XML representation
//            plots.addPlot("dataPointSet", XMLUtils.createXMLString((IManagedObject) pdthst));
                plots.addData("dataPointSet", XMLUtils.createXMLString((IManagedObject) pdthst));
            } catch (IOException ex) {
                log.error("Unable to create XML rep of AidPlot object!", ex);
            }

//Publish the plots somewhere in your subsystem
            log.info("publishing plot");

            KeyValueData kd = new KeyValueData("PDtimeHist", plots);
            s.publishSubsystemDataOnStatusBus(kd);
        }
//        getSubsystem().publishStatus("PDtimeHist", plots);

//        publish("photodiodebuff",buff);
        return buff;
    }

    /**
     * Reads buffer and dumps it to a file
     *
     * @param filename
     * @return
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, name = "readBuffer", description = "read the buffer and dump to file")
    public double[][] readBuffer(@Argument(name = "filename", description = "output filename for the Keysight buffered values") String filename)
            throws DriverException {
        double[][] buff = readBuffer();
        FileWriter fstream = null;
        File pdFl = new File(filename);
        try {
            if (pdFl.exists()) {
                pdFl.delete();
            }
            pdFl.createNewFile();
        } catch (Exception e) {
            log.error("Unable to create file (" + filename + ") for reason " , e);
        }
        try {
            fstream = new FileWriter(pdFl);
        } catch (IOException e) {
            log.error("Failed to open writer stream for file (" + filename + ") for reason " , e);
        }
        try {
            if (fstream != null) {
                log.info("writing file of buffered values from either Bias or PhotoDiode device");
//            BufferedWriter out = new BufferedWriter(new OutputStreamWriter(fstream));
                BufferedWriter out = new BufferedWriter(fstream);

                for (int i = 0; i < buff[0].length; i++) {
                    String line = buff[1][i] + " " + buff[0][i];
                    try {
                        out.write(line);
                        out.newLine();
                        out.flush();
                    } catch (IOException e) {
                        log.error("Error writing file: " + e);
                    }

                }
                try {
                    out.close();
                } catch (IOException e) {
                    log.error("Error closing file: " + e);
                }

            }
        } catch (Exception ee) {
            log.error("Failed to remove and create a new file: " + ee);
        }
        return buff;
    }

    /**
     * Reads buffer and dumps it to a remote file
     *
     * @param filename
     * @param host
     * @return
     * @throws IOException
     * @throws DriverException
     */
    @Command(type = CommandType.QUERY, name = "readBuffer", description = "read the buffer and dump to file")
    public double[][] readBuffer(@Argument(name = "filename", description = "output filename for the Keysight buffered values") String filename,
                                 @Argument(name = "host", description = "user@host") String host) throws IOException, DriverException {
          
        String fsplit[] = filename.split("/");
        String flocal = "/tmp/"+fsplit[fsplit.length-1];

        log.info("Storing temporary data at "+flocal+" then copying to "+filename);
        
        double[][] data=readBuffer(flocal);
        
        Runtime r = Runtime.getRuntime();
        
        String execStr = "scp -p "+flocal+" "+host+":"+filename ;
        
        log.info("Executing: " + execStr);
        Process p = r.exec(execStr);
    
        return data;
    }

    /**
     * Reads buffered data and returns it as a string
     * 
     * @return
     * @throws DriverException 
     */
    @Command(name = "readBufferStr", description = "read the buffer and returns as a string")
    public String readBufferStr() throws DriverException {
        pico.waitAccum(60.0);  // One minute should be enough
        double data[][] = pico.readBuffer();
        return "Data = " + getString(data[0]) + "\n"
                + " Times = " + getString(data[1]) + "\n";
    }

    /**
     * Sets the buffer size
     * 
     * @param value
     * @throws DriverException
     */
    @Override
    @Command(name = "setBuffSize", description = "Set the buffer size.")
    public void setBuffSize(@Argument(description = "buffer size") int value) throws DriverException {
        pico.setBuffSize(value);
    }

    /**
     * Sets the voltage range
     *
     * @param value
     * @throws DriverException
     */
    @Override
    @Command(name = "setVoltageRange", description = "Set the voltage range. Can be 10V, 50V or 500V")
    public void setVoltageRange(@Argument(description = "Voltage Range to set") double value) throws DriverException {
        pico.setVoltageRange(value);
    }

    /**
     * Sets the voltage.
     * 
     * @param value
     * @throws DriverException
     */
    @Override
    @Command(name = "setVoltage", description = "Set the voltage")
    public void setVoltage(@Argument(name = "value", description = "Voltage to set") double value) throws DriverException {
        if (kstate.equals(TSState.pwrstates.OK) || kstate.equals(TSState.pwrstates.OFF) || kstate.equals(TSState.pwrstates.ON)) {
            pico.setVoltage(value);
        } else {
            log.error("Unable to set output because state = " + kstate.toString());
        }
    }

    /**
     * Sets the channel.
     *
     * @param value
     */
    @Override
    @Command(name = "setChannel", description = "not implemented")
    public void setChannel(@Argument(name = "value", description = "Channel to select") int value) {
    }

    /**
     * Sets the voltage to the value needed for acquisition.
     *
     * @param cfgstate
     * @throws DriverException
     */
    @Override
    @Command(name = "setVoltageAcq", description = "Set the voltage to the acquisition set point")
    public void setVoltageAcq(int cfgstate) throws DriverException {
        rampVolts(5.0, getRunBias(cfgstate));
    }

    /**
     * Sets the 
     *
     * @param cfgstate
     * @throws DriverException
     */
    @Override
    @Command(name = "setAvgOn", description = "Control whether the averaging is on")
    public void setAvgOn(boolean doAvg) throws DriverException {
      pico.setDo_avg(doAvg);
    }
    
    /**
     * Sets the 
     *
     * @param cfgstate
     * @throws DriverException
     */
    @Override
    @Command(name = "setNAvg", description = "Controls have many samples to average over")
    public void setNAvg(int nAvg) throws DriverException {
      pico.setnAvg(nAvg);
    }
    
    /**
     * Sets the 
     *
     * @param cfgstate
     * @throws DriverException
     */
    @Override
    @Command(name = "getAvgState", description = "true = averaging on")
    public String getAvgState() throws DriverException {
      return(pico.isDo_avg());
    }
    
    
    
    /**
     * Ramps to the desired voltage
     *
     * @param duration
     * @param value
     * @throws DriverException
     */
    @Override
    @Command(type = CommandType.QUERY, name = "rampVolts", description = "ramp the voltage")
    public void rampVolts(@Argument(description = "number of seconds to ramp from current voltage to desired voltage") double duration,
                          @Argument(description = "Voltage to ramp to") double value) throws DriverException {
        pico.rampVolts(duration, value);
    }

    /**
     * Ramps to the desired voltage over a given time and for nsteps
     *
     * @param duration
     * @param value
     * @param nsteps
     * @throws DriverException
     */
    @Override
    @Command(type = CommandType.QUERY, name = "rampVolts", description = "ramp the voltage")
    public void rampVolts(@Argument(name = "duration", description = "number of seconds to ramp from current voltage to desired voltage") double duration,
            @Argument(name = "value", description = "Voltage to ramp to") double value,
            @Argument(name = "nsteps", description = "number of steps") int nsteps) throws DriverException {
        pico.rampVolts(duration, value, nsteps);
    }

    /**
     * Gets the last current measurement
     * 
     * @return The current
     */
    @Override
    @Command(description = "return last current measurement")
    public double getLastCurrent() {
        return lastCurrent;
    }

    @Deprecated
    @Command(description = "return last current measurement")
    public double getLast_curr() {
        return getLastCurrent();
    }

    /**
     * Gets the last voltage measurement
     * 
     * @return The voltage
     */
    @Override
    @Command(description = "return last voltage measurement")
    public double getLastVoltage() {
        return lastVoltage;
    }

    @Deprecated
    @Command(description = "return last voltage measurement")
    public double getLast_volt() {
        return getLastVoltage();
    }

    /**
     * Shows the set voltage.
     *
     * @return
     * @throws DriverException
     */
    @Command(description = "Show the set voltage")
    public double getVoltage() throws DriverException {
        return pico.getVoltage();
    }

    /**
     * Reads the actual voltage.
     *
     * @return
     */
    @Override
    @Command(type = CommandType.QUERY, description = "Read the actual voltage")
    public double readVoltage() {
        double volt = 0.0;
        try {
            if (pico.isOkToTalk()) {
                volt = pico.readVoltage();
                lastVoltage = volt;
                kstate = TSState.pwrstates.ON; // indicate successful communication
            } else {
                //System.out.print("x");  ???
                volt = lastVoltage; // see readCurrent remark
            }
        } catch (DriverException e) {
            kstate = TSState.pwrstates.NOTCONFIGURED; // indicate a problem
            log.error("Keysight driver failed to read voltage!");
        }
        return volt;
    }

    /**
     * Sets the current range
     *
     * @param value The current range
     * @throws DriverException
     */
    @Override
    @Command(name = "setCurrentRange", description = "Set the current range")
    public void setCurrentRange(@Argument(description = "Current Range to set") double value) throws DriverException {
        pico.setCurrentRange(value);
    }

    /**
     * Zero correct the current
     *
     * @throws DriverException
     */
    @Command(name = "zeroCorrectCurrent", description = "Zero correct the current")
    public void zeroCorrectCurrent() throws DriverException {
        pico.zeroCorrectCurrent();
    }

    /**
     * Reads the actual current.
     *
     * @return
     */
    @Override
    @Command(type = CommandType.QUERY, name = "readCurrent", description = "Read the actual current")
    public double readCurrent() {
        double curr = 0.0;
        try {
            if (pico.isOkToTalk()) {
                curr = pico.readCurrent();
                lastCurrent = curr;
                kstate = TSState.pwrstates.ON; // indicate successful communication
            } else {
                //System.out.print("x"); ???
                curr = lastCurrent; // I want this to return a reception but there is a lot to change before I can do that
            }
        } catch (DriverException e) {
            kstate = TSState.pwrstates.NOTCONFIGURED; // indicate a problem
            log.error("Keysight driver failed to retrieve the current measurement!");
        }
        return curr;
    }

    /**
     * Tells whether PD accumulation is in progress
     *
     * @return
     */
    @Override
    @Command(type = CommandType.QUERY, name = "isAccumInProgress", description = "Are the current readings being accumulated")
    public boolean isAccumInProgress() {
        return pico.isAccumInProgress();
    }

    /**
     * Sets the soft current limit.
     *
     * @param value The limit value
     * @throws DriverException
     */
    @Override
    @Command(name = "setCurrentLimit", description = "Set the soft current limit")
    public void setCurrentLimit(@Argument(description = "Current limit to set") double value) throws DriverException {
        pico.setCurrentLimit(value);
    }

    /**
     * Shows the soft current limit.
     *
     * @return The limit
     * @throws DriverException
     */
    @Override
    @Command(type = CommandType.QUERY, name = "showcurrlim", description = "Show the soft current limit")
    public String getCurrentLimit() throws DriverException {
        return "Current limit = " + pico.getCurrentLimit();
    }

    /**
     * Sets the read rate in PLC
     *
     * @param rate The inverse rate, in clock ticks
     * @throws DriverException
     */
    @Override
    @Command(name = "setRate", description = "sets the read rate in plc")
    public void setRate(@Argument(description = "number of PLC") double rate) throws DriverException {
        pico.setRate(rate);
    }

    /**
     * Turns the output on or off.
     *
     * @param state Output state (on if 1, off otherwise)
     * @throws DriverException
     */
    @Override
    @Command(name = "setOutput", description = "Turn output on or off. 1 ==> ON")
    public void setOutput(@Argument(description = "Output state: on or off") int state) throws DriverException {
        if (kstate.equals(TSState.pwrstates.OK) || kstate.equals(TSState.pwrstates.OFF) || kstate.equals(TSState.pwrstates.ON)) {
                pico.setOutput(state == 1);
        } else {
            log.error("Unable to set output because state = " + kstate.toString());
        }
    }

    /**
     * Shows the output state.
     *
     * @return The output state (true = on, false = off)
     * @throws DriverException
     */
    @Override
    @Command(type = CommandType.QUERY, name = "showoutput", description = "Show the output state")
    public boolean showOutput() throws DriverException {
        return pico.getOutput();
    }

    /**
     * Converts an array of numbers to a string.
     *
     */
    StringBuilder getString(double[] values) {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }

    /**
     * Converts a boolean to on/off.
     */
    private String getOnOff(boolean on) {
        return on ? "on" : "off";
    }

    /**
     * Sets the timeout
     * 
     * @param nsec Timeout (secs)
     */
    @Override
    @Command(name = "setTimeout", description = "set timeout for Keysight SCPI communication")
    public void setTimeout(@Argument(name = "nsec",
            description = "length of timeout in seconds") double nsec) {
        try {
            pico.setTimeout(nsec);
        } catch (DriverException ex) {
            log.error("Failed to set keysight scpi timeout:" , ex);
        }
    }

    /**
     * Resets the Keysight to a known default state
     */
    @Override
    @Command(name = "reset", description = "reset trips etc...")
    public void reset() {
        if (!kstate.equals(TSState.pwrstates.TRIPPED)) {
            try {
                pico.reset();
                pico.setTrip(false); // Does nothing
                kstate = TSState.pwrstates.ON;
            } catch (Exception ex) {
                log.error("failed attempt to reset device:", ex);
                kstate = TSState.pwrstates.NOTCONFIGURED;
            }
        } else {
            log.error("Unable to set output because state = " + kstate.toString()
                    + "MUST USE SUBSYSTEM resetTrip COMMAND TO RECOVER");
        }

        kstate = TSState.pwrstates.OFF;

    }

    /**
     * Resets the Keysight to a known default state w/o initializing all settings
     */
    @Override
    @Command(name = "softReset", description = "soft reset etc...")
    public void softReset() {
        if (!kstate.equals(TSState.pwrstates.TRIPPED)) {
            try {
                pico.softReset();
                log.info("Keysight soft reset performed.");
                pico.setTrip(false);
                pico.setOKTOTALK(true);
                kstate = TSState.pwrstates.ON;
            } catch (Exception ex) {
                log.error("failed attempt to softreset device:", ex);
                kstate = TSState.pwrstates.NOTCONFIGURED;
            }
        } else {
            log.error("Unable to set output because state = " + kstate.toString()
                    + "MUST USE SUBSYSTEM resetTrip COMMAND TO RECOVER");
        }

    }

    /**
     * Sets the bias value for acquisition
     * 
     * @param runBias
     * @param cfgstate
     */
    @Override
    @Command(name = "setrunbias", description = "Sets the bias value for acquisition")
    public void setRunBias(double runBias, int cfgstate) {
        this.runBias[cfgstate] = runBias;
    }

    /**
     * Returns the bias value for acquisition
     * 
     * @param cfgstate
     * @return The run bias
     */
    @Override
    @Command(type = CommandType.QUERY, name = "getRunBias", description = "Returns the bias value for acquisition")
    public double getRunBias(int cfgstate) {
        return runBias[cfgstate];
    }

    /**
     * Sets the abort state
     * 
     * @param state 
     */
    @Override
    @Command(name = "abort", description = "abort or clear abort for any long operation")
    public void setAbort(boolean state) {
        pico.setAbort(state);  // Does nothing
    }

    /**
     * Sets the state
     * 
     * @param istate 
     */
    @Override
    @Command(name = "setstate", description = "set keysight device status")
    public void setState(int istate) {
        if (istate == TSState.pwrstates.TRIPPED.ordinal()) {
            pico.setTrip(true);  // Does nothing
        }
        kstate = TSState.pwrstates.values()[istate];
    }

    /**
     * Gets the state
     * 
     * @return The ordinal of the state
     */
    @Override
    @Command(type = CommandType.QUERY, name = "getState", description = "get keysight device status")
    public int getState() {
        return (kstate.ordinal());
    }

    /**
     * Gets whether PD plots are produced
     * 
     * @return 
     */
    public boolean isDoPDPlot() {
        return doPDPlot;
    }

    /**
     * Sets whether PD plots are produced
     * 
     * @param doPDPlot Whether plots are produced
     */
    @Override
    @Command(description = "set whether PD plots are produced")
    public void setDoPDPlot(@Argument(description = "true for do plots") boolean doPDPlot) {
        this.doPDPlot = doPDPlot;
    }

    /**
     * Writes a command to the Keysight
     * 
     * @param str 
     * @throws DriverException
     */
    @Override
    @Command(name = "send", description = "write a string to the device")
    public void send(String str) throws DriverException {
        pico.writeKthly(str);
    }

    /**
     * Writes a command to the Keysight
     * 
     * @param str1
     * @param str2
     * @throws DriverException
     */
    @Override
    @Command(name = "send", description = "write a string to the device")
    public void send(String str1, String str2) throws DriverException {
        pico.writeKthly(str1 + " " + str2);
    }

    /**
     * Writes a command to the Keysight and returns response
     * 
     * @param str
     * @return the response
     * @throws DriverException
     */
    @Override
    @Command(name = "query", description = "write a command and read response")
    public String query(String str) throws DriverException {
        return pico.readStringKthly(str);
    }

    /**
     * Writes a command to the Keysight and returns response
     * 
     * @param str1
     * @param str2
     * @return the response
     * @throws DriverException
     */
    @Override
    @Command(name = "query", description = "write a command and read response")
    public String query(String str1, String str2) throws DriverException {
        return pico.readStringKthly(str1 + " " + str2);
    }

}
