package org.lsst.ccs.subsystem.teststand;

import java.time.Duration;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Predicate;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.devices.power.distribution.APC7900Device;
import org.lsst.ccs.subsystem.power.states.RebPowerState;
import org.lsst.ccs.subsystem.teststand.data.TS7Outlets;
import org.lsst.ccs.subsystem.teststand.data.TestStandAgentProperties;
import org.lsst.ccs.subsystem.teststand.limits.ChannelLimits;
import org.lsst.ccs.subsystem.teststand.limits.LimitAlgorithm;
import org.lsst.ccs.subsystem.teststand.limits.Limits;
import org.lsst.ccs.subsystem.teststand.limits.TransitionAlgorithm;
import org.lsst.ccs.subsystem.teststand.states.ThermalState;
import org.lsst.ccs.subsystem.teststand.states.ThermalTarget;
import org.lsst.ccs.subsystem.teststand.states.VacuumState;
import org.lsst.ccs.subsystem.teststand.states.VacuumTarget;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Implementation of SLAC TS7
 *
 * @author The LSST CCS Team
 */
public class TS7 implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    CryoDevice cryoDevc;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    VQMDevice vqmDevc;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    protected final Map<String, APC7900Device> pduDevicesMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    AP9630UPSDevice upsDevc;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configurationService;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    ThermalConfiguration thermalConfiguration;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    VacuumConfiguration vacuumConfiguration;

    @ConfigurationParameter(isFinal = true)
    private String rebPsSubsystem = "ccs-rebps";

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService als;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;

    // NOTE: These variables are magically set from the constructor in the groovy file
    // Although netbeans may thing otherwise they cannot be marked final
    private Channel coldPlate1;
    private Channel coldPlate2;
    private Channel cryoPlate;
    private Channel pressureDevice;

    private final Logger LOGGER = Logger.getLogger(getClass().getPackage().getName());

    private final Object initializationLock = new Object();
    private boolean initializedRebPowerState = false;
    private RebPowerState currentRebsPowerState = RebPowerState.OFF;
    private TransitionAlgorithm vacuumTransitionAlgorithm;
    private TransitionAlgorithm coldTransitionAlgorithm1;
    private TransitionAlgorithm coldTransitionAlgorithm2;
    private TransitionAlgorithm cryoTransitionAlgorithm;

    private volatile boolean alertResponseEnabled = false;
    private UPSMonitor upsMon;

    /**
     * Post-initialization
     */
    @Override
    public void postInit() {
        if (cryoDevc == null) {
            //What do you want to do if cryo is not present?
            LOGGER.warning("Cryo device not defined");
        }

        
        // By setting TESTSTAND_TYPE_AGENT_PROPERTY we signal to consoles that this subsystem is compatible with the teststand subsystm GUI
        subsys.getAgentService(AgentPropertiesService.class).setAgentProperty(TestStandAgentProperties.TESTSTAND_TYPE_AGENT_PROPERTY, TS7.class.getCanonicalName());
        
        // Done in one call to avoid multiple state changes.
        agentStateService.updateAgentState(ThermalState.UNKNOWN, ThermalTarget.UNKNOWN, VacuumState.UNKNOWN, VacuumTarget.UNKNOWN);
        // Set up timer to handle transitions
        upsMon = new UPSMonitor(this, upsDevc, LOGGER, pts, als);
    }

    @Override
    public void build() {
        pts.scheduleAgentPeriodicTask(new AgentPeriodicTask("transition-timer", () -> {
            updateThermalLimits();
            updatePressureLimits();
        }));

        pts.scheduleAgentPeriodicTask(new AgentPeriodicTask("ups-timer", () -> {
            upsMon.monitorUPS();
        }));
        pts.setPeriodicTaskPeriod("ups-timer", Duration.ofSeconds(1));
    }
    


    /**
     * Allows the thermal goal to be set.
     *
     * @param goal The new thermal target
     */
    @Command(type = Command.CommandType.ACTION, description = "Change the thermal goal for the subsystem")
    public void setThermalGoal(ThermalTarget goal) {
        if (!agentStateService.isInState(goal) || goal == ThermalTarget.CURRENT) {
            // Check if the transition is legal
            if (goal == ThermalTarget.UNKNOWN) {
                throw new IllegalArgumentException("Cannot transition to " + goal);
            }
            LOGGER.info("Switching to thermal goal " + goal);
            
            if ( !agentStateService.isComponentInState(coldPlate1.getName(), DataProviderState.NOMINAL) ||         
                    ! agentStateService.isComponentInState(coldPlate2.getName(), DataProviderState.NOMINAL) ||
                    ! agentStateService.isComponentInState(cryoPlate.getName(), DataProviderState.NOMINAL) ) {
                throw new IllegalArgumentException("Cannot start a Thermal transition when the temperatures are not NOMINAL.");                        
            }

            
            final ChannelLimits coldPlateLimits1 = new ChannelLimits(coldPlate1);
            final ChannelLimits coldPlateLimits2 = new ChannelLimits(coldPlate2);
            final ChannelLimits cryoPlateLimits = new ChannelLimits(cryoPlate);
            
            Limits coldPlate1InitialLimits, coldPlate2InitialLimits, cryoPlateInitialLimits;
            double coldPlate1NominalInitialValue, coldPlate2NominalInitialValue, cryoPlateNominalInitialValue;
            if ( agentStateService.isInState(ThermalState.AT_TARGET) ) {
                //If we are at target use the values stored in the thermalConfiguration
                coldPlate1InitialLimits = new Limits(thermalConfiguration.coldSetPointLimits);
                coldPlate2InitialLimits = new Limits(thermalConfiguration.coldSetPointLimits);
                cryoPlateInitialLimits = new Limits(thermalConfiguration.cryoSetPointLimits);
                coldPlate1NominalInitialValue = thermalConfiguration.coldSetPoint;
                coldPlate2NominalInitialValue = thermalConfiguration.coldSetPoint;
                cryoPlateNominalInitialValue = thermalConfiguration.cryoSetPoint;
            } else {
                //Otherwise use the current values
                coldPlate1InitialLimits = new Limits(coldPlateLimits1);
                coldPlate2InitialLimits = new Limits(coldPlateLimits2);
                cryoPlateInitialLimits = new Limits(cryoPlateLimits);
                coldPlate1NominalInitialValue = coldPlate1.getValue();
                coldPlate2NominalInitialValue = coldPlate2.getValue();
                cryoPlateNominalInitialValue = cryoPlate.getValue();                
            }
            
            if ( goal == ThermalTarget.CURRENT ) {    
                String thermalConfigurationName = subsys.getComponentLookup().getNameOfComponent(thermalConfiguration);
                configurationService.submitChanges(thermalConfigurationName, getCurrentThermalConfigurationParameters());
                configurationService.commitBulkChange();
            } else {            
                if ( agentStateService.isInState(ThermalTarget.UNKNOWN) ) {
                    throw new IllegalArgumentException("Cannot initiate a transition to "+goal+" from "+ThermalTarget.UNKNOWN+"."
                            + " You must first set the ThermalTarget to "+ThermalTarget.CURRENT);
                }
                configurationService.loadCategories("thermal:" + goal.name().toLowerCase());
            }

            long time = System.currentTimeMillis();

            LimitAlgorithm coldLimitsAlgorithm1 = thermalConfiguration.createColdLimitsAlgorithm(coldPlate1NominalInitialValue, coldPlate1InitialLimits);
            coldTransitionAlgorithm1 = thermalConfiguration.createColdTransitionAlgorithm(coldLimitsAlgorithm1, coldPlate1.getValue(), coldPlateLimits1, time);

            LimitAlgorithm coldLimitsAlgorithm2 = thermalConfiguration.createColdLimitsAlgorithm(coldPlate2NominalInitialValue, coldPlate2InitialLimits);
            coldTransitionAlgorithm2 = thermalConfiguration.createColdTransitionAlgorithm(coldLimitsAlgorithm2, coldPlate2.getValue(), coldPlateLimits2, time);

            LimitAlgorithm cryoLimitsAlgorithm = thermalConfiguration.createCryoLimitsAlgorithm(cryoPlateNominalInitialValue, cryoPlateInitialLimits);
            cryoTransitionAlgorithm = thermalConfiguration.createCryoTransitionAlgorithm(cryoLimitsAlgorithm, cryoPlate.getValue(), coldPlateLimits2, time);

            agentStateService.updateAgentState(goal, ThermalState.IN_TRANSITION);
            updateThermalLimits();
            // TODO: Eventually we should actually set the setPoints
        }
    }

    /**
     * Allows the vacuum goal to be set.
     *
     * @param goal The new vacuum target
     */
    @Command(type = Command.CommandType.ACTION, description = "Change the thermal goal for the subsystem")
    public void setVacuumGoal(VacuumTarget goal) {
        if (!agentStateService.isInState(goal) || goal == VacuumTarget.CURRENT) {
            // Check if the transition is legal
            if (goal == VacuumTarget.UNKNOWN) {
                throw new IllegalArgumentException("Cannot transition to " + goal);
            }
            if ( !agentStateService.isComponentInState(pressureDevice.getName(), DataProviderState.NOMINAL) ) {
                throw new IllegalArgumentException("Cannot start a Vacuum transition when the pressure is not NOMINAL.");                        
            }

            final ChannelLimits pressureLimits = new ChannelLimits(pressureDevice);
            
            Limits pressureInitialLimits;
            double pressureNominalInitialValue;
            if ( agentStateService.isInState(VacuumState.AT_TARGET) ) {
                //If we are at target use the values stored in the vacuumConfiguration
                pressureInitialLimits = new Limits(vacuumConfiguration.pressureSetPointLimits);
                pressureNominalInitialValue = vacuumConfiguration.pressureSetPoint;
            } else {
                //Otherwise use the current values
                pressureInitialLimits = new Limits(pressureLimits);
                pressureNominalInitialValue = pressureDevice.getValue();
            }
            
            if ( goal == VacuumTarget.CURRENT ) {    
                String vacuumConfigurationName = subsys.getComponentLookup().getNameOfComponent(vacuumConfiguration);
                configurationService.submitChanges(vacuumConfigurationName, getCurrentVacuumConfigurationParameters());
                configurationService.commitBulkChange();
            } else {            
                if ( agentStateService.isInState(VacuumTarget.UNKNOWN) ) {
                    throw new IllegalArgumentException("Cannot initiate a transition to "+goal+" from "+VacuumTarget.UNKNOWN+"."
                            + " You must first set the VacuumTarget to "+VacuumTarget.CURRENT);
                }
                configurationService.loadCategories("vacuum:" + goal.name().toLowerCase());
            }
            
            LimitAlgorithm pressureLimitsAlgorithm = vacuumConfiguration.createVacuumLimitsAlgorithm(pressureNominalInitialValue, pressureInitialLimits);
            vacuumTransitionAlgorithm = vacuumConfiguration.createVacuumTransitionAlgorithm(pressureLimitsAlgorithm, pressureDevice.getValue(), pressureLimits, System.currentTimeMillis());
            
            agentStateService.updateAgentState(goal, VacuumState.IN_TRANSITION);
            updatePressureLimits();
            // TODO: Eventually we should actually control the pumps
        }
    }

    /**
     * Method called periodically to update limits in transition
     */
    private synchronized void updateThermalLimits() {
        long time = System.currentTimeMillis();
        if (agentStateService.isInState(ThermalState.IN_TRANSITION)) {
            boolean allAtTarget = true;
            double dataPoint = coldPlate1.getValue();
            boolean atTarget = coldTransitionAlgorithm1.isAtTarget(dataPoint, time);
            if (!atTarget) {
                coldTransitionAlgorithm1.adjustLimits(dataPoint, time);
                allAtTarget = false;
            } else {
                coldTransitionAlgorithm1.completeTransition(time);
            }
            LOGGER.info("Update: Cold plate 1 state is " + (atTarget ? "in transition" : "at target") + " current limits are " + coldTransitionAlgorithm1.getLimits());
            dataPoint = coldPlate2.getValue();
            atTarget = coldTransitionAlgorithm2.isAtTarget(dataPoint, time);
            if (!atTarget) {
                coldTransitionAlgorithm2.adjustLimits(dataPoint, time);
                allAtTarget = false;
            } else {
                coldTransitionAlgorithm2.completeTransition(time);
            }
            LOGGER.info("Update: Cold plate 2 state is " + (atTarget ? "in transition" : "at target") + " current limits are " + coldTransitionAlgorithm2.getLimits());
            dataPoint = cryoPlate.getValue();
            atTarget = cryoTransitionAlgorithm.isAtTarget(dataPoint, time);
            if (!atTarget) {
                cryoTransitionAlgorithm.adjustLimits(dataPoint, time);
                allAtTarget = false;
            } else {
                cryoTransitionAlgorithm.completeTransition(time);
            }
            if (allAtTarget) agentStateService.updateAgentState(ThermalState.AT_TARGET);
        } else {
            ThermalState state = getCurrentThermalState();
            agentStateService.updateAgentState(state);
        }
    }
    
    private ThermalState getCurrentThermalState() {
        ThermalState state = ThermalState.UNKNOWN;
        if ( coldTransitionAlgorithm1 != null && coldTransitionAlgorithm2 != null && cryoTransitionAlgorithm != null ) {
            state = ThermalState.AT_TARGET;
            long time = System.currentTimeMillis();
            double dataPoint = coldPlate1.getValue();
            if( ! coldTransitionAlgorithm1.isAtTarget(dataPoint, time) ) {
                return ThermalState.OFF_TARGET;
            }
            dataPoint = coldPlate2.getValue();
            if( ! coldTransitionAlgorithm2.isAtTarget(dataPoint, time) ) {
                return ThermalState.OFF_TARGET;
            }
            dataPoint = cryoPlate.getValue();
            if( ! cryoTransitionAlgorithm.isAtTarget(dataPoint, time) ) {
                return ThermalState.OFF_TARGET;
            }            
        }
        return state;
    }
    
    private synchronized void updatePressureLimits() {
        long time = System.currentTimeMillis();
        if (agentStateService.isInState(VacuumState.IN_TRANSITION)) {
            double dataPoint = pressureDevice.getValue();
            boolean atTarget = vacuumTransitionAlgorithm.isAtTarget(dataPoint, time);
            if (!atTarget) {
                vacuumTransitionAlgorithm.adjustLimits(dataPoint, time);
            } else {
                vacuumTransitionAlgorithm.completeTransition(time);
                agentStateService.updateAgentState(VacuumState.AT_TARGET);
            }
        } else {
            VacuumState state = getCurrentVacuumState();
            agentStateService.updateAgentState(state);            
        }
    }

    private VacuumState getCurrentVacuumState() {
        VacuumState state = VacuumState.UNKNOWN;
        if ( vacuumTransitionAlgorithm != null ) {
            state = VacuumState.AT_TARGET;
            long time = System.currentTimeMillis();
            double dataPoint = pressureDevice.getValue();
            if( ! vacuumTransitionAlgorithm.isAtTarget(dataPoint, time) ) {
                return VacuumState.OFF_TARGET;
            }
        }
        return state;
    }


    /**
     * Post start processing
     */
    @Override
    public void postStart() {
        Predicate<BusMessage<? extends Serializable, ?>> filter
                = BusMessageFilterFactory.messageOrigin(rebPsSubsystem);

        subsys.getMessagingAccess().addStatusMessageListener((msg) -> handlePowerSupplyMessage(msg), filter);
    }

    /**
     * Gets the list of PDU names.
     *
     * @return The list of names
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the list of PDU names")
    public List<String> getPduNames() {
        return new ArrayList<>(pduDevicesMap.keySet());
    }

    /**
     * Gets the list of PDU outlet names.
     *
     * @return The list of names (of the form "pdu/outlet")
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the list of PDU outlet names")
    public List<String> getOutletNames() {
        List names = new ArrayList<>();
        for (String pduName : pduDevicesMap.keySet()) {
            for (String outletName : pduDevicesMap.get(pduName).getOutletNames()) {
                names.add(pduName + "/" + outletName);
            }
        }
        return names;
    }

    /**
     * Gets the map of PDU outlet on states.
     *
     * @return The map of names (of the form "pdu/outlet") to on states
     * (true/false)
     * @throws DriverException
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the map of PDU outlet on states")
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException {
        Map states = new HashMap<>();
        for (String pduName : pduDevicesMap.keySet()) {
            Map<String, Boolean> pduStates = pduDevicesMap.get(pduName).getOutletOnStateMap();
            for (String outletName : pduStates.keySet()) {
                states.put(pduName + "/" + outletName, pduStates.get(outletName));
            }
        }
        return states;
    }

    /**
     * Handles REB PS state messages. NOTICE: This method is executed on the
     * Message dispatch thread. It should be light weight.
     */
    private void handlePowerSupplyMessage(StatusMessage msg) {
        final StateBundle sb;
        synchronized (initializationLock) {
            if (initializedRebPowerState && msg instanceof StatusStateChangeNotification) {
                StatusStateChangeNotification ssn = (StatusStateChangeNotification) msg;
                sb = ssn.getNewState();
            } else if (!initializedRebPowerState) {
                sb = msg.getState();
            } else {
                sb = null;
            }
            if (sb != null) {
                RebPowerState powerState = (RebPowerState) sb.getState(RebPowerState.class);
                if (powerState != null) {
                    initializedRebPowerState = true;
                }
                if (currentRebsPowerState != powerState) {
                    currentRebsPowerState = powerState;
                    // TODO: We should have some support in core for handling firing off actions
                    new Thread(() -> updatePowerState(currentRebsPowerState, sb.getComponentsWithState(RebPowerState.class))).start();
                }
            }

        }

    }

    /**
     * Gets the current REB power state
     * 
     * @return The REB power state
     */
    public RebPowerState getCurrentRebsPowerState() {
        return currentRebsPowerState;
    }

    private void updatePowerState(RebPowerState state, Map<String, RebPowerState> map) {
        String rebStateStr = "";
        for (Entry<String, RebPowerState> e : map.entrySet()) {
            rebStateStr += e.getKey() + "=" + e.getValue() + " ";
        }
        LOGGER.info("Updating power state to " + state + " " + rebStateStr);
        // This no longer does anything, since limits no longer depend on power state
    }

    @Command(type = Command.CommandType.ACTION, description = "Turn Off named outlet")
    public void turnOutletOff(TS7Outlets outlet) throws DriverException {
        changeOutletState(outlet, false);
    }

    @Command(type = Command.CommandType.ACTION, description = "Turn On named outlet")
    public void turnOutletOn(TS7Outlets outlet) throws DriverException {
        changeOutletState(outlet, true);
    }

    private void changeOutletState(TS7Outlets outlet, boolean on) throws DriverException {
        String outletName = outlet.getOutletName();
        for (APC7900Device pdu : pduDevicesMap.values()) {
            if (pdu.getOutletNames().contains(outletName)) {
                if (on) {
                    pdu.forceOutletOn(outletName);
                } else {
                    pdu.forceOutletOff(outletName);
                }
                return;
            }
        }
        throw new IllegalArgumentException("Could not find device to turn off outlet " + outletName);
    }

    @Command(type = Command.CommandType.ACTION, description = "Enable/disable alert response")
    public void enableAlertResponse(boolean enable) {
        this.alertResponseEnabled = enable;
    }

    @Command(type = Command.CommandType.QUERY, description = "Get true/false if the alert response is enabled")
    public boolean isAlertResponseEnabled() {
        return alertResponseEnabled;
    }
        
    private Map getCurrentThermalConfigurationParameters() {
        Map changes = new HashMap();
        changes.put("coldSetPoint", (coldPlate1.getValue() + coldPlate2.getValue()) / 2.0);
        changes.put("cryoSetPoint", cryoPlate.getValue());

        double coldPlateLimitLow = Math.min(coldPlate1.getLimitLo(), coldPlate2.getLimitLo());
        double coldPlateWarningLow = coldPlateLimitLow + Math.min(coldPlate1.getDbandLo(), coldPlate2.getDbandLo());
        double coldPlateLimitHigh = Math.max(coldPlate1.getLimitHi(), coldPlate1.getLimitHi());
        double coldPlateWarningHigh = coldPlateLimitHigh - Math.min(coldPlate1.getDbandHi(), coldPlate2.getDbandHi());

        List<Double> coldSetPointLimits = new ArrayList<>();
        //These have to be in the right order!!!
        coldSetPointLimits.add(coldPlateLimitLow);
        coldSetPointLimits.add(coldPlateWarningLow);
        coldSetPointLimits.add(coldPlateWarningHigh);
        coldSetPointLimits.add(coldPlateLimitHigh);

        double cryoPlateLimitLow = cryoPlate.getLimitLo();
        double cryoPlateWarningLow = cryoPlateLimitLow + cryoPlate.getDbandLo();
        double cryoPlateLimitHigh = cryoPlate.getLimitHi();
        double cryoPlateWarningHigh = cryoPlateLimitHigh - cryoPlate.getDbandHi();

        List<Double> cryoSetPointLimits = new ArrayList<>();
        //These have to be in the right order!!!
        cryoSetPointLimits.add(cryoPlateLimitLow);
        cryoSetPointLimits.add(cryoPlateWarningLow);
        cryoSetPointLimits.add(cryoPlateWarningHigh);
        cryoSetPointLimits.add(cryoPlateLimitHigh);

        changes.put("coldSetPointLimits", coldSetPointLimits);
        changes.put("cryoSetPointLimits", cryoSetPointLimits);

        changes.put("coldMonitoringLimitAlgorithm", "DELTA");
        changes.put("cryoMonitoringLimitAlgorithm", "DELTA");

        changes.put("coldMonitoringLimitAlgorithmParameters", new ArrayList<>());
        changes.put("cryoMonitoringLimitAlgorithmParameters", new ArrayList<>());

        changes.put("coldMonitoringTransitionAlgorithm", "RATCHET");
        changes.put("cryoMonitoringTransitionAlgorithm", "RATCHET");

        changes.put("coldMonitoringTransitionAlgorithmParameters", new ArrayList<>());
        changes.put("cryoMonitoringTransitionAlgorithmParameters", new ArrayList<>());
        return changes;
    }
    
    private Map getCurrentVacuumConfigurationParameters() {
        Map changes = new HashMap();
        changes.put("pressureSetPoint", pressureDevice.getValue());

        double pressurePlateLimitLow = pressureDevice.getLimitLo();
        double pressurePlateWarningLow = pressurePlateLimitLow + pressureDevice.getDbandLo();
        double pressurePlateLimitHigh = pressureDevice.getLimitHi();
        double pressurePlateWarningHigh = pressurePlateLimitHigh - pressureDevice.getDbandHi();

        List<Double> pressureSetPointLimits = new ArrayList<>();
        //These have to be in the right order!!!
        pressureSetPointLimits.add(pressurePlateLimitLow);
        pressureSetPointLimits.add(pressurePlateWarningLow);
        pressureSetPointLimits.add(pressurePlateWarningHigh);
        pressureSetPointLimits.add(pressurePlateLimitHigh);

        changes.put("pressureSetPointLimits", pressureSetPointLimits);

        changes.put("pressureMonitoringLimitAlgorithm", "PROPORTIONAL");

        changes.put("pressureMonitoringLimitAlgorithmParameters", new ArrayList<>());

        changes.put("pressureMonitoringTransitionAlgorithm", "RATCHET");

        changes.put("pressureMonitoringTransitionAlgorithmParameters", new ArrayList<>());
        return changes;
    }
}
