package org.lsst.ccs.subsystem.teststand.limits;

import java.util.List;


class DeltaLimitAlgorithm extends LimitAlgorithm {

    private double initialDeltaAlarmLow;
    private double initialDeltaWarnLow;
    private double initialDeltaAlarmHigh;
    private double initialDeltaWarnHigh;
    private double targetDeltaAlarmLow;
    private double targetDeltaWarnLow;
    private double targetDeltaAlarmHigh;
    private double targetDeltaWarnHigh;

    @Override
    public void init(List<Double> parameters, double initialValue, LimitsInterface limits, double targetValue, LimitsInterface targetLimits) {
        super.init(parameters, initialValue, limits, targetValue, targetLimits);
        // Parameters are interpreted as deltas from target giving alarmLow,warnLow,warnHigh,alarmHigh
//        if (parameters.size() != 4) {
//            throw new IllegalArgumentException("Delta limit algorithm requires 4 parameters");
//        }

        //Evaluate the target deltas using the target value and the target limits
        this.targetDeltaAlarmLow = -Math.abs(targetValue - targetLimits.getAlarmLow());
        this.targetDeltaWarnLow = -Math.abs(targetValue - targetLimits.getWarnLow());
        this.targetDeltaWarnHigh = Math.abs(targetLimits.getWarnHigh() - targetValue);
        this.targetDeltaAlarmHigh = Math.abs(targetLimits.getAlarmHigh() - targetValue);
        
        //Evaluate the initial deltas based on the nominal initial value
        this.initialDeltaAlarmLow =  -Math.abs(getInitialValue() - limits.getAlarmLow());
        this.initialDeltaWarnLow = -Math.abs(getInitialValue() - limits.getWarnLow());
        this.initialDeltaWarnHigh = Math.abs(limits.getWarnHigh() - getInitialValue());
        this.initialDeltaAlarmHigh = Math.abs(limits.getAlarmHigh() - getInitialValue());

        if (initialDeltaAlarmLow > initialDeltaWarnLow || initialDeltaWarnLow >= 0 || initialDeltaWarnHigh <= 0 || initialDeltaAlarmHigh < initialDeltaWarnHigh) {
            throw new IllegalArgumentException("Delta limit algorith parameters are invalid");
        }
    }

    @Override
    public void adjust(double currentValue, LimitsInterface limits) {
        

        double alpha = Math.abs((currentValue - getTargetValue())/(getInitialValue() - getTargetValue()));
        double deltaAlarmLow = alpha*initialDeltaAlarmLow + (1.0-alpha)*targetDeltaAlarmLow;
        double deltaWarnLow = alpha*initialDeltaWarnLow + (1.0-alpha)*targetDeltaWarnLow;
        double deltaWarnHigh = alpha*initialDeltaWarnHigh + (1.0-alpha)*targetDeltaWarnHigh;
        double deltaAlarmHigh = alpha*initialDeltaAlarmHigh + (1.0-alpha)*targetDeltaAlarmHigh;

        limits.setLimits(currentValue + deltaAlarmLow, currentValue + deltaWarnLow, currentValue + deltaWarnHigh, currentValue + deltaAlarmHigh);
    }
}
